from typing import TYPE_CHECKING, Optional, Protocol, Type, TypeVar

from requests import Session

from gretel_client.workflows.configs.registry import Registry

# import submodules to ensure we don't introduce any circular
# dependencies
if TYPE_CHECKING:
    from gretel_client.files.interface import FileClient
    from gretel_client.projects.projects import Project
    from gretel_client.workflows.manager import WorkflowManager
else:
    FileClient = None
    Project = None
    WorkflowManager = None

T = TypeVar("T")


class GretelApiProviderProtocol(Protocol):
    """
    Protocol defining the interface for Gretel API service providers.

    This protocol defines the contract for classes that provide access to
    Gretel's API services. Implementations should handle API authentication,
    request routing, and session management.

    Note: High-level SDKs should generally not rely on this class, and
    should instead use GretelResourceProvider.
    """

    def get_api(
        self,
        api_interface: Type[T],
        max_retry_attempts: int = 5,
        backoff_factor: float = 1,
        *,
        default_headers: Optional[dict[str, str]] = None,
    ) -> T:
        """Get an instance of the specified API interface.

        Args:
            api_interface: The type of API interface to instantiate
            max_retry_attempts: Maximum number of retry attempts for failed requests
            backoff_factor: Factor to calculate delay between retries
            default_headers: Optional headers to include in all requests

        Returns:
            An instance of the requested API interface
        """
        ...

    def requests(self) -> Session:
        """Get a requests Session instance used for API calls. These
        call are already authenticated and interpolate the correct host
        name. This is useful for endpoints that don't have autogenerated
        bindings.

        Returns:
            Active requests Session object
        """
        ...


class GretelResourceProviderProtocol(Protocol):
    """Protocol defining the interface for Gretel resource access.

    This protocol defines the contract for classes that provide access to
    core Gretel resources like projects, files, and workflows.

    This class is suitable for high-level SDKs to use for interacting with
    Gretel resources.
    """

    @property
    def project_id(self) -> str:
        """Return the default project_id for the session."""
        ...

    @property
    def files(self) -> FileClient:
        """Manage Gretel Files"""
        ...

    @property
    def project(self) -> Project:
        """
        Access to the Project SDK for the currently configured default
        project.
        """
        ...

    @property
    def workflows(self) -> WorkflowManager:
        """Manage Gretel Workflows."""
        ...

    @property
    def console_url(self) -> str:
        """Get the Gretel Console URL for this project."""
        ...

    @property
    def tasks(self) -> Registry: ...
