import json
import logging
import tempfile
import time
import webbrowser

from pathlib import Path
from typing import IO, Callable, Literal, Optional, Union

import pandas as pd

from rich.table import Table
from typing_extensions import Self

logger = logging.getLogger(__name__)

OutputDownloaderT = Callable[[Optional[Literal["json", "html"]]], IO]


class Dataset:
    """
    Represents tabular data generated by a Workflow.

    The Dataset class provides a wrapper around a pandas DataFrame to represent
    output from Workflow.

    This class should never be directly instantiated, but should instead get
    called from the parent Workflow Run, eg::

        gretel.workflows.get_workflow_run("workflow run id").dataset

    """

    def __init__(self, df: pd.DataFrame) -> None:
        self._df = df

    @classmethod
    def from_bytes(cls, parquet_bytes: IO) -> Self:
        return cls(pd.read_parquet(parquet_bytes))

    @classmethod
    def from_records(cls, records: list[dict]) -> Self:
        return cls(pd.DataFrame.from_records(records))

    @property
    def df(self) -> pd.DataFrame:
        """Get the Datasets as a pandas DataFrame"""
        return self._df

    def download(
        self, file: Union[str, Path, IO], format: Literal["csv", "parquet"] = "parquet"
    ) -> None:
        """
        Save the dataset to a file in either CSV or parquet format.

        Args:
            file: The target file path or file-like object where the data will be saved.
            format: The output format, either "csv" or "parquet". Defaults to "parquet".

        Note:
            If a string or Path is provided, any necessary parent directories will be created automatically.
        """
        if isinstance(file, (str, Path)):
            file = Path(file)
            file.parent.mkdir(parents=True, exist_ok=True)

        if format == "csv":
            self._df.to_csv(file, index=False)
        else:
            self._df.to_parquet(file, index=False)


class Report:
    """
    Represents an evaluation report for synthetic data generated by workflows.

    The Report class provides functionality to display, and save evaluation
    report comparing output data with the reference dataset.

    This class should never be directly instantiated, but should instead get
    called from the parent Workflow Run, eg::

        gretel.workflows.get_workflow_run("workflow run id").report

    """

    def __init__(self, report_dict: dict, report_downloader: OutputDownloaderT):
        self._report_dict = report_dict
        self._report_downloader = report_downloader
        self._report_html = None

    @classmethod
    def from_bytes(cls, report_bytes: IO, report_downloader: OutputDownloaderT) -> Self:
        byte_str = report_bytes.read()
        try:
            return cls(json.loads(byte_str), report_downloader)
        except json.JSONDecodeError as ex:
            logger.error(f"Could not deserialize report from json: {ex}")
            logger.error(f"Report contents: {byte_str}")
            raise ex

    @property
    def table(self) -> Table:
        """
        Get a formatted rich Table representation of the report.

        Returns:
            Table: A rich Table instance containing the report data formatted
                for display.
        """
        table = Table(
            show_header=False,
            border_style="medium_purple1",
            show_lines=True,
        )

        table_fields = [
            "synthetic_quality_score",
            "column_correlation_stability",
            "deep_structure_stability",
            "column_distribution_stability",
            "text_semantic_similarity",
            "text_structure_similarity",
            "data_privacy_score",
            "membership_inference_protection_score",
            "attribute_inference_protection_score",
        ]

        for field in table_fields:
            if val := self.dict.get(field):
                table.add_row(str(field), str(val))

        return table

    @property
    def dict(self) -> dict:
        """Get the report as a dictionary"""
        return self._report_dict

    def download(
        self, file: Union[str, Path, IO], format: Literal["json", "html"] = "html"
    ):
        """
        Save the report to a file in either JSON or HTML format.

        Args:
            file: The target file path or file-like object where the report will
                be saved.
            format: The output format, either "json" or "html". Defaults to "json".

        Note:
            If a string or Path is provided, any necessary parent directories
            will be created automatically.
        """
        if isinstance(file, IO):
            return self._report_downloader(format)

        if isinstance(file, (str, Path)):
            file_path = Path(file)
            file_path.parent.mkdir(parents=True, exist_ok=True)

            with open(file_path, "wb") as f:
                f.write(self._report_downloader(format).read())

    def display_in_notebook(self):
        """Display the HTML report in a notebook."""
        try:
            from IPython.display import HTML, display
        except ImportError:
            raise ImportError(
                "IPython is required to display HTML Report in notebooks."
            )
        if self._report_html is None:
            try:
                self._report_html = (
                    self._report_downloader("html").read().decode("utf-8")
                )
            except:
                logger.warning("No HTML report to be displayed in notebook.")
                return
        display(HTML(data=self._report_html, metadata={"isolated": True}))

    def display_in_browser(self):
        """Display the HTML report in a browser."""
        if self._report_html is None:
            try:
                self._report_html = (
                    self._report_downloader("html").read().decode("utf-8")
                )
            except:
                logger.warning("No HTML report to be displayed in browser.")
                return
        with tempfile.NamedTemporaryFile(suffix=".html") as file:
            file.write(bytes(self._report_html, "utf-8"))
            webbrowser.open_new_tab(f"file:///{file.name}")
            time.sleep(1)


class PydanticModel:
    """
    Some Workflow steps produce structured data as pydantic objects. This
    class is a wrapper around those objects providing methods to interact
    with the underlying data structure.
    """

    def __init__(self, model_dict: dict):
        self._model_dict = model_dict

    @classmethod
    def from_bytes(cls, report_bytes: IO) -> Self:
        return cls(json.loads(report_bytes.read()))

    @property
    def dict(self) -> dict:
        """Return the dictionary representation of the output"""
        return self._model_dict
