<%!
import re

def get_var_access(var_name, var_arrays, var_indices, variables, constants_names, config_names, model_name):
    """
    Get C++ array access for a variable.
    """
    # special variables
    if var_name == 'globalinput':
        return 'tmp_globalinput'
    if var_name == 'noise_idx':
        return 'noise_idx'
    
    # other variables (states, intermediates, params, etc.)
    if var_name in variables:
        var = variables[var_name]
        array = var_arrays[var_name]
        index = var_indices[var_name]
        
        # Special handling for noise
        if var.var_type == 'noise':
            if index > 0:
                return f"{array}[noise_idx + {index}]"
            else:
                return f"{array}[noise_idx]"
        
        return f"{array}[{index}]"
    
    # constants
    if var_name in constants_names:
        return f"d_{model_name}c.{var_name}"
    
    # configs
    if var_name in config_names:
        return f"this->conf.{var_name}"
    
    # otherwise return as-is
    return var_name

def process_equation(equation, var_arrays, var_indices, variables, constants_names, config_names, model_name):
    """
    Process an equation, replacing variable names with array accesses.
    """
    equation = equation.strip()
    # skip comments and empty lines
    if not equation or equation.startswith('#'):
        return equation
    
    # split into left and right side
    if '=' not in equation:
        return equation
    
    # handle compound operators (+=, -=, etc.)
    compound_ops = ['+=', '-=', '*=', '/=']
    is_compound = any(op in equation for op in compound_ops)
    
    if is_compound:
        for op in compound_ops:
            if op in equation:
                parts = equation.split(op, 1)
                lhs = parts[0].strip()
                rhs = parts[1].strip()
                
                lhs_code = get_var_access(lhs, var_arrays, var_indices, variables, constants_names, config_names, model_name)
                rhs_code = process_expression(rhs, var_arrays, var_indices, variables, constants_names, config_names, model_name)
                
                return f"{lhs_code} {op} {rhs_code};"
    else:
        parts = equation.split('=', 1)
        lhs = parts[0].strip()
        rhs = parts[1].strip()
        
        lhs_code = get_var_access(lhs, var_arrays, var_indices, variables, constants_names, config_names, model_name)
        rhs_code = process_expression(rhs, var_arrays, var_indices, variables, constants_names, config_names, model_name)
        
        return f"{lhs_code} = {rhs_code};"

def process_expression(expr, var_arrays, var_indices, variables, constants_names, config_names, model_name):
    """
    Process an expression, replacing variable names with array accesses.
    """
    # Find all identifiers in the expression
    identifier_pattern = r'\b[a-zA-Z_][a-zA-Z0-9_]*\b'
    
    def replace_identifier(match):
        name = match.group(0)
        # Don't replace function names or keywords
        keywords = {'max', 'min', 'exp', 'sqrt', 'pow', 'abs', 'sin', 'cos', 'tan', 
                   'log', 'floor', 'ceil', 'EXP', 'SQRT', 'POW', 'ABS'}
        if name in keywords:
            return name
        return get_var_access(name, var_arrays, var_indices, variables, constants_names, config_names, model_name)
    
    result = re.sub(identifier_pattern, replace_identifier, expr)
    return result

def format_equations(equations, var_arrays, var_indices, variables, constants_names, config_names, model_name, indent='    '):
    """
    Format a list of equations with proper indentation and comments.
    """
    lines = []
    for eq in equations:
        eq = eq.strip()
        if not eq:
            lines.append('')
        elif eq.startswith('#'):
            # Comment
            lines.append(f"{indent}{eq.replace('#', '//')}")
        else:
            # Equation
            processed = process_equation(eq, var_arrays, var_indices, variables, constants_names, config_names, model_name)
            lines.append(f"{indent}// {eq}")
            lines.append(f"{indent}{processed}")
    return '\n'.join(lines)

def get_constants_names(constants):
    """
    Create a list of constant names
    """
    const_list = []
    for const in constants:
        const_list.append(const[1])
    return const_list

def get_config_names(config):
    """
    Create a list of constant names
    """
    conf_list = []
    for conf in config:
        conf_list.append(conf[1])
    return conf_list
%>\
<%
# get names of constants and configs
constants_names = get_constants_names(constants)
config_names = get_config_names(config)
%>\
/* 
    Autogenerated using:
    - Template: '${template_path}'
    - Model definition: '${yaml_path}'
    Do not modify this autogenerated code. Instead modify template
    or model definition.
*/
#include "cubnm/includes.cuh"
#include "cubnm/defines.h"
#include "cubnm/models/${model_name.lower()}.cuh"

__device__ __NOINLINE__ void ${model_name}Model::init(
    double* _state_vars, double* _intermediate_vars,
    double* _global_params, double* _regional_params,
    int* _ext_int, bool* _ext_bool,
    int* _ext_int_shared, bool* _ext_bool_shared
) {
${format_equations(init_equations, var_arrays, var_indices, variables, constants_names, config_names, model_name)}
}

__device__ __NOINLINE__ void ${model_name}Model::restart(
    double* _state_vars, double* _intermediate_vars, 
    double* _global_params, double* _regional_params,
    int* _ext_int, bool* _ext_bool,
    int* _ext_int_shared, bool* _ext_bool_shared
) {
${format_equations(restart_equations, var_arrays, var_indices, variables, constants_names, config_names, model_name)}
}

__device__ void ${model_name}Model::step(
        double* _state_vars, double* _intermediate_vars,
        double* _global_params, double* _regional_params,
        double& tmp_globalinput,
        double* noise, long& noise_idx
        ) {
${format_equations(step_equations, var_arrays, var_indices, variables, constants_names, config_names, model_name)}
}
% if has_post_bw_step:
${custom_methods.get('post_bw_step', '')}
% endif
% if has_post_integration:
${custom_methods.get('post_integration', '')}
% endif