from typing import Dict, List, Optional, Union

import pytest

from ..utils import (
    _ensure_newline,
    name2title,
    sanitize_value,
    strip_top_level_optional,
    version2tuple,
)


@pytest.mark.parametrize(
    "name,title",
    [
        ("relatedResource", "Related Resource"),
        ("identifier", "Identifier"),
        ("wasGeneratedBy", "Was Generated by"),
        ("sameAs", "Same as"),
        ("includeInCitation", "Include in Citation"),
        ("anExtraField", "An Extra Field"),
        ("propertyID", "Property ID"),
        ("fieldINeed", "Field I Need"),
        ("needsADatum", "Needs a Datum"),
        ("contentUrl", "Content URL"),
        ("ContactPoint", "Contact Point"),
    ],
)
def test_name2title(name: str, title: str) -> None:
    assert name2title(name) == title


@pytest.mark.parametrize(
    "ver,error",
    [
        ("ContactPoint", True),
        ("0.1.2", False),
        ("0.12.20", False),
        ("0.1.2a", True),
        ("0.1.2-rc1", True),
    ],
)
def test_version(ver: str, error: bool) -> None:
    if error:
        with pytest.raises(ValueError):
            version2tuple(ver)
    else:
        assert len(version2tuple(ver)) == 3


def test_newline() -> None:
    obj = "\n"
    assert _ensure_newline(obj).endswith("\n")
    obj = ""
    assert _ensure_newline(obj).endswith("\n")


@pytest.mark.parametrize(
    "input_, expected_output",
    [
        (Union[str, int, None], Union[str, int, None]),
        (Optional[Union[str, int]], Optional[Union[str, int]]),
        (Union[int], Union[int]),
        (Union[None], Union[None]),
        (Union[str, int, None, None], Union[str, int, None, None]),
        (Union[None, int, str], Union[None, int, str]),
        (Union[None, int, None, str], Union[None, int, None, str]),
        (Optional[str], str),
        (Optional[Optional[str]], str),
        (Optional[List[Optional[str]]], List[Optional[str]]),
        (Union[None, int], int),
        (Union[None, int, None], int),
        (Union[None, Dict[str, int]], Dict[str, int]),
        (int, int),
        (float, float),
    ],
)
def test_strip_top_level_optional(input_: type, expected_output: type) -> None:
    assert strip_top_level_optional(input_) == expected_output


def test_sanitize_value() -> None:
    # . is not sanitized in extension but elsewhere
    assert sanitize_value("_.ext", "extension") == "-.ext"
    assert sanitize_value("_.ext", "unrelated") == "--ext"
    assert sanitize_value("_.ext") == "--ext"
    assert sanitize_value("A;B") == "A-B"
    assert sanitize_value("A\\/B") == "A--B"
    assert sanitize_value("A\"'B") == "A--B"
