import * as secretsmanager from 'aws-cdk-lib/aws-secretsmanager';
import * as constructs from 'constructs';
import * as client from './client';
import * as node from './node';
import * as user from './user';
/**
 * Define which Hyperledger Fabric framework to use
 */
export declare enum FrameworkVersion {
    VERSION_1_2 = "1.2",
    VERSION_1_4 = "1.4",
    VERSION_2_2 = "2.2"
}
/**
 * Starter networks are cheaper, but are limited to 2 nodes that
 * can only be from a subset of types (see node.ts for the list)
 */
export declare enum NetworkEdition {
    STARTER = "STARTER",
    STANDARD = "STANDARD"
}
/**
 * Constants to define ties in voting for new members
 */
export declare enum ThresholdComparator {
    GREATER_THAN = "GREATER_THAN",
    GREATER_THAN_OR_EQUAL_TO = "GREATER_THAN_OR_EQUAL_TO"
}
/**
 * Construct properties for `HyperledgerFabricNetwork`
 */
export interface HyperledgerFabricNetworkProps {
    /**
     * Managed Blockchain network name
     */
    readonly networkName: string;
    /**
     * Managed Blockchain network description
     *
     * @default - Set to match network name
     */
    readonly networkDescription?: string;
    /**
     * Managed Blockchain member name
     */
    readonly memberName: string;
    /**
     * Managed Blockchain member description
     *
     * @default - Set to match member name
     */
    readonly memberDescription?: string;
    /**
     * Hyperledger Fabric framework version
     *
     * @default - FrameworkVersion.VERSION_1_4
     */
    readonly frameworkVersion?: FrameworkVersion;
    /**
     * Managed Blockchain network edition
     *
     * @default - NetworkEdition.STANDARD
     */
    readonly networkEdition?: NetworkEdition;
    /**
     * The duration from the time that a proposal is created until it expires
     * @default - 24 hours
     */
    readonly proposalDurationInHours?: number;
    /**
     * The percentage of votes among all members that must be yes for a proposal to be approved
     * @default - 50 percent
     */
    readonly thresholdPercentage?: number;
    /**
     * Determines whether the yes votes must be greater than the threshold percentage
     * or must be greater than or equal to the threhold percentage to be approved
     * @default - GREATER_THAN
     */
    readonly thresholdComparator?: ThresholdComparator;
    /**
     * The configuration to enable or disable certificate authority logging
     * @default - true
     */
    readonly enableCaLogging?: boolean;
    /**
     * List of nodes to create on the network
     *
     * @default - One node with default configuration
     */
    readonly nodes?: Array<node.HyperledgerFabricNodeProps>;
    /**
     * The Client network to interact with the Hyperledger Fabric network
     * @default - Client network with Default properties
     * (CIDR-`10.0.0.0/16` and subnets of type `PRIVATE_ISOLATED`)
     */
    readonly client?: client.HyperledgerFabricClientProps;
    /**
     * Configuration to enable/disable enrollment of admin user
     * @default - true
     */
    readonly enrollAdmin?: boolean;
    /**
     * List of users to register with Fabric CA
     * Note: enrollAdmin property has to be enabled for registering users
     */
    readonly users?: Array<user.HyperledgerFabricUserProps>;
}
/**
 * Creates a Hyperledger Fabric network on Amazon Managed Blockchain
 */
export declare class HyperledgerFabricNetwork extends constructs.Construct {
    /**
     * Managed Blockchain network name
     */
    readonly networkName: string;
    /**
     * Managed Blockchain network description
     */
    readonly networkDescription: string;
    /**
     * Managed Blockchain network identifier generated on construction
     */
    readonly networkId: string;
    /**
     * Managed Blockchain member name
     */
    readonly memberName: string;
    /**
     * Managed Blockchain member description
     */
    readonly memberDescription: string;
    /**
     * Managed Blockchain member identifier generated on construction
     */
    readonly memberId: string;
    /**
     * Hyperledger Fabric framework version
     */
    readonly frameworkVersion: FrameworkVersion;
    /**
     * Managed Blockchain network edition
     */
    readonly networkEdition: NetworkEdition;
    /**
     * The duration from the time that a proposal is created until it expires
     */
    readonly proposalDurationInHours: number;
    /**
     * The percentage of votes among all members that must be yes for a proposal to be approved
     */
    readonly thresholdPercentage: number;
    /**
     * Determines whether the yes votes must be greater than the threshold percentage
     * or must be greater than or equal to the threhold percentage to be approved
     */
    readonly thresholdComparator: ThresholdComparator;
    /**
     * The configuration to enable or disable certificate authority logging
     */
    readonly enableCaLogging: boolean;
    /**
     * Managed Blockchain network VPC endpoint service name
     */
    readonly vpcEndpointServiceName: string;
    /**
     * Managed Blockchain network ordering service endpoint
     */
    readonly ordererEndpoint: string;
    /**
     * Managed Blockchain member CA endpoint
     */
    readonly caEndpoint: string;
    /**
     * Secret ARN for the Hyperledger Fabric admin password
     */
    readonly adminPasswordSecret: secretsmanager.Secret;
    /**
     * Secret for Hyperledger Fabric admin private key
     */
    readonly adminPrivateKeySecret: secretsmanager.Secret;
    /**
     * Secret for Hyperledger Fabric admin signed certificate
     */
    readonly adminSignedCertSecret: secretsmanager.Secret;
    /**
     * List of nodes created in the network
     */
    readonly nodes: Array<node.HyperledgerFabricNode>;
    /**
     * The client network to interact with the Hyperledger Fabric network
     */
    readonly client: client.HyperledgerFabricClient;
    /**
     * Configuration to enable/disable admin user enrollment
     */
    readonly enrollAdmin: boolean;
    /**
     * List of users registered with CA
     */
    readonly users: Array<user.HyperledgerFabricUser>;
    constructor(scope: constructs.Construct, id: string, props: HyperledgerFabricNetworkProps);
}
