#!/usr/bin/env python3
"""
Python script to test decompression of data generated by TypeScript compression utilities.
This verifies cross-language compatibility between JavaScript/TypeScript and Python.
"""

import base64
import gzip
import json
import sys
from typing import Dict, Any, List, Tuple


def decompress_gzip_base64(compressed_data: str) -> str:
    """
    Decompress base64-encoded gzip data (matching TypeScript compressString function).
    
    Args:
        compressed_data: Base64-encoded gzip compressed string
        
    Returns:
        Original decompressed string
        
    Raises:
        Exception: If decompression fails
    """
    try:
        # Decode from base64
        compressed_bytes = base64.b64decode(compressed_data)
        
        # Decompress using gzip
        decompressed_bytes = gzip.decompress(compressed_bytes)
        
        # Convert back to string using UTF-8
        original_string = decompressed_bytes.decode('utf-8')
        
        return original_string
    except Exception as e:
        raise Exception(f"Failed to decompress data: {e}")


def decode_base64_utf8(encoded_data: str) -> str:
    """
    Decode base64-encoded UTF-8 string (matching TypeScript encodeBase64 function).
    
    Args:
        encoded_data: Base64-encoded string
        
    Returns:
        Original decoded string
        
    Raises:
        Exception: If decoding fails
    """
    try:
        # Decode from base64
        decoded_bytes = base64.b64decode(encoded_data)
        
        # Convert back to string using UTF-8
        original_string = decoded_bytes.decode('utf-8')
        
        return original_string
    except Exception as e:
        raise Exception(f"Failed to decode base64 data: {e}")


def test_decompression_with_sample_data():
    """Test decompression with various sample data types."""
    
    print("🧪 Testing Python decompression of TypeScript-generated data")
    print("=" * 60)
    
    # Test cases with expected results
    test_cases = [
        {
            'name': 'Simple String',
            'original': 'Hello, World!',
            'description': 'Basic ASCII string'
        },
        {
            'name': 'Unicode String',
            'original': 'Hello 🌍! Testing émojis and spëcial chars: 中文 العربية',
            'description': 'String with Unicode characters, emojis, and special chars'
        },
        {
            'name': 'JSON Data',
            'original': json.dumps({
                'message_type': 'chat_request',
                'call_id': 'test-123',
                'username': 'testuser',
                'chatID': 'chat-456',
                'content': 'This is a test message with some content.',
                'metadata': {
                    'timestamp': '2025-01-23T20:30:00.000Z',
                    'source': 'test'
                }
            }),
            'description': 'JSON protocol message'
        },
        {
            'name': 'Large Repeated String',
            'original': 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. ' * 100,
            'description': 'Large string with repeated patterns (good compression)'
        },
        {
            'name': 'Empty String',
            'original': '',
            'description': 'Empty string edge case'
        },
        {
            'name': 'Whitespace String',
            'original': '   \n\t   ',
            'description': 'String with various whitespace characters'
        },
        {
            'name': 'String with Null Bytes',
            'original': 'Hello\x00World\x00Test',
            'description': 'String containing null bytes'
        }
    ]
    
    return test_cases


def generate_typescript_test_data():
    """
    Generate a Node.js script that will create compressed test data using TypeScript functions.
    This data will then be used to test Python decompression.
    """
    
    test_cases = test_decompression_with_sample_data()
    
    # Generate Node.js script
    node_script = '''
// Auto-generated Node.js script to create test data for Python verification
const fs = require('fs');

// Mock pako for Node.js environment (simplified gzip implementation)
const zlib = require('zlib');

// Simulate TypeScript compression functions
function compressString(input) {
    try {
        // Convert string to UTF-8 bytes
        const utf8Bytes = Buffer.from(input, 'utf8');
        
        // Compress using gzip
        const compressed = zlib.gzipSync(utf8Bytes);
        
        // Convert to base64
        const base64 = compressed.toString('base64');
        
        return base64;
    } catch (error) {
        throw new Error(`Failed to compress string: ${error.message}`);
    }
}

function encodeBase64(input) {
    try {
        // Convert string to UTF-8 bytes then to base64
        const utf8Bytes = Buffer.from(input, 'utf8');
        const base64 = utf8Bytes.toString('base64');
        
        return base64;
    } catch (error) {
        throw new Error(`Failed to encode string as base64: ${error.message}`);
    }
}

// Test cases
const testCases = [
'''
    
    # Add test cases to the Node.js script
    for i, case in enumerate(test_cases):
        original_escaped = case['original'].replace('\\', '\\\\').replace("'", "\\'").replace('\n', '\\n').replace('\t', '\\t').replace('\x00', '\\x00')
        node_script += f'''    {{
        name: '{case["name"]}',
        original: '{original_escaped}',
        description: '{case["description"]}'
    }}{',' if i < len(test_cases) - 1 else ''}
'''
    
    node_script += '''
];

// Generate test data
const results = {
    timestamp: new Date().toISOString(),
    testCases: []
};

console.log('Generating test data for Python verification...');

testCases.forEach((testCase, index) => {
    try {
        const compressed = compressString(testCase.original);
        const base64Encoded = encodeBase64(testCase.original);
        
        results.testCases.push({
            name: testCase.name,
            description: testCase.description,
            original: testCase.original,
            compressed: compressed,
            base64Encoded: base64Encoded,
            originalLength: Buffer.from(testCase.original, 'utf8').length,
            compressedLength: Buffer.from(compressed, 'base64').length
        });
        
        console.log(`✓ ${testCase.name}: ${Buffer.from(testCase.original, 'utf8').length} → ${Buffer.from(compressed, 'base64').length} bytes`);
    } catch (error) {
        console.error(`✗ ${testCase.name}: ${error.message}`);
        results.testCases.push({
            name: testCase.name,
            description: testCase.description,
            original: testCase.original,
            error: error.message
        });
    }
});

// Save results to JSON file
fs.writeFileSync('compression_test_data.json', JSON.stringify(results, null, 2));
console.log('\\nTest data saved to compression_test_data.json');
console.log(`Generated ${results.testCases.length} test cases`);
'''
    
    return node_script


def run_python_verification_tests():
    """Run Python verification tests using the generated test data."""
    
    try:
        # Load test data
        with open('compression_test_data.json', 'r', encoding='utf-8') as f:
            test_data = json.load(f)
        
        print(f"📊 Loaded {len(test_data['testCases'])} test cases")
        print(f"📅 Generated: {test_data['timestamp']}")
        print("=" * 60)
        
        passed = 0
        failed = 0
        
        for i, test_case in enumerate(test_data['testCases'], 1):
            print(f"\n{i}. {test_case['name']}")
            print(f"   Description: {test_case['description']}")
            
            if 'error' in test_case:
                print(f"   ❌ Skipped (generation error): {test_case['error']}")
                failed += 1
                continue
            
            original = test_case['original']
            
            # Test gzip decompression
            try:
                decompressed = decompress_gzip_base64(test_case['compressed'])
                if decompressed == original:
                    print(f"   ✅ Gzip decompression: PASS")
                else:
                    print(f"   ❌ Gzip decompression: FAIL (content mismatch)")
                    print(f"      Expected: {repr(original[:50])}...")
                    print(f"      Got:      {repr(decompressed[:50])}...")
                    failed += 1
                    continue
            except Exception as e:
                print(f"   ❌ Gzip decompression: FAIL ({e})")
                failed += 1
                continue
            
            # Test base64 decoding
            try:
                decoded = decode_base64_utf8(test_case['base64Encoded'])
                if decoded == original:
                    print(f"   ✅ Base64 decoding: PASS")
                else:
                    print(f"   ❌ Base64 decoding: FAIL (content mismatch)")
                    failed += 1
                    continue
            except Exception as e:
                print(f"   ❌ Base64 decoding: FAIL ({e})")
                failed += 1
                continue
            
            # Show compression stats
            original_size = test_case['originalLength']
            compressed_size = test_case['compressedLength']
            if original_size > 0:
                ratio = round((1 - compressed_size / original_size) * 100, 1)
                print(f"   📈 Compression: {original_size} → {compressed_size} bytes ({ratio}% reduction)")
            
            passed += 1
        
        print("\n" + "=" * 60)
        print(f"🎯 Results: {passed} passed, {failed} failed")
        
        if failed == 0:
            print("🎉 All cross-language compatibility tests passed!")
            return True
        else:
            print("⚠️  Some tests failed - check compatibility issues")
            return False
            
    except FileNotFoundError:
        print("❌ Test data file not found. Run the Node.js script first.")
        return False
    except Exception as e:
        print(f"❌ Error running verification tests: {e}")
        return False


def main():
    """Main function to run the Python verification tests."""
    
    if len(sys.argv) > 1 and sys.argv[1] == '--generate-node-script':
        # Generate Node.js script for creating test data
        script = generate_typescript_test_data()
        with open('generate_test_data.js', 'w', encoding='utf-8') as f:
            f.write(script)
        print("📝 Generated generate_test_data.js")
        print("Run: node generate_test_data.js")
        return
    
    # Run verification tests
    success = run_python_verification_tests()
    sys.exit(0 if success else 1)


if __name__ == '__main__':
    main()
