"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = exports.EventInputType = void 0;
const client_cost_explorer_1 = require("@aws-sdk/client-cost-explorer");
const web_api_1 = require("@slack/web-api");
const aws_lambda_errors_1 = require("@yicr/aws-lambda-errors");
const get_billing_command_1 = require("./lib/get-billing-command");
const get_date_range_1 = require("./lib/get-date-range");
var EventInputType;
(function (EventInputType) {
    EventInputType["ACCOUNTS"] = "Accounts";
    EventInputType["SERVICES"] = "Services";
})(EventInputType = exports.EventInputType || (exports.EventInputType = {}));
const ceClient = new client_cost_explorer_1.CostExplorerClient({
    region: 'us-east-1',
});
exports.handler = async (event, context) => {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    console.log(`Context: ${JSON.stringify(context, null, 2)}`);
    // do validation
    if (!process.env.SLACK_TOKEN) {
        throw new aws_lambda_errors_1.MissingEnvironmentVariableError('missing environment variable SLACK_TOKEN.');
    }
    if (!process.env.SLACK_CHANNEL) {
        throw new aws_lambda_errors_1.MissingEnvironmentVariableError('missing environment variable SLACK_CHANNEL.');
    }
    if (!event.Type) {
        throw new aws_lambda_errors_1.MissingInputVariableError('missing input variable Type');
    }
    else {
        if (!Object.values(EventInputType).includes(event.Type)) {
            throw new aws_lambda_errors_1.InvalidInputVariableFormatError('invalid input variable format is Accounts or Services.');
        }
    }
    // 👇Calculate Date Range
    const dateRange = new get_date_range_1.GetDateRange();
    console.log(`DateRange::${JSON.stringify(dateRange, null, 2)}`);
    // 👇Get Total Billing
    const totalBilling = await (new get_billing_command_1.GetTotalBilling(ceClient)).execute(dateRange);
    console.log(`TotalBilling: ${JSON.stringify(totalBilling, null, 2)}`);
    const fields = await (async () => {
        switch (event.Type) {
            case EventInputType.ACCOUNTS:
                // 👇Get Accounts Billings
                const accountBillings = await (new get_billing_command_1.GetAccountBillings(ceClient).execute(dateRange));
                console.log(`AccountBillings: ${JSON.stringify(accountBillings, null, 2)}`);
                return accountBillings?.map((value) => {
                    return {
                        title: value.account,
                        value: `${value.amount} ${value.unit}`,
                    };
                });
            case EventInputType.SERVICES:
                // 👇Get Service Billings
                const serviceBillings = await (new get_billing_command_1.GetServiceBilling(ceClient)).execute(dateRange);
                console.log(`ServiceBilling: ${JSON.stringify(serviceBillings, null, 2)}`);
                return serviceBillings?.map((value) => {
                    return {
                        title: value.service,
                        value: `${value.amount} ${value.unit}`,
                    };
                });
        }
    })();
    const client = new web_api_1.WebClient(process.env.SLACK_TOKEN);
    const channel = process.env.SLACK_CHANNEL;
    // Send the notification
    await (async () => {
        const result = await client.chat.postMessage({
            channel,
            icon_emoji: ':money-with-wings:',
            text: `AWS Cost Reports (${dateRange.start} - ${dateRange.end})`,
            attachments: [
                {
                    title: ':moneybag: Total',
                    text: `${totalBilling?.amount} ${totalBilling?.unit}`,
                    color: '#ff8c00',
                },
            ],
        });
        if (result.ok) {
            await client.chat.postMessage({
                channel,
                thread_ts: result.ts,
                attachments: [
                    {
                        color: '#ffd700',
                        fields: fields?.map((filed) => {
                            return {
                                title: `:aws: ${filed.title}`,
                                value: filed.value,
                                short: false,
                            };
                        }),
                    },
                ],
            });
        }
    })();
    return 'OK';
};
//# sourceMappingURL=data:application/json;base64,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