"""
The Flask-SecurityTxt extension.
"""
import os
from collections.abc import Iterable

from datetime import datetime as dt, timedelta as td, timezone as tz
from importlib.metadata import version

from flask import Flask, Response, request, url_for, current_app
from flask_babel import get_babel
from pgpy import PGPKey, PGPMessage
from werkzeug.routing import BuildError

DEFAULT_HEADER = None

DEFAULT_FOOTER = """
#
# GENERATED BY FLASK-SECURITYTXT {version}
#
# Flask-SecurityTxt has been developed by M.P. van de Weerd for
# Parcifal Programmatuur under the GPLv3 licence.
#
# https://pypi.org/project/Flask-SecurityTxt/
# https://gitlab.com/parcifal/flask-security-txt
#
# https://www.van-de-weerd.net/
# https://www.parcifal.dev/
#
"""


# pylint: disable=too-few-public-methods, too-many-instance-attributes
class SecurityTxt:
    """
    Extends Flask application with a dynamically generated security.txt.
    """

    # pylint: disable=too-many-arguments, too-many-positional-arguments
    def __init__(self, app: Flask = None,
                 default_endpoint: str = "security_txt",
                 default_contact_mailbox: str = "security",
                 default_expires_offset: tuple = (0, 0, 0, 0, 0, 0, 1),
                 default_preferred_languages: tuple = ("en", ),
                 default_canonical: str = None,
                 default_dir: str = ".well-known",
                 default_file_name: str = "security.txt",
                 default_sign_key: str = None,
                 default_header: str = DEFAULT_HEADER,
                 default_footer: str = DEFAULT_FOOTER):
        self.app = app
        self.sign_key = None

        self._default_endpoint = default_endpoint
        self._default_contact_mailbox = default_contact_mailbox
        self._default_expires_offset = default_expires_offset
        self._default_preferred_languages = default_preferred_languages
        self._default_canonical = default_canonical or default_endpoint
        self._default_dir = default_dir
        self._default_file_name = default_file_name
        self._default_sign_key = default_sign_key
        self._default_header = default_header
        self._default_footer = default_footer

        if app is not None:
            self.init_app(app)

    def init_app(self, app: Flask):
        """
        Initialize the specified Flask application to generate a security.txt.
        """
        if not hasattr(app, "extensions"):
            app.extensions = {}

        app.extensions["security_txt"] = self

        app.config.setdefault("SECURITY_TXT_ENDPOINT",
                              self._default_endpoint)
        app.config.setdefault("SECURITY_TXT_CONTACT_MAILBOX",
                              self._default_contact_mailbox)
        app.config.setdefault("SECURITY_TXT_EXPIRES_OFFSET",
                              self._default_expires_offset)
        app.config.setdefault("SECURITY_TXT_CANONICAL",
                              self._default_canonical)
        app.config.setdefault("SECURITY_TXT_HEADER",
                              self._default_header)
        app.config.setdefault("SECURITY_TXT_FOOTER",
                              self._default_footer)

        _dir = app.config.get("WELL_KNOWN_DIR",
                              self._default_dir)
        name = app.config.get("SECURITY_TXT_FILE_NAME",
                              self._default_file_name)

        app.add_url_rule("/" + _dir + "/" + name,
                         app.config.get("SECURITY_TXT_ENDPOINT"),
                         self._send_security_txt)

        sign_key = app.config.get("SECURITY_TXT_SIGN_KEY",
                                  self._default_sign_key)

        if sign_key is not None:
            self.sign_key, _ = PGPKey.from_file(sign_key)

    def _send_security_txt(self):
        """
        @return:
            The full security.txt as an HTTP response.
        """
        lines = []

        header = current_app.config.get("SECURITY_TXT_HEADER")

        if header is not None:
            assert isinstance(header, str)
            lines.append(header.format(version=version("Flask-SecurityTxt")))

        for key, item in self._get_fields().items():
            if item is None:
                continue

            comment_key = f"SECURITY_TXT_{key.upper()}_COMMENT"

            if comment_key in current_app.config:
                comment = current_app.config.get(comment_key)
                lines.append(comment)

            if isinstance(item, str):
                item = [item]

            assert isinstance(item, Iterable)

            for value in item:

                assert isinstance(value, str)
                lines.append(f"{key}: {value}")

        footer = current_app.config.get("SECURITY_TXT_FOOTER")

        if footer is not None:
            assert isinstance(footer, str)
            lines.append(footer.format(version=version("Flask-SecurityTxt")))

        body = "\n".join(lines)

        if self.sign_key is not None:
            assert isinstance(self.sign_key, PGPKey)

            message = PGPMessage.new(body, cleartext=True)
            message |= self.sign_key.sign(message)

            body = str(message)

        return Response(body, mimetype="text/plain")

    def _get_fields(self) -> dict:
        """
        @return:
            A dict mapping the keys to the values of the security.txt fields.
        """
        return {
            "Contact": self._get_field_value_contact(),
            "Expires": self._get_field_value_expires(),
            "Encryption": self._get_field_value_encryption(),
            "Acknowledgments": self._get_field_value_acknowledgements(),
            "Preferred-Languages": self._get_field_value_preferred_languages(),
            "Canonical": self._get_field_value_canonical(),
            "Policy": self._get_field_value_policy(),
            "Hiring": self._get_field_value_hiring()
        }

    def _get_urls_from_value(self, value, schemes=None):
        """
        Parse zero or more URLs from the specified config value. The URLs can
        be defined as either a full URL string starting with one of the
        specified schemes, or as an end-point that can be resolved by the
        Flask application.
        """
        if not value:
            return
        if schemes is None:
            schemes = ["https://"]
        if isinstance(value, str):
            value = [value]

        assert isinstance(value, Iterable) and isinstance(schemes, Iterable)

        for item in value:
            assert isinstance(item, str)

            if any(item.startswith(scheme) for scheme in schemes):
                yield item
                continue

            if current_app.has_static_folder and os.path.exists(
                    os.path.join(current_app.static_folder, item)):
                yield url_for("static", filename=item,
                              _external=True, _scheme="https")
                continue

            try:
                yield url_for(item, _external=True, _scheme="https")
            except BuildError as error:
                raise ValueError(
                    "An URL field value must either be string starting with "
                    "https://, the name of a static file or an end-point "
                    "name that can be resolved by the flask application."
                ) from error

    def _get_field_value_contact(self) -> str:
        """
        @return:
            The value of the contact field.
        """
        value = current_app.config.get("SECURITY_TXT_CONTACT")

        if not value:
            mailbox = current_app.config.get("SECURITY_TXT_CONTACT_MAILBOX")
            assert isinstance(mailbox, str)
            domain = request.host.rsplit(":", 1)[0]
            yield f"mailto:{mailbox}@{domain}"
            return

        try:
            yield from self._get_urls_from_value(value, [
                "https://", "mailto:", "tel:"])
        except ValueError as error:
            raise ValueError(
                "Invalid contact configuration value.") from error

    def _get_field_value_expires(self) -> str:
        """
        @return:
            The value of the expires field.
        """
        value = current_app.config.get("SECURITY_TXT_EXPIRES")

        if isinstance(value, str):
            return value
        if isinstance(value, dt):
            return value.replace(microsecond=0).isoformat()

        assert value is None

        offset = current_app.config.get("SECURITY_TXT_EXPIRES_OFFSET")

        if isinstance(offset, td):
            value = dt.now(tz.utc) + offset
        if isinstance(offset, tuple):
            value = dt.now(tz.utc) + td(*offset)

        assert isinstance(value, dt)
        return value.replace(microsecond=0).isoformat()

    def _get_field_value_encryption(self):
        """
        @return:
            The value of the encryption field.
        """
        try:
            yield from self._get_urls_from_value(
                current_app.config.get("SECURITY_TXT_ENCRYPTION"), [
                    "https://", "dns:", "openpgp4fpr:"])
        except ValueError as error:
            raise ValueError(
                "Invalid encryption configuration value.") from error

    def _get_field_value_acknowledgements(self):
        """
        @return:
            The value of the acknowledgements field.
        """
        try:
            yield from self._get_urls_from_value(
                current_app.config.get("SECURITY_TXT_ACKNOWLEDGEMENTS"))
        except ValueError as error:
            raise ValueError(
                "Invalid acknowledgements configuration value.") from error

    def _get_field_value_preferred_languages(self):
        """
        @return:
            The value of the preferred languages field.
        """
        value = current_app.config.get("SECURITY_TXT_PREFERRED_LANGUAGES")

        if "babel" not in current_app.extensions:
            value = self._default_preferred_languages or ""
        if isinstance(value, str):
            return value
        if isinstance(value, (list, tuple)):
            return ", ".join(value)

        babel = get_babel()

        return ", ".join([
            t.language for t in getattr(babel.instance, "list_translations")()
        ])

    def _get_field_value_canonical(self):
        """
        @return:
            The value of the canonical field.
        """
        try:
            yield from self._get_urls_from_value(
                current_app.config.get("SECURITY_TXT_CANONICAL"))
        except ValueError as error:
            raise ValueError(
                "Invalid canonical configuration value.") from error

    def _get_field_value_policy(self):
        """
        @return:
            The value of the policy field.
        """
        try:
            yield from self._get_urls_from_value(
                current_app.config.get("SECURITY_TXT_POLICY"))
        except ValueError as error:
            raise ValueError(
                "Invalid policy configuration value.") from error

    def _get_field_value_hiring(self):
        """
        @return:
            The value of the hiring field.
        """
        try:
            yield from self._get_urls_from_value(
                current_app.config.get("SECURITY_TXT_HIRING"))
        except ValueError as error:
            raise ValueError(
                "Invalid hiring configuration value.") from error
