"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.after = exports.main = void 0;
const path = require("path");
const fs = require("fs-extra");
const fact_tables_1 = require("./fact-tables");
const aws_entities_1 = require("../lib/aws-entities");
const default_1 = require("../lib/default");
async function main() {
    checkRegions(fact_tables_1.APPMESH_ECR_ACCOUNTS);
    checkRegions(fact_tables_1.DLC_REPOSITORY_ACCOUNTS);
    checkRegions(fact_tables_1.ELBV2_ACCOUNTS);
    checkRegions(fact_tables_1.FIREHOSE_CIDR_BLOCKS);
    checkRegions(fact_tables_1.ROUTE_53_BUCKET_WEBSITE_ZONE_IDS);
    checkRegionsSubMap(fact_tables_1.CLOUDWATCH_LAMBDA_INSIGHTS_ARNS);
    checkRegionsSubMap(fact_tables_1.APPCONFIG_LAMBDA_LAYER_ARNS);
    const lines = [
        "import { Fact, FactName } from './fact';",
        '',
        '/* eslint-disable quote-props */',
        '/* eslint-disable max-len */',
        '',
        '/**',
        ' * Built-in regional information, re-generated by `npm run build`.',
        ' *',
        ` * @generated ${new Date().toISOString()}`,
        ' */',
        'export class BuiltIns {',
        '  /**',
        '   * Registers all the built in regional data in the RegionInfo database.',
        '   */',
        '  public static register(): void {',
    ];
    const defaultMap = 'default';
    for (const region of aws_entities_1.AWS_REGIONS) {
        let partition = fact_tables_1.PARTITION_MAP[defaultMap].partition;
        let domainSuffix = fact_tables_1.PARTITION_MAP[defaultMap].domainSuffix;
        for (const key in fact_tables_1.PARTITION_MAP) {
            if (region.startsWith(key)) {
                partition = fact_tables_1.PARTITION_MAP[key].partition;
                domainSuffix = fact_tables_1.PARTITION_MAP[key].domainSuffix;
            }
        }
        registerFact(region, 'PARTITION', partition);
        registerFact(region, 'DOMAIN_SUFFIX', domainSuffix);
        registerFact(region, 'CDK_METADATA_RESOURCE_AVAILABLE', fact_tables_1.AWS_CDK_METADATA.has(region) ? 'YES' : 'NO');
        registerFact(region, 'IS_OPT_IN_REGION', partition === 'aws' && after(region, aws_entities_1.RULE_CLASSIC_PARTITION_BECOMES_OPT_IN) ? 'YES' : 'NO');
        registerFact(region, 'S3_STATIC_WEBSITE_ENDPOINT', (0, aws_entities_1.before)(region, aws_entities_1.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN)
            ? `s3-website-${region}.${domainSuffix}`
            : `s3-website.${region}.${domainSuffix}`);
        registerFact(region, 'S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID', fact_tables_1.ROUTE_53_BUCKET_WEBSITE_ZONE_IDS[region] || '');
        registerFact(region, 'EBS_ENV_ENDPOINT_HOSTED_ZONE_ID', fact_tables_1.EBS_ENV_ENDPOINT_HOSTED_ZONE_IDS[region] || '');
        registerFact(region, 'ELBV2_ACCOUNT', fact_tables_1.ELBV2_ACCOUNTS[region]);
        registerFact(region, 'DLC_REPOSITORY_ACCOUNT', fact_tables_1.DLC_REPOSITORY_ACCOUNTS[region]);
        registerFact(region, 'APPMESH_ECR_ACCOUNT', fact_tables_1.APPMESH_ECR_ACCOUNTS[region]);
        const firehoseCidrBlock = fact_tables_1.FIREHOSE_CIDR_BLOCKS[region];
        if (firehoseCidrBlock) {
            registerFact(region, 'FIREHOSE_CIDR_BLOCK', `${fact_tables_1.FIREHOSE_CIDR_BLOCKS[region]}/27`);
        }
        const vpcEndpointServiceNamePrefix = `${domainSuffix.split('.').reverse().join('.')}.vpce`;
        registerFact(region, 'VPC_ENDPOINT_SERVICE_NAME_PREFIX', vpcEndpointServiceNamePrefix);
        for (const service of aws_entities_1.AWS_SERVICES) {
            registerFact(region, ['servicePrincipal', service], default_1.Default.servicePrincipal(service, region, domainSuffix));
        }
        for (const version in fact_tables_1.CLOUDWATCH_LAMBDA_INSIGHTS_ARNS) {
            for (const arch in fact_tables_1.CLOUDWATCH_LAMBDA_INSIGHTS_ARNS[version]) {
                registerFact(region, ['cloudwatchLambdaInsightsVersion', version, arch], fact_tables_1.CLOUDWATCH_LAMBDA_INSIGHTS_ARNS[version][arch][region]);
            }
        }
        for (const version in fact_tables_1.APPCONFIG_LAMBDA_LAYER_ARNS) {
            for (const arch in fact_tables_1.APPCONFIG_LAMBDA_LAYER_ARNS[version]) {
                registerFact(region, ['appConfigLambdaLayerVersion', version, arch], fact_tables_1.APPCONFIG_LAMBDA_LAYER_ARNS[version][arch][region]);
            }
        }
        for (const type in fact_tables_1.ADOT_LAMBDA_LAYER_ARNS) {
            for (const version in fact_tables_1.ADOT_LAMBDA_LAYER_ARNS[type]) {
                for (const arch in fact_tables_1.ADOT_LAMBDA_LAYER_ARNS[type][version]) {
                    registerFact(region, ['adotLambdaLayer', type, version, arch], fact_tables_1.ADOT_LAMBDA_LAYER_ARNS[type][version][arch][region]);
                }
            }
        }
        for (const version in fact_tables_1.PARAMS_AND_SECRETS_LAMBDA_LAYER_ARNS) {
            for (const arch in fact_tables_1.PARAMS_AND_SECRETS_LAMBDA_LAYER_ARNS[version]) {
                registerFact(region, ['paramsAndSecretsLambdaLayer', version, arch], fact_tables_1.PARAMS_AND_SECRETS_LAMBDA_LAYER_ARNS[version][arch][region]);
            }
        }
    }
    lines.push('  }');
    lines.push('');
    lines.push('  private constructor() {}');
    lines.push('}');
    await fs.writeFile(path.resolve(__dirname, '..', 'lib', 'built-ins.generated.ts'), lines.join('\n'));
    function registerFact(region, name, value) {
        const factName = typeof name === 'string' ? name : `${name[0]}(${name.slice(1).map(s => JSON.stringify(s)).join(', ')})`;
        lines.push(`    Fact.register({ region: ${JSON.stringify(region)}, name: FactName.${factName}, value: ${JSON.stringify(value)} });`);
    }
}
exports.main = main;
/**
 * Verifies that the provided map of region to fact does not contain an entry
 * for a region that was not registered in `AWS_REGIONS`.
 */
function checkRegions(map) {
    const allRegions = new Set(aws_entities_1.AWS_REGIONS);
    for (const region of Object.keys(map)) {
        if (!allRegions.has(region)) {
            throw new Error(`Un-registered region fact found: ${region}. Add to AWS_REGIONS list!`);
        }
    }
}
/**
 * Verifies that the provided map of <KEY> to region to fact does not contain an entry
 * for a region that was not registered in `AWS_REGIONS`.
 */
function checkRegionsSubMap(map) {
    const allRegions = new Set(aws_entities_1.AWS_REGIONS);
    for (const key of Object.keys(map)) {
        for (const subKey of Object.keys(map[key])) {
            for (const region of Object.keys(map[key][subKey])) {
                if (!allRegions.has(region)) {
                    throw new Error(`Un-registered region fact found: ${region}. Add to AWS_REGIONS list!`);
                }
            }
        }
    }
}
function after(region, ruleOrRegion) {
    return region !== ruleOrRegion && !(0, aws_entities_1.before)(region, ruleOrRegion);
}
exports.after = after;
main().catch(e => {
    // eslint-disable-next-line no-console
    console.error(e);
    process.exit(-1);
});
//# sourceMappingURL=data:application/json;base64,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