"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.partitionInformation = exports.regionsBefore = exports.before = exports.AWS_SERVICES = exports.AWS_REGIONS = exports.AWS_REGIONS_AND_RULES = exports.RULE_CLASSIC_PARTITION_BECOMES_OPT_IN = exports.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN = void 0;
/**
 * After this point, S3 website domains look like `s3-website.REGION.s3.amazonaws.com`
 *
 * Before this point, S3 website domains look like `s3-website-REGION.s3.amazonaws.com`.
 */
exports.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN = Symbol('S3_WEBSITE_REGIONAL_SUBDOMAIN');
/**
 * After this point, all regions in the 'aws' partition are opt-in.
 */
exports.RULE_CLASSIC_PARTITION_BECOMES_OPT_IN = Symbol('CLASSIC_PARTITION_BECOMES_OPT_IN');
/**
 * List of AWS region, ordered by launch date (oldest to newest)
 *
 * The significance of this is that standards and conventions change over time.
 * Generally, as rules are changed they only apply to new regions, and old
 * regions are left as-is.
 *
 * We mix the list of regions with a list of rules that were introduced over
 * time (rules are symbols).
 *
 * Therefore, if we want to know if a rule applies to a certain region, we
 * only need to check its position in the list and compare it to when a
 * rule was introduced.
 */
exports.AWS_REGIONS_AND_RULES = [
    'us-east-1',
    'eu-west-1',
    'us-west-1',
    'ap-southeast-1',
    'ap-northeast-1',
    'us-gov-west-1',
    'us-west-2',
    'sa-east-1',
    'ap-southeast-2',
    exports.RULE_S3_WEBSITE_REGIONAL_SUBDOMAIN,
    'cn-north-1',
    'us-iso-east-1',
    'eu-central-1',
    'ap-northeast-2',
    'ap-south-1',
    'us-east-2',
    'ca-central-1',
    'eu-west-2',
    'us-isob-east-1',
    'cn-northwest-1',
    'eu-west-3',
    'ap-northeast-3',
    'us-gov-east-1',
    'eu-north-1',
    exports.RULE_CLASSIC_PARTITION_BECOMES_OPT_IN,
    'ap-east-1',
    'me-south-1',
    'eu-south-1',
    'af-south-1',
    'us-iso-west-1',
    'eu-south-2',
    'ap-southeast-3',
    'me-central-1',
    'ap-south-2',
    'eu-central-2',
    'il-central-1',
    'ap-southeast-4', // Asia Pacific (Melbourne)
];
/**
 * The names of all (known) AWS regions
 *
 * Not in the list ==> no built-in data for that region.
 */
exports.AWS_REGIONS = exports.AWS_REGIONS_AND_RULES
    .filter((x) => typeof x === 'string')
    .sort();
/**
 * Possibly non-exhaustive list of all service names, used to locate service principals.
 *
 * Not in the list ==> default service principal mappings.
 */
exports.AWS_SERVICES = [
    'application-autoscaling',
    'autoscaling',
    'codedeploy',
    'ec2',
    'events',
    'lambda',
    'logs',
    's3',
    'ssm',
    'sns',
    'sqs',
    'states',
].sort();
/**
 * Whether or not a region predates a given rule (or region).
 *
 * Unknown region => we have to assume no.
 */
function before(region, ruleOrRegion) {
    const ruleIx = exports.AWS_REGIONS_AND_RULES.indexOf(ruleOrRegion);
    if (ruleIx === -1) {
        throw new Error(`Unknown rule: ${String(ruleOrRegion)}`);
    }
    const regionIx = exports.AWS_REGIONS_AND_RULES.indexOf(region);
    return regionIx === -1 ? false : regionIx < ruleIx;
}
exports.before = before;
/**
 * Return all regions before a given rule was introduced (or region)
 */
function regionsBefore(ruleOrRegion) {
    const ruleIx = exports.AWS_REGIONS_AND_RULES.indexOf(ruleOrRegion);
    if (ruleIx === -1) {
        throw new Error(`Unknown rule: ${String(ruleOrRegion)}`);
    }
    return exports.AWS_REGIONS_AND_RULES.slice(0, ruleIx)
        .filter((entry) => typeof entry === 'string')
        .sort();
}
exports.regionsBefore = regionsBefore;
const PARTITION_MAP = {
    'default': { partition: 'aws', domainSuffix: 'amazonaws.com' },
    'cn-': { partition: 'aws-cn', domainSuffix: 'amazonaws.com.cn' },
    'us-gov-': { partition: 'aws-us-gov', domainSuffix: 'amazonaws.com' },
    'us-iso-': { partition: 'aws-iso', domainSuffix: 'c2s.ic.gov' },
    'us-isob-': { partition: 'aws-iso-b', domainSuffix: 'sc2s.sgov.gov' },
};
function partitionInformation(region) {
    for (const [prefix, info] of Object.entries(PARTITION_MAP)) {
        if (region.startsWith(prefix)) {
            return info;
        }
    }
    return PARTITION_MAP.default;
}
exports.partitionInformation = partitionInformation;
//# sourceMappingURL=data:application/json;base64,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