"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FactName = exports.Fact = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_entities_1 = require("./aws-entities");
/**
 * A database of regional information.
 */
class Fact {
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     *          may not be an exhaustive list of all available AWS regions.
     */
    static get regions() {
        // Return by copy to ensure no modifications can be made to the undelying constant.
        return Array.from(aws_entities_1.AWS_REGIONS);
    }
    /**
     * Retrieves a fact from this Fact database.
     *
     * @param region the name of the region (e.g: `us-east-1`)
     * @param name   the name of the fact being looked up (see the `FactName` class for details)
     *
     * @returns the fact value if it is known, and `undefined` otherwise.
     */
    static find(region, name) {
        const regionFacts = this.database[region];
        return regionFacts && regionFacts[name];
    }
    /**
     * Retrieve a fact from the Fact database. (retrieval will fail if the specified region or
     * fact name does not exist.)
     *
     * @param region the name of the region (e.g: `us-east-1`)
     * @param name the name of the fact being looked up (see the `FactName` class for details)
     */
    static requireFact(region, name) {
        const foundFact = this.find(region, name);
        if (!foundFact) {
            throw new Error(`No fact ${name} could be found for region: ${region} and name: ${name}`);
        }
        return foundFact;
    }
    /**
     * Registers a new fact in this Fact database.
     *
     * @param fact           the new fact to be registered.
     * @param allowReplacing whether new facts can replace existing facts or not.
     */
    static register(fact, allowReplacing = false) {
        try {
            jsiiDeprecationWarnings._aws_cdk_region_info_IFact(fact);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.register);
            }
            throw error;
        }
        const regionFacts = this.database[fact.region] || (this.database[fact.region] = {});
        if (fact.name in regionFacts && regionFacts[fact.name] !== fact.value && !allowReplacing) {
            throw new Error(`Region ${fact.region} already has a fact ${fact.name}, with value ${regionFacts[fact.name]}`);
        }
        if (fact.value !== undefined) {
            regionFacts[fact.name] = fact.value;
        }
    }
    /**
     * Removes a fact from the database.
     * @param region the region for which the fact is to be removed.
     * @param name   the name of the fact to remove.
     * @param value  the value that should be removed (removal will fail if the value is specified, but does not match the
     *               current stored value).
     */
    static unregister(region, name, value) {
        const regionFacts = this.database[region] || {};
        if (name in regionFacts && value && regionFacts[name] !== value) {
            throw new Error(`Attempted to remove ${name} from ${region} with value ${value}, but the fact's value is ${regionFacts[name]}`);
        }
        delete regionFacts[name];
    }
    constructor() {
        throw new Error('Use the static methods of Fact instead!');
    }
}
exports.Fact = Fact;
_a = JSII_RTTI_SYMBOL_1;
Fact[_a] = { fqn: "@aws-cdk/region-info.Fact", version: "2.97.0" };
Fact.database = {};
/**
 * All standardized fact names.
 */
class FactName {
    /**
     * The ARN of CloudWatch Lambda Insights for a version (e.g. 1.0.98.0)
     */
    static cloudwatchLambdaInsightsVersion(version, arch) {
        // if we are provided an architecture use that, otherwise
        // default to x86_64 for backwards compatibility
        const suffix = version.split('.').join('_') + `_${arch ?? 'x86_64'}`;
        return `cloudwatch-lambda-insights-version:${suffix}`;
    }
    /**
     * The ARN of AppConfig Lambda Layer for a given version (e.g. 2.0.181)
     * @param version The layer version.
     * @param arch The architecture (optional), defaults to x86_64.
     */
    static appConfigLambdaLayerVersion(version, arch) {
        const suffix = version.split('.').join('_') + `_${arch ?? 'x86_64'}`;
        return `appconfig-lambda-layer:${suffix}`;
    }
    /**
     * The name of the regional service principal for a given service.
     *
     * @param service the service name, either simple (e.g: `s3`, `codedeploy`) or qualified (e.g: `s3.amazonaws.com`).
     *                The `.amazonaws.com` and `.amazonaws.com.cn` domains are stripped from service names, so they are
     *                canonicalized in that respect.
     */
    static servicePrincipal(service) {
        return `service-principal:${service.replace(/\.amazonaws\.com(\.cn)?$/, '')}`;
    }
    /**
     * The ARN of Amazon Distro for OpenTelemetry (ADOT) Lambda layer for a given lambda type, version and architecture.
     *
     * @param type the type of the ADOT lambda layer
     * @param version the layer version.
     * @param architecture the Lambda Function architecture (e.g. 'x86_64' or 'arm64')
     */
    static adotLambdaLayer(type, version, architecture) {
        const suffix = type + '_' + version.split('.').join('_') + '_' + architecture;
        return `adot-lambda-layer:${suffix}`;
    }
    /**
     * The ARN of Parameters and Secrets Lambda layer for a given lambda architecture.
     *
     * @param version the layer version
     * @param architecture the Lambda Function architecture (e.g. 'x86_64' or 'arm64')
     */
    static paramsAndSecretsLambdaLayer(version, architecture) {
        const suffix = version.split('.').join('_') + `_${architecture}`;
        return `params-and-secrets-layer:${suffix}`;
    }
}
exports.FactName = FactName;
_b = JSII_RTTI_SYMBOL_1;
FactName[_b] = { fqn: "@aws-cdk/region-info.FactName", version: "2.97.0" };
/**
 * The name of the partition for a region (e.g: 'aws', 'aws-cn', ...)
 */
FactName.PARTITION = 'partition';
/**
 * The domain suffix for a region (e.g: 'amazonaws.com`)
 */
FactName.DOMAIN_SUFFIX = 'domainSuffix';
/**
 * Whether the AWS::CDK::Metadata CloudFormation Resource is available in-region or not. The value is a boolean
 * modelled as `YES` or `NO`.
 */
FactName.CDK_METADATA_RESOURCE_AVAILABLE = 'cdk:metadata-resource:available';
/**
 * Whether the given region is an opt-in region or not. The value is a boolean
 * modelled as `YES` or `NO`.
 */
FactName.IS_OPT_IN_REGION = 'aws:is-opt-in-region';
/**
 * The endpoint used for hosting S3 static websites
 */
FactName.S3_STATIC_WEBSITE_ENDPOINT = 's3-static-website:endpoint';
/**
 * The endpoint used for aliasing S3 static websites in Route 53
 */
FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID = 's3-static-website:route-53-hosted-zone-id';
/**
 * The hosted zone ID used by Route 53 to alias a EBS environment endpoint in this region (e.g: Z2O1EMRO9K5GLX)
 */
FactName.EBS_ENV_ENDPOINT_HOSTED_ZONE_ID = 'ebs-environment:route-53-hosted-zone-id';
/**
 * The prefix for VPC Endpoint Service names,
 * cn.com.amazonaws.vpce for China regions,
 * com.amazonaws.vpce otherwise.
 */
FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX = 'vpcEndpointServiceNamePrefix';
/**
 * The account for ELBv2 in this region
 */
FactName.ELBV2_ACCOUNT = 'elbv2Account';
/**
 * The ID of the AWS account that owns the public ECR repository that contains the
 * AWS Deep Learning Containers images in a given region.
 */
FactName.DLC_REPOSITORY_ACCOUNT = 'dlcRepositoryAccount';
/**
 * The ID of the AWS account that owns the public ECR repository that contains the
 * AWS App Mesh Envoy Proxy images in a given region.
 */
FactName.APPMESH_ECR_ACCOUNT = 'appMeshRepositoryAccount';
/**
 * The CIDR block used by Kinesis Data Firehose servers.
 */
FactName.FIREHOSE_CIDR_BLOCK = 'firehoseCidrBlock';
//# sourceMappingURL=data:application/json;base64,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