"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RegionInfo = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_entities_1 = require("./aws-entities");
const fact_1 = require("./fact");
/**
 * Information pertaining to an AWS region.
 */
class RegionInfo {
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     *          may not be an exaustive list of all available AWS regions.
     */
    static get regions() {
        return fact_1.Fact.regions.map(RegionInfo.get);
    }
    /**
     * Retrieves a collection of all fact values for all regions that fact is defined in.
     *
     * @param factName the name of the fact to retrieve values for.
     *   For a list of common fact names, see the FactName class
     * @returns a mapping with AWS region codes as the keys,
     *   and the fact in the given region as the value for that key
     */
    static regionMap(factName) {
        const ret = {};
        for (const regionInfo of RegionInfo.regions) {
            const fact = fact_1.Fact.find(regionInfo.name, factName);
            if (fact) {
                ret[regionInfo.name] = fact;
            }
        }
        return ret;
    }
    /**
     * Retrieves a collection of all fact values for all regions, limited to some partitions
     *
     * @param factName the name of the fact to retrieve values for.
     *   For a list of common fact names, see the FactName class
     * @param partitions list of partitions to retrieve facts for. Defaults
     *   to `['aws', 'aws-cn']`.
     * @returns a mapping with AWS region codes as the keys,
     *   and the fact in the given region as the value for that key
     */
    static limitedRegionMap(factName, partitions) {
        const ret = {};
        for (const [region, value] of Object.entries(RegionInfo.regionMap(factName))) {
            if (partitions.includes((0, aws_entities_1.partitionInformation)(region).partition)) {
                ret[region] = value;
            }
        }
        return ret;
    }
    /**
     * Obtain region info for a given region name.
     *
     * @param name the name of the region (e.g: us-east-1)
     */
    static get(name) {
        return new RegionInfo(name);
    }
    constructor(name) {
        this.name = name;
    }
    /**
     * Whether the `AWS::CDK::Metadata` CloudFormation Resource is available in this region or not.
     */
    get cdkMetadataResourceAvailable() {
        return fact_1.Fact.find(this.name, fact_1.FactName.CDK_METADATA_RESOURCE_AVAILABLE) === 'YES';
    }
    /**
     * Whether the given region is an opt-in region
     */
    get isOptInRegion() {
        return fact_1.Fact.find(this.name, fact_1.FactName.IS_OPT_IN_REGION) === 'YES';
    }
    /**
     * The domain name suffix (e.g: amazonaws.com) for this region.
     */
    get domainSuffix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.DOMAIN_SUFFIX);
    }
    /**
     * The name of the ARN partition for this region (e.g: aws).
     */
    get partition() {
        return fact_1.Fact.find(this.name, fact_1.FactName.PARTITION);
    }
    /**
     * The endpoint used by S3 static website hosting in this region (e.g: s3-static-website-us-east-1.amazonaws.com)
     */
    get s3StaticWebsiteEndpoint() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ENDPOINT);
    }
    /**
     * The hosted zone ID used by Route 53 to alias a S3 static website in this region (e.g: Z2O1EMRO9K5GLX)
     */
    get s3StaticWebsiteHostedZoneId() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID);
    }
    /**
     * The hosted zone ID used by Route 53 to alias a EBS environment endpoint in this region (e.g: Z2O1EMRO9K5GLX)
     */
    get ebsEnvEndpointHostedZoneId() {
        return fact_1.Fact.find(this.name, fact_1.FactName.EBS_ENV_ENDPOINT_HOSTED_ZONE_ID);
    }
    /**
     * The prefix for VPC Endpoint Service names,
     * cn.com.amazonaws.vpce for China regions,
     * com.amazonaws.vpce otherwise.
     */
    get vpcEndpointServiceNamePrefix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX);
    }
    /**
     * The name of the service principal for a given service in this region.
     * @param service the service name (e.g: s3.amazonaws.com)
     */
    servicePrincipal(service) {
        return fact_1.Fact.find(this.name, fact_1.FactName.servicePrincipal(service));
    }
    /**
     * The account ID for ELBv2 in this region
     */
    get elbv2Account() {
        return fact_1.Fact.find(this.name, fact_1.FactName.ELBV2_ACCOUNT);
    }
    /**
     * The ID of the AWS account that owns the public ECR repository containing the
     * AWS Deep Learning Containers images in this region.
     */
    get dlcRepositoryAccount() {
        return fact_1.Fact.find(this.name, fact_1.FactName.DLC_REPOSITORY_ACCOUNT);
    }
    /**
     * The ARN of the CloudWatch Lambda Insights extension, for the given version.
     * @param insightsVersion the version (e.g. 1.0.98.0)
     * @param architecture the Lambda Function architecture (e.g. 'x86_64' or 'arm64')
     */
    cloudwatchLambdaInsightsArn(insightsVersion, architecture) {
        return fact_1.Fact.find(this.name, fact_1.FactName.cloudwatchLambdaInsightsVersion(insightsVersion, architecture));
    }
    /**
     * The ARN of the AppConfig Lambda Layer, for the given version.
     * @param layerVersion The layer version (e.g. 2.0.181)
     * @param architecture The Lambda Function architecture (e.g. 'x86_64' or 'arm64'), defaults to x86_64
     */
    appConfigLambdaArn(layerVersion, architecture) {
        return fact_1.Fact.find(this.name, fact_1.FactName.appConfigLambdaLayerVersion(layerVersion, architecture));
    }
    /**
     * The ID of the AWS account that owns the public ECR repository that contains the
     * AWS App Mesh Envoy Proxy images in a given region.
     */
    get appMeshRepositoryAccount() {
        return fact_1.Fact.find(this.name, fact_1.FactName.APPMESH_ECR_ACCOUNT);
    }
    /**
     * The CIDR block used by Kinesis Data Firehose servers.
     */
    get firehoseCidrBlock() {
        return fact_1.Fact.find(this.name, fact_1.FactName.FIREHOSE_CIDR_BLOCK);
    }
    /**
     * The ARN of the ADOT Lambda layer, for the given layer type, version and architecture.
     *
     * @param type the type of the ADOT lambda layer
     * @param version the layer version.
     * @param architecture the Lambda Function architecture (e.g. 'x86_64' or 'arm64')
     */
    adotLambdaLayerArn(type, version, architecture) {
        return fact_1.Fact.find(this.name, fact_1.FactName.adotLambdaLayer(type, version, architecture));
    }
    /**
     * The ARN of the Parameters and Secrets Lambda layer for the given lambda architecture.
     *
     * @param version the layer version
     * @param architecture the Lambda Function architecture (e.g. 'x86_64' or 'arm64')
     */
    paramsAndSecretsLambdaLayerArn(version, architecture) {
        return fact_1.Fact.find(this.name, fact_1.FactName.paramsAndSecretsLambdaLayer(version, architecture));
    }
}
exports.RegionInfo = RegionInfo;
_a = JSII_RTTI_SYMBOL_1;
RegionInfo[_a] = { fqn: "@aws-cdk/region-info.RegionInfo", version: "2.97.0" };
//# sourceMappingURL=data:application/json;base64,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