# MIT License
#
# Copyright (c) 2021 Eugenio Parodi <ceccopierangiolieugenio AT googlemail DOT com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

__all__ = ['TTkKeyEvent']

from TermTk.TTkCore.constant import TTkK

class TTkKeyEvent:
    ''' Keyboard Events

    :Demo: `test.input.py <https://github.com/ceccopierangiolieugenio/pyTermTk/blob/main/tests/test.input.py>`_

    :param type: The key input type recorded
    :type type: :py:class:`TTkConstant.KeyType`
    :param key: the key
    :type key: str
    :param code: The terminal code used to represent this input
    :type code: str
    :param mod: The modifier used by the :py:class:`~TermTk.TTkCore.constant.TTkConstant.KeyType.SpecialKey` type
    :type mod: :py:class:`TTkConstant.KeyModifier`

    .. py:attribute:: type
        :type: KeyType

        The key input :py:class:`TTkConstant.KeyType` recorded

    .. py:attribute:: key
        :type: str

        the input key

    .. py:attribute:: code
        :type: str

        The terminal code used to represent this input

    .. py:attribute:: mod
        :type: KeyModifier

        The :py:class:`TTkConstant.KeyModifier` used by the :py:class:`~TermTk.TTkCore.constant.TTkConstant.KeyType.SpecialKey` type

    '''
    __slots__ = ('type', 'key', 'code', 'mod')
    def __init__(self, type:int, key: str, code: str, mod: int):
        self.type = type
        self.key = key
        self.mod = mod
        self.code = code

    def __str__(self):
        code = self.code.replace('\033','<ESC>')
        return f"KeyEvent: {self.key} {key2str(self.key)} {mod2str(self.mod)} {code}"

    def __eq__(self, other):
        if other is None: return False
        return (
            self.type == other.type and
            self.key  == other.key  and
            self.mod  == other.mod  )

    def __hash__(self) -> int:
        return hash((self.type,self.key,self.mod))

    @staticmethod
    def parse(input_key):  # from: Space           except "DEL"
        if len(input_key) == 1 and "\040" <= input_key != "\177":
            return TTkKeyEvent(TTkK.Character, input_key, input_key, TTkK.NoModifier)
        else:
            key, mod = _translate_key(input_key)
            if key is not None:
                return TTkKeyEvent(TTkK.SpecialKey, key, input_key, mod)
        return None

_translate_key_list = {
        "\177"      : ( TTkK.Key_Backspace , TTkK.NoModifier ) ,
        "\t"        : ( TTkK.Key_Tab       , TTkK.NoModifier ) ,
        "\033[Z"    : ( TTkK.Key_Tab       , TTkK.ShiftModifier ) ,
        "\n"        : ( TTkK.Key_Enter     , TTkK.NoModifier ) ,
        "\r"        : ( TTkK.Key_Enter     , TTkK.NoModifier ) ,

        "\033[A"    : ( TTkK.Key_Up        , TTkK.NoModifier ) ,
        "\033[B"    : ( TTkK.Key_Down      , TTkK.NoModifier ) ,
        "\033[C"    : ( TTkK.Key_Right     , TTkK.NoModifier ) ,
        "\033[D"    : ( TTkK.Key_Left      , TTkK.NoModifier ) ,

        # macOS after Command is pressed
        "\033OA"    : ( TTkK.Key_Up        , TTkK.NoModifier ) ,
        "\033OB"    : ( TTkK.Key_Down      , TTkK.NoModifier ) ,
        "\033OC"    : ( TTkK.Key_Right     , TTkK.NoModifier ) ,
        "\033OD"    : ( TTkK.Key_Left      , TTkK.NoModifier ) ,

        "\033[1;2A" : ( TTkK.Key_Up        , TTkK.ShiftModifier ) ,
        "\033[1;2B" : ( TTkK.Key_Down      , TTkK.ShiftModifier ) ,
        "\033[1;2C" : ( TTkK.Key_Right     , TTkK.ShiftModifier ) ,
        "\033[1;2D" : ( TTkK.Key_Left      , TTkK.ShiftModifier ) ,

        "\033[1;3A" : ( TTkK.Key_Up        , TTkK.AltModifier ) ,
        "\033[1;3B" : ( TTkK.Key_Down      , TTkK.AltModifier ) ,
        "\033[1;3C" : ( TTkK.Key_Right     , TTkK.AltModifier ) ,
        "\033[1;3D" : ( TTkK.Key_Left      , TTkK.AltModifier ) ,

        "\033[1;4A" : ( TTkK.Key_Up        , TTkK.AltModifier | TTkK.ShiftModifier ) ,
        "\033[1;4B" : ( TTkK.Key_Down      , TTkK.AltModifier | TTkK.ShiftModifier ) ,
        "\033[1;4C" : ( TTkK.Key_Right     , TTkK.AltModifier | TTkK.ShiftModifier ) ,
        "\033[1;4D" : ( TTkK.Key_Left      , TTkK.AltModifier | TTkK.ShiftModifier ) ,

        "\033[1;5A" : ( TTkK.Key_Up        , TTkK.ControlModifier ) ,
        "\033[1;5B" : ( TTkK.Key_Down      , TTkK.ControlModifier ) ,
        "\033[1;5C" : ( TTkK.Key_Right     , TTkK.ControlModifier ) ,
        "\033[1;5D" : ( TTkK.Key_Left      , TTkK.ControlModifier ) ,

        "\033[1;6A" : ( TTkK.Key_Up        , TTkK.ControlModifier | TTkK.ShiftModifier ) ,
        "\033[1;6B" : ( TTkK.Key_Down      , TTkK.ControlModifier | TTkK.ShiftModifier ) ,
        "\033[1;6C" : ( TTkK.Key_Right     , TTkK.ControlModifier | TTkK.ShiftModifier ) ,
        "\033[1;6D" : ( TTkK.Key_Left      , TTkK.ControlModifier | TTkK.ShiftModifier ) ,

        "\033[1;8A" : ( TTkK.Key_Up        , TTkK.ControlModifier | TTkK.AltModifier | TTkK.ShiftModifier ) ,
        "\033[1;8B" : ( TTkK.Key_Down      , TTkK.ControlModifier | TTkK.AltModifier | TTkK.ShiftModifier ) ,
        "\033[1;8C" : ( TTkK.Key_Right     , TTkK.ControlModifier | TTkK.AltModifier | TTkK.ShiftModifier ) ,
        "\033[1;8D" : ( TTkK.Key_Left      , TTkK.ControlModifier | TTkK.AltModifier | TTkK.ShiftModifier ) ,

        "\033[5~"   : ( TTkK.Key_PageUp    , TTkK.NoModifier ) ,
        "\033[6~"   : ( TTkK.Key_PageDown  , TTkK.NoModifier ) ,
        "\033[5;2~" : ( TTkK.Key_PageUp    , TTkK.ShiftModifier ) ,
        "\033[6;2~" : ( TTkK.Key_PageDown  , TTkK.ShiftModifier ) ,
        "\033[5;3~" : ( TTkK.Key_PageUp    , TTkK.AltModifier ) ,
        "\033[6;3~" : ( TTkK.Key_PageDown  , TTkK.AltModifier ) ,
        "\033[5;4~" : ( TTkK.Key_PageUp    , TTkK.AltModifier | TTkK.ShiftModifier ) ,
        "\033[6;4~" : ( TTkK.Key_PageDown  , TTkK.AltModifier | TTkK.ShiftModifier ) ,
        "\033[5;5~" : ( TTkK.Key_PageUp    , TTkK.ControlModifier ) ,
        "\033[6;5~" : ( TTkK.Key_PageDown  , TTkK.ControlModifier ) ,
        "\033[5;6~" : ( TTkK.Key_PageUp    , TTkK.ControlModifier | TTkK.ShiftModifier ) ,
        "\033[6;6~" : ( TTkK.Key_PageDown  , TTkK.ControlModifier | TTkK.ShiftModifier ) ,
        "\033[5;7~" : ( TTkK.Key_PageUp    , TTkK.AltModifier | TTkK.ControlModifier ) ,
        "\033[6;7~" : ( TTkK.Key_PageDown  , TTkK.AltModifier | TTkK.ControlModifier ) ,

    # Xterm
        "\033[F"    : ( TTkK.Key_End       , TTkK.NoModifier ) ,
        "\033[H"    : ( TTkK.Key_Home      , TTkK.NoModifier ) ,
    # Terminator + tmux
        "\033[4~"   : ( TTkK.Key_End       , TTkK.NoModifier ) ,
        "\033[1~"   : ( TTkK.Key_Home      , TTkK.NoModifier ) ,
        "\033[2~"   : ( TTkK.Key_Insert    , TTkK.NoModifier ) ,
        "\033[3~"   : ( TTkK.Key_Delete    , TTkK.NoModifier ) ,
    # macOS - Ghostty
        "\033OF"    : ( TTkK.Key_End       , TTkK.NoModifier ) ,
        "\033OH"    : ( TTkK.Key_Home      , TTkK.NoModifier ) ,

        "\033[1;2F" : ( TTkK.Key_End       , TTkK.ShiftModifier ) ,
        "\033[1;2H" : ( TTkK.Key_Home      , TTkK.ShiftModifier ) ,
        "\033[2;2~" : ( TTkK.Key_Insert    , TTkK.ShiftModifier ) ,
        "\033[3;2~" : ( TTkK.Key_Delete    , TTkK.ShiftModifier ) ,

        "\033[1;3F" : ( TTkK.Key_End       , TTkK.AltModifier ) ,
        "\033[1;3H" : ( TTkK.Key_Home      , TTkK.AltModifier ) ,
        "\033[2;3~" : ( TTkK.Key_Insert    , TTkK.AltModifier ) ,
        "\033[3;3~" : ( TTkK.Key_Delete    , TTkK.AltModifier ) ,

        "\033[1;4F" : ( TTkK.Key_End       , TTkK.ShiftModifier | TTkK.AltModifier ) ,
        "\033[1;4H" : ( TTkK.Key_Home      , TTkK.ShiftModifier | TTkK.AltModifier ) ,
        "\033[2;4~" : ( TTkK.Key_Insert    , TTkK.ShiftModifier | TTkK.AltModifier ) ,
        "\033[3;4~" : ( TTkK.Key_Delete    , TTkK.ShiftModifier | TTkK.AltModifier ) ,

        "\033[1;5F" : ( TTkK.Key_End       , TTkK.ControlModifier ) ,
        "\033[1;5H" : ( TTkK.Key_Home      , TTkK.ControlModifier ) ,
        "\033[2;5~" : ( TTkK.Key_Insert    , TTkK.ControlModifier ) ,
        "\033[3;5~" : ( TTkK.Key_Delete    , TTkK.ControlModifier ) ,

        "\033[1;6F" : ( TTkK.Key_End       , TTkK.ShiftModifier | TTkK.ControlModifier) ,
        "\033[1;6H" : ( TTkK.Key_Home      , TTkK.ShiftModifier | TTkK.ControlModifier) ,
        "\033[2;6~" : ( TTkK.Key_Insert    , TTkK.ShiftModifier | TTkK.ControlModifier) ,
        "\033[3;6~" : ( TTkK.Key_Delete    , TTkK.ShiftModifier | TTkK.ControlModifier) ,

        "\033[1;7F" : ( TTkK.Key_End       , TTkK.AltModifier | TTkK.ControlModifier) ,
        "\033[1;7H" : ( TTkK.Key_Home      , TTkK.AltModifier | TTkK.ControlModifier) ,
        "\033[2;7~" : ( TTkK.Key_Insert    , TTkK.AltModifier | TTkK.ControlModifier) ,
        "\033[3;7~" : ( TTkK.Key_Delete    , TTkK.AltModifier | TTkK.ControlModifier) ,

        "\033[1;8F" : ( TTkK.Key_End       , TTkK.ShiftModifier | TTkK.AltModifier | TTkK.ControlModifier) ,
        "\033[1;8H" : ( TTkK.Key_Home      , TTkK.ShiftModifier | TTkK.AltModifier | TTkK.ControlModifier) ,
        "\033[2;8~" : ( TTkK.Key_Insert    , TTkK.ShiftModifier | TTkK.AltModifier | TTkK.ControlModifier) ,
        "\033[3;8~" : ( TTkK.Key_Delete    , TTkK.ShiftModifier | TTkK.AltModifier | TTkK.ControlModifier) ,

        "\033"      : ( TTkK.Key_Escape    , TTkK.NoModifier ) ,
    # Function Key
        "\033OP"    : ( TTkK.Key_F1        , TTkK.NoModifier ) ,
        "\033OQ"    : ( TTkK.Key_F2        , TTkK.NoModifier ) ,
        "\033OR"    : ( TTkK.Key_F3        , TTkK.NoModifier ) ,
        "\033OS"    : ( TTkK.Key_F4        , TTkK.NoModifier ) ,
        "\033[15~"  : ( TTkK.Key_F5        , TTkK.NoModifier ) ,
        "\033[17~"  : ( TTkK.Key_F6        , TTkK.NoModifier ) ,
        "\033[18~"  : ( TTkK.Key_F7        , TTkK.NoModifier ) ,
        "\033[19~"  : ( TTkK.Key_F8        , TTkK.NoModifier ) ,
        "\033[20~"  : ( TTkK.Key_F9        , TTkK.NoModifier ) ,
        "\033[21~"  : ( TTkK.Key_F10       , TTkK.NoModifier ) ,
        "\033[23~"  : ( TTkK.Key_F11       , TTkK.NoModifier ) ,
        "\033[24~"  : ( TTkK.Key_F12       , TTkK.NoModifier ) ,
        "\033[1;2P" : ( TTkK.Key_F1        , TTkK.ShiftModifier ) ,
        "\033[1;2Q" : ( TTkK.Key_F2        , TTkK.ShiftModifier ) ,
        "\033[1;2R" : ( TTkK.Key_F3        , TTkK.ShiftModifier ) ,
        "\033[1;2S" : ( TTkK.Key_F4        , TTkK.ShiftModifier ) ,
        "\033[15;2~": ( TTkK.Key_F5        , TTkK.ShiftModifier ) ,
        "\033[17;2~": ( TTkK.Key_F6        , TTkK.ShiftModifier ) ,
        "\033[18;2~": ( TTkK.Key_F7        , TTkK.ShiftModifier ) ,
        "\033[19;2~": ( TTkK.Key_F8        , TTkK.ShiftModifier ) ,
        "\033[20;2~": ( TTkK.Key_F9        , TTkK.ShiftModifier ) ,
        "\033[21;2~": ( TTkK.Key_F10       , TTkK.ShiftModifier ) ,
        "\033[23;2~": ( TTkK.Key_F11       , TTkK.ShiftModifier ) ,
        "\033[24;2~": ( TTkK.Key_F12       , TTkK.ShiftModifier ) ,
        "\033[1;5P" : ( TTkK.Key_F1        , TTkK.ControlModifier ) ,
        "\033[1;5Q" : ( TTkK.Key_F2        , TTkK.ControlModifier ) ,
        "\033[1;5R" : ( TTkK.Key_F3        , TTkK.ControlModifier ) ,
        "\033[1;5S" : ( TTkK.Key_F4        , TTkK.ControlModifier ) ,
        "\033[15;5~": ( TTkK.Key_F5        , TTkK.ControlModifier ) ,
        "\033[17;5~": ( TTkK.Key_F6        , TTkK.ControlModifier ) ,
        "\033[18;5~": ( TTkK.Key_F7        , TTkK.ControlModifier ) ,
        "\033[19;5~": ( TTkK.Key_F8        , TTkK.ControlModifier ) ,
        "\033[20;5~": ( TTkK.Key_F9        , TTkK.ControlModifier ) ,
        "\033[21;5~": ( TTkK.Key_F10       , TTkK.ControlModifier ) ,
        "\033[23;5~": ( TTkK.Key_F11       , TTkK.ControlModifier ) ,
        "\033[24;5~": ( TTkK.Key_F12       , TTkK.ControlModifier ) ,
        # "\033[1;3P" : ( TTkK.Key_F1        , TTkK.AltModifier ) ,
        # "\033[1;3Q" : ( TTkK.Key_F2        , TTkK.AltModifier ) ,
        "\033[1;3R" : ( TTkK.Key_F3        , TTkK.AltModifier ) ,
        # "\033[1;3S" : ( TTkK.Key_F4        , TTkK.AltModifier ) ,
        # "\033[15;3~": ( TTkK.Key_F5        , TTkK.AltModifier ) ,
        "\033[17;3~": ( TTkK.Key_F6        , TTkK.AltModifier ) ,
        # "\033[18;3~": ( TTkK.Key_F7        , TTkK.AltModifier ) ,
        # "\033[19;3~": ( TTkK.Key_F8        , TTkK.AltModifier ) ,
        "\033[20;3~": ( TTkK.Key_F9        , TTkK.AltModifier ) ,
        # "\033[21;3~": ( TTkK.Key_F10       , TTkK.AltModifier ) ,
        "\033[23;3~": ( TTkK.Key_F11       , TTkK.AltModifier ) ,
        "\033[24;3~": ( TTkK.Key_F12       , TTkK.AltModifier ) ,

        '\x01':       ( TTkK.Key_A         , TTkK.ControlModifier ),
        '\x02':       ( TTkK.Key_B         , TTkK.ControlModifier ),
        '\x03':       ( TTkK.Key_C         , TTkK.ControlModifier ),
        '\x04':       ( TTkK.Key_D         , TTkK.ControlModifier ),
        '\x05':       ( TTkK.Key_E         , TTkK.ControlModifier ),
        '\x06':       ( TTkK.Key_F         , TTkK.ControlModifier ),
        '\x07':       ( TTkK.Key_G         , TTkK.ControlModifier ),
        '\x08':       ( TTkK.Key_H         , TTkK.ControlModifier ),
        # '\x09':       ( TTkK.Key_I         , TTkK.ControlModifier ), # Tab   = '\t' = 0x09
        '\x0a':       ( TTkK.Key_J         , TTkK.ControlModifier ), # Enter = '\n' = 0x0a
        '\x0b':       ( TTkK.Key_K         , TTkK.ControlModifier ),
        '\x0c':       ( TTkK.Key_L         , TTkK.ControlModifier ),
        # '\x0d':       ( TTkK.Key_M         , TTkK.ControlModifier ), # Enter = '\r' = 0x0d
        '\x0e':       ( TTkK.Key_N         , TTkK.ControlModifier ),
        '\x0f':       ( TTkK.Key_O         , TTkK.ControlModifier ),
        '\x10':       ( TTkK.Key_P         , TTkK.ControlModifier ),
        '\x11':       ( TTkK.Key_Q         , TTkK.ControlModifier ),
        '\x12':       ( TTkK.Key_R         , TTkK.ControlModifier ),
        '\x13':       ( TTkK.Key_S         , TTkK.ControlModifier ),
        '\x14':       ( TTkK.Key_T         , TTkK.ControlModifier ),
        '\x15':       ( TTkK.Key_U         , TTkK.ControlModifier ),
        '\x16':       ( TTkK.Key_V         , TTkK.ControlModifier ),
        '\x17':       ( TTkK.Key_W         , TTkK.ControlModifier ),
        '\x18':       ( TTkK.Key_X         , TTkK.ControlModifier ),
        '\x19':       ( TTkK.Key_Y         , TTkK.ControlModifier ),
        '\x1a':       ( TTkK.Key_Z         , TTkK.ControlModifier ),

        '\033\x61':  ( TTkK.Key_A         , TTkK.AltModifier ),
        '\033\x62':  ( TTkK.Key_B         , TTkK.AltModifier ),
        '\033\x63':  ( TTkK.Key_C         , TTkK.AltModifier ),
        '\033\x64':  ( TTkK.Key_D         , TTkK.AltModifier ),
        '\033\x65':  ( TTkK.Key_E         , TTkK.AltModifier ),
        '\033\x66':  ( TTkK.Key_F         , TTkK.AltModifier ),
        '\033\x67':  ( TTkK.Key_G         , TTkK.AltModifier ),
        '\033\x68':  ( TTkK.Key_H         , TTkK.AltModifier ),
        '\033\x69':  ( TTkK.Key_I         , TTkK.AltModifier ),
        '\033\x6a':  ( TTkK.Key_J         , TTkK.AltModifier ),
        '\033\x6b':  ( TTkK.Key_K         , TTkK.AltModifier ),
        '\033\x6c':  ( TTkK.Key_L         , TTkK.AltModifier ),
        '\033\x6d':  ( TTkK.Key_M         , TTkK.AltModifier ),
        '\033\x6e':  ( TTkK.Key_N         , TTkK.AltModifier ),
        '\033\x6f':  ( TTkK.Key_O         , TTkK.AltModifier ),
        '\033\x70':  ( TTkK.Key_P         , TTkK.AltModifier ),
        '\033\x71':  ( TTkK.Key_Q         , TTkK.AltModifier ),
        '\033\x72':  ( TTkK.Key_R         , TTkK.AltModifier ),
        '\033\x73':  ( TTkK.Key_S         , TTkK.AltModifier ),
        '\033\x74':  ( TTkK.Key_T         , TTkK.AltModifier ),
        '\033\x75':  ( TTkK.Key_U         , TTkK.AltModifier ),
        '\033\x76':  ( TTkK.Key_V         , TTkK.AltModifier ),
        '\033\x77':  ( TTkK.Key_W         , TTkK.AltModifier ),
        '\033\x78':  ( TTkK.Key_X         , TTkK.AltModifier ),
        '\033\x79':  ( TTkK.Key_Y         , TTkK.AltModifier ),
        '\033\x7a':  ( TTkK.Key_Z         , TTkK.AltModifier ),

        '\033\x41':  ( TTkK.Key_A         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x42':  ( TTkK.Key_B         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x43':  ( TTkK.Key_C         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x44':  ( TTkK.Key_D         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x45':  ( TTkK.Key_E         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x46':  ( TTkK.Key_F         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x47':  ( TTkK.Key_G         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x48':  ( TTkK.Key_H         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x49':  ( TTkK.Key_I         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x4a':  ( TTkK.Key_J         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x4b':  ( TTkK.Key_K         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x4c':  ( TTkK.Key_L         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x4d':  ( TTkK.Key_M         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x4e':  ( TTkK.Key_N         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x4f':  ( TTkK.Key_O         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x50':  ( TTkK.Key_P         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x51':  ( TTkK.Key_Q         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x52':  ( TTkK.Key_R         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x53':  ( TTkK.Key_S         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x54':  ( TTkK.Key_T         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x55':  ( TTkK.Key_U         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x56':  ( TTkK.Key_V         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x57':  ( TTkK.Key_W         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x58':  ( TTkK.Key_X         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x59':  ( TTkK.Key_Y         , TTkK.AltModifier | TTkK.ShiftModifier ),
        '\033\x5a':  ( TTkK.Key_Z         , TTkK.AltModifier | TTkK.ShiftModifier ),

        '\033\x01':  ( TTkK.Key_A         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x02':  ( TTkK.Key_B         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x03':  ( TTkK.Key_C         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x04':  ( TTkK.Key_D         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x05':  ( TTkK.Key_E         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x06':  ( TTkK.Key_F         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x07':  ( TTkK.Key_G         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x08':  ( TTkK.Key_H         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x09':  ( TTkK.Key_I         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x0a':  ( TTkK.Key_J         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x0b':  ( TTkK.Key_K         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x0c':  ( TTkK.Key_L         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x0d':  ( TTkK.Key_M         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x0e':  ( TTkK.Key_N         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x0f':  ( TTkK.Key_O         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x10':  ( TTkK.Key_P         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x11':  ( TTkK.Key_Q         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x12':  ( TTkK.Key_R         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x13':  ( TTkK.Key_S         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x14':  ( TTkK.Key_T         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x15':  ( TTkK.Key_U         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x16':  ( TTkK.Key_V         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x17':  ( TTkK.Key_W         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x18':  ( TTkK.Key_X         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x19':  ( TTkK.Key_Y         , TTkK.ControlModifier | TTkK.AltModifier ),
        '\033\x1a':  ( TTkK.Key_Z         , TTkK.ControlModifier | TTkK.AltModifier ),

        # # "\033": return( ey_Tab ) ,
        # if True: return None
        # "\033": return( ey_Backtab ) ,
        # "\033": return( ey_Backspace ) ,
        # "\033": return( ey_Return ) ,
        # "\033": return( ey_Enter ) ,
        # "\033": return( ey_Pause ) ,
        # "\033": return( ey_Print ) ,
        # "\033": return( ey_SysReq ) ,
        # "\033": return( ey_Clear ) ,
        # "\033": return( ey_Shift ) ,
        # "\033": return( ey_Control ) ,
        # "\033": return( ey_Meta ) ,
        # "\033": return( ey_Alt ) ,
        # "\033": return( ey_AltGr ) ,
        # "\033": return( ey_CapsLock ) ,
        # "\033": return( ey_NumLock ) ,
        # "\033": return( ey_ScrollLock ) ,
        # "\033": return( ey_F13 ) ,
        # "\033": return( ey_F14 ) ,
        # "\033": return( ey_F15 ) ,
        # "\033": return( ey_F16 ) ,
        # "\033": return( ey_F17 ) ,
        # "\033": return( ey_F18 ) ,
        # "\033": return( ey_F19 ) ,
        # "\033": return( ey_F20 ) ,
        # "\033": return( ey_F21 ) ,
        # "\033": return( ey_F22 ) ,
        # "\033": return( ey_F23 ) ,
        # "\033": return( ey_F24 ) ,
        # "\033": return( ey_F25 ) ,
        # "\033": return( ey_F26 ) ,
        # "\033": return( ey_F27 ) ,
        # "\033": return( ey_F28 ) ,
        # "\033": return( ey_F29 ) ,
        # "\033": return( ey_F30 ) ,
        # "\033": return( ey_F31 ) ,
        # "\033": return( ey_F32 ) ,
        # "\033": return( ey_F33 ) ,
        # "\033": return( ey_F34 ) ,
        # "\033": return( ey_F35 ) ,
        # "\033": return( ey_Super_L ) ,
        # "\033": return( ey_Super_R ) ,
        # "\033": return( ey_Menu ) ,
        # "\033": return( ey_Hyper_L ) ,
        # "\033": return( ey_Hyper_R ) ,
        # "\033": return( ey_Help ) ,
        # "\033": return( ey_Direction_L ) ,
        # "\033": return( ey_Direction_R ) ,
        # "\033": return( ey_Space ) ,
        # "\033": return( ey_Any ) ,
        # return TTkK.NONE
    }

def _translate_key(key):
    return _translate_key_list.get(key,(None, None))

def mod2str(k):
    if k == TTkK.NoModifier         : return ""
    ret = []
    if k & TTkK.ShiftModifier       : ret.append("Shift")
    if k & TTkK.ControlModifier     : ret.append("Control")
    if k & TTkK.AltModifier         : ret.append("Alt")
    if k & TTkK.MetaModifier        : ret.append("Meta")
    if k & TTkK.KeypadModifier      : ret.append("Keypad")
    if k & TTkK.GroupSwitchModifier : ret.append("GroupSwitch")
    if ret: return ",".join(ret)
    return "NONE!!!"

_def_hey_map= {
        TTkK.Key_Escape                   :  "Key_Escape" ,
        TTkK.Key_Tab                      :  "Key_Tab" ,
        TTkK.Key_Backtab                  :  "Key_Backtab" ,
        TTkK.Key_Backspace                :  "Key_Backspace" ,
        TTkK.Key_Return                   :  "Key_Return" ,
        TTkK.Key_Enter                    :  "Key_Enter" ,
        TTkK.Key_Insert                   :  "Key_Insert" ,
        TTkK.Key_Delete                   :  "Key_Delete" ,
        TTkK.Key_Pause                    :  "Key_Pause" ,
        TTkK.Key_Print                    :  "Key_Print" ,
        TTkK.Key_SysReq                   :  "Key_SysReq" ,
        TTkK.Key_Clear                    :  "Key_Clear" ,
        TTkK.Key_Home                     :  "Key_Home" ,
        TTkK.Key_End                      :  "Key_End" ,
        TTkK.Key_Left                     :  "Key_Left" ,
        TTkK.Key_Up                       :  "Key_Up" ,
        TTkK.Key_Right                    :  "Key_Right" ,
        TTkK.Key_Down                     :  "Key_Down" ,
        TTkK.Key_PageUp                   :  "Key_PageUp" ,
        TTkK.Key_PageDown                 :  "Key_PageDown" ,
        TTkK.Key_Shift                    :  "Key_Shift" ,
        TTkK.Key_Control                  :  "Key_Control" ,
        TTkK.Key_Meta                     :  "Key_Meta" ,
        TTkK.Key_Alt                      :  "Key_Alt" ,
        TTkK.Key_AltGr                    :  "Key_AltGr" ,
        TTkK.Key_CapsLock                 :  "Key_CapsLock" ,
        TTkK.Key_NumLock                  :  "Key_NumLock" ,
        TTkK.Key_ScrollLock               :  "Key_ScrollLock" ,
        TTkK.Key_F1                       :  "Key_F1" ,
        TTkK.Key_F2                       :  "Key_F2" ,
        TTkK.Key_F3                       :  "Key_F3" ,
        TTkK.Key_F4                       :  "Key_F4" ,
        TTkK.Key_F5                       :  "Key_F5" ,
        TTkK.Key_F6                       :  "Key_F6" ,
        TTkK.Key_F7                       :  "Key_F7" ,
        TTkK.Key_F8                       :  "Key_F8" ,
        TTkK.Key_F9                       :  "Key_F9" ,
        TTkK.Key_F10                      :  "Key_F10" ,
        TTkK.Key_F11                      :  "Key_F11" ,
        TTkK.Key_F12                      :  "Key_F12" ,
        TTkK.Key_F13                      :  "Key_F13" ,
        TTkK.Key_F14                      :  "Key_F14" ,
        TTkK.Key_F15                      :  "Key_F15" ,
        TTkK.Key_F16                      :  "Key_F16" ,
        TTkK.Key_F17                      :  "Key_F17" ,
        TTkK.Key_F18                      :  "Key_F18" ,
        TTkK.Key_F19                      :  "Key_F19" ,
        TTkK.Key_F20                      :  "Key_F20" ,
        TTkK.Key_F21                      :  "Key_F21" ,
        TTkK.Key_F22                      :  "Key_F22" ,
        TTkK.Key_F23                      :  "Key_F23" ,
        TTkK.Key_F24                      :  "Key_F24" ,
        TTkK.Key_F25                      :  "Key_F25" ,
        TTkK.Key_F26                      :  "Key_F26" ,
        TTkK.Key_F27                      :  "Key_F27" ,
        TTkK.Key_F28                      :  "Key_F28" ,
        TTkK.Key_F29                      :  "Key_F29" ,
        TTkK.Key_F30                      :  "Key_F30" ,
        TTkK.Key_F31                      :  "Key_F31" ,
        TTkK.Key_F32                      :  "Key_F32" ,
        TTkK.Key_F33                      :  "Key_F33" ,
        TTkK.Key_F34                      :  "Key_F34" ,
        TTkK.Key_F35                      :  "Key_F35" ,
        TTkK.Key_Super_L                  :  "Key_Super_L" ,
        TTkK.Key_Super_R                  :  "Key_Super_R" ,
        TTkK.Key_Menu                     :  "Key_Menu" ,
        TTkK.Key_Hyper_L                  :  "Key_Hyper_L" ,
        TTkK.Key_Hyper_R                  :  "Key_Hyper_R" ,
        TTkK.Key_Help                     :  "Key_Help" ,
        TTkK.Key_Direction_L              :  "Key_Direction_L" ,
        TTkK.Key_Direction_R              :  "Key_Direction_R" ,
        TTkK.Key_Space                    :  "Key_Space" ,
        TTkK.Key_Any                      :  "Key_Any" ,
        TTkK.Key_Space                    :  "Key_Space" ,
        TTkK.Key_Exclam                   :  "Key_Exclam" ,
        TTkK.Key_QuoteDbl                 :  "Key_QuoteDbl" ,
        TTkK.Key_NumberSign               :  "Key_NumberSign" ,
        TTkK.Key_Dollar                   :  "Key_Dollar" ,
        TTkK.Key_Percent                  :  "Key_Percent" ,
        TTkK.Key_Ampersand                :  "Key_Ampersand" ,
        TTkK.Key_Apostrophe               :  "Key_Apostrophe" ,
        TTkK.Key_ParenLeft                :  "Key_ParenLeft" ,
        TTkK.Key_ParenRight               :  "Key_ParenRight" ,
        TTkK.Key_Asterisk                 :  "Key_Asterisk" ,
        TTkK.Key_Plus                     :  "Key_Plus" ,
        TTkK.Key_Comma                    :  "Key_Comma" ,
        TTkK.Key_Minus                    :  "Key_Minus" ,
        TTkK.Key_Period                   :  "Key_Period" ,
        TTkK.Key_Slash                    :  "Key_Slash" ,
        TTkK.Key_0                        :  "Key_0" ,
        TTkK.Key_1                        :  "Key_1" ,
        TTkK.Key_2                        :  "Key_2" ,
        TTkK.Key_3                        :  "Key_3" ,
        TTkK.Key_4                        :  "Key_4" ,
        TTkK.Key_5                        :  "Key_5" ,
        TTkK.Key_6                        :  "Key_6" ,
        TTkK.Key_7                        :  "Key_7" ,
        TTkK.Key_8                        :  "Key_8" ,
        TTkK.Key_9                        :  "Key_9" ,
        TTkK.Key_Colon                    :  "Key_Colon" ,
        TTkK.Key_Semicolon                :  "Key_Semicolon" ,
        TTkK.Key_Less                     :  "Key_Less" ,
        TTkK.Key_Equal                    :  "Key_Equal" ,
        TTkK.Key_Greater                  :  "Key_Greater" ,
        TTkK.Key_Question                 :  "Key_Question" ,
        TTkK.Key_At                       :  "Key_At" ,
        TTkK.Key_A                        :  "Key_A" ,
        TTkK.Key_B                        :  "Key_B" ,
        TTkK.Key_C                        :  "Key_C" ,
        TTkK.Key_D                        :  "Key_D" ,
        TTkK.Key_E                        :  "Key_E" ,
        TTkK.Key_F                        :  "Key_F" ,
        TTkK.Key_G                        :  "Key_G" ,
        TTkK.Key_H                        :  "Key_H" ,
        TTkK.Key_I                        :  "Key_I" ,
        TTkK.Key_J                        :  "Key_J" ,
        TTkK.Key_K                        :  "Key_K" ,
        TTkK.Key_L                        :  "Key_L" ,
        TTkK.Key_M                        :  "Key_M" ,
        TTkK.Key_N                        :  "Key_N" ,
        TTkK.Key_O                        :  "Key_O" ,
        TTkK.Key_P                        :  "Key_P" ,
        TTkK.Key_Q                        :  "Key_Q" ,
        TTkK.Key_R                        :  "Key_R" ,
        TTkK.Key_S                        :  "Key_S" ,
        TTkK.Key_T                        :  "Key_T" ,
        TTkK.Key_U                        :  "Key_U" ,
        TTkK.Key_V                        :  "Key_V" ,
        TTkK.Key_W                        :  "Key_W" ,
        TTkK.Key_X                        :  "Key_X" ,
        TTkK.Key_Y                        :  "Key_Y" ,
        TTkK.Key_Z                        :  "Key_Z" ,
        TTkK.Key_BracketLeft              :  "Key_BracketLeft" ,
        TTkK.Key_Backslash                :  "Key_Backslash" ,
        TTkK.Key_BracketRight             :  "Key_BracketRight" ,
        TTkK.Key_AsciiCircum              :  "Key_AsciiCircum" ,
        TTkK.Key_Underscore               :  "Key_Underscore" ,
        TTkK.Key_QuoteLeft                :  "Key_QuoteLeft" ,
        TTkK.Key_BraceLeft                :  "Key_BraceLeft" ,
        TTkK.Key_Bar                      :  "Key_Bar" ,
        TTkK.Key_BraceRight               :  "Key_BraceRight" ,
        TTkK.Key_AsciiTilde               :  "Key_AsciiTilde" ,
        TTkK.Key_nobreakspace             :  "Key_nobreakspace" ,
        TTkK.Key_exclamdown               :  "Key_exclamdown" ,
        TTkK.Key_cent                     :  "Key_cent" ,
        TTkK.Key_sterling                 :  "Key_sterling" ,
        TTkK.Key_currency                 :  "Key_currency" ,
        TTkK.Key_yen                      :  "Key_yen" ,
        TTkK.Key_brokenbar                :  "Key_brokenbar" ,
        TTkK.Key_section                  :  "Key_section" ,
        TTkK.Key_diaeresis                :  "Key_diaeresis" ,
        TTkK.Key_copyright                :  "Key_copyright" ,
        TTkK.Key_ordfeminine              :  "Key_ordfeminine" ,
        TTkK.Key_guillemotleft            :  "Key_guillemotleft" ,
        TTkK.Key_notsign                  :  "Key_notsign" ,
        TTkK.Key_hyphen                   :  "Key_hyphen" ,
        TTkK.Key_registered               :  "Key_registered" ,
        TTkK.Key_macron                   :  "Key_macron" ,
        TTkK.Key_degree                   :  "Key_degree" ,
        TTkK.Key_plusminus                :  "Key_plusminus" ,
        TTkK.Key_twosuperior              :  "Key_twosuperior" ,
        TTkK.Key_threesuperior            :  "Key_threesuperior" ,
        TTkK.Key_acute                    :  "Key_acute" ,
        TTkK.Key_mu                       :  "Key_mu" ,
        TTkK.Key_paragraph                :  "Key_paragraph" ,
        TTkK.Key_periodcentered           :  "Key_periodcentered" ,
        TTkK.Key_cedilla                  :  "Key_cedilla" ,
        TTkK.Key_onesuperior              :  "Key_onesuperior" ,
        TTkK.Key_masculine                :  "Key_masculine" ,
        TTkK.Key_guillemotright           :  "Key_guillemotright" ,
        TTkK.Key_onequarter               :  "Key_onequarter" ,
        TTkK.Key_onehalf                  :  "Key_onehalf" ,
        TTkK.Key_threequarters            :  "Key_threequarters" ,
        TTkK.Key_questiondown             :  "Key_questiondown" ,
        TTkK.Key_Agrave                   :  "Key_Agrave" ,
        TTkK.Key_Aacute                   :  "Key_Aacute" ,
        TTkK.Key_Acircumflex              :  "Key_Acircumflex" ,
        TTkK.Key_Atilde                   :  "Key_Atilde" ,
        TTkK.Key_Adiaeresis               :  "Key_Adiaeresis" ,
        TTkK.Key_Aring                    :  "Key_Aring" ,
        TTkK.Key_AE                       :  "Key_AE" ,
        TTkK.Key_Ccedilla                 :  "Key_Ccedilla" ,
        TTkK.Key_Egrave                   :  "Key_Egrave" ,
        TTkK.Key_Eacute                   :  "Key_Eacute" ,
        TTkK.Key_Ecircumflex              :  "Key_Ecircumflex" ,
        TTkK.Key_Ediaeresis               :  "Key_Ediaeresis" ,
        TTkK.Key_Igrave                   :  "Key_Igrave" ,
        TTkK.Key_Iacute                   :  "Key_Iacute" ,
        TTkK.Key_Icircumflex              :  "Key_Icircumflex" ,
        TTkK.Key_Idiaeresis               :  "Key_Idiaeresis" ,
        TTkK.Key_ETH                      :  "Key_ETH" ,
        TTkK.Key_Ntilde                   :  "Key_Ntilde" ,
        TTkK.Key_Ograve                   :  "Key_Ograve" ,
        TTkK.Key_Oacute                   :  "Key_Oacute" ,
        TTkK.Key_Ocircumflex              :  "Key_Ocircumflex" ,
        TTkK.Key_Otilde                   :  "Key_Otilde" ,
        TTkK.Key_Odiaeresis               :  "Key_Odiaeresis" ,
        TTkK.Key_multiply                 :  "Key_multiply" ,
        TTkK.Key_Ooblique                 :  "Key_Ooblique" ,
        TTkK.Key_Ugrave                   :  "Key_Ugrave" ,
        TTkK.Key_Uacute                   :  "Key_Uacute" ,
        TTkK.Key_Ucircumflex              :  "Key_Ucircumflex" ,
        TTkK.Key_Udiaeresis               :  "Key_Udiaeresis" ,
        TTkK.Key_Yacute                   :  "Key_Yacute" ,
        TTkK.Key_THORN                    :  "Key_THORN" ,
        TTkK.Key_ssharp                   :  "Key_ssharp" ,
        TTkK.Key_division                 :  "Key_division" ,
        TTkK.Key_ydiaeresis               :  "Key_ydiaeresis" ,
        TTkK.Key_Multi_key                :  "Key_Multi_key" ,
        TTkK.Key_Codeinput                :  "Key_Codeinput" ,
        TTkK.Key_SingleCandidate          :  "Key_SingleCandidate" ,
        TTkK.Key_MultipleCandidate        :  "Key_MultipleCandidate" ,
        TTkK.Key_PreviousCandidate        :  "Key_PreviousCandidate" ,
        TTkK.Key_Mode_switch              :  "Key_Mode_switch" ,
        TTkK.Key_Kanji                    :  "Key_Kanji" ,
        TTkK.Key_Muhenkan                 :  "Key_Muhenkan" ,
        TTkK.Key_Henkan                   :  "Key_Henkan" ,
        TTkK.Key_Romaji                   :  "Key_Romaji" ,
        TTkK.Key_Hiragana                 :  "Key_Hiragana" ,
        TTkK.Key_Katakana                 :  "Key_Katakana" ,
        TTkK.Key_Hiragana_Katakana        :  "Key_Hiragana_Katakana" ,
        TTkK.Key_Zenkaku                  :  "Key_Zenkaku" ,
        TTkK.Key_Hankaku                  :  "Key_Hankaku" ,
        TTkK.Key_Zenkaku_Hankaku          :  "Key_Zenkaku_Hankaku" ,
        TTkK.Key_Touroku                  :  "Key_Touroku" ,
        TTkK.Key_Massyo                   :  "Key_Massyo" ,
        TTkK.Key_Kana_Lock                :  "Key_Kana_Lock" ,
        TTkK.Key_Kana_Shift               :  "Key_Kana_Shift" ,
        TTkK.Key_Eisu_Shift               :  "Key_Eisu_Shift" ,
        TTkK.Key_Eisu_toggle              :  "Key_Eisu_toggle" ,
        TTkK.Key_Hangul                   :  "Key_Hangul" ,
        TTkK.Key_Hangul_Start             :  "Key_Hangul_Start" ,
        TTkK.Key_Hangul_End               :  "Key_Hangul_End" ,
        TTkK.Key_Hangul_Hanja             :  "Key_Hangul_Hanja" ,
        TTkK.Key_Hangul_Jamo              :  "Key_Hangul_Jamo" ,
        TTkK.Key_Hangul_Romaja            :  "Key_Hangul_Romaja" ,
        TTkK.Key_Hangul_Jeonja            :  "Key_Hangul_Jeonja" ,
        TTkK.Key_Hangul_Banja             :  "Key_Hangul_Banja" ,
        TTkK.Key_Hangul_PreHanja          :  "Key_Hangul_PreHanja" ,
        TTkK.Key_Hangul_PostHanja         :  "Key_Hangul_PostHanja" ,
        TTkK.Key_Hangul_Special           :  "Key_Hangul_Special" ,
        TTkK.Key_Dead_Grave               :  "Key_Dead_Grave" ,
        TTkK.Key_Dead_Acute               :  "Key_Dead_Acute" ,
        TTkK.Key_Dead_Circumflex          :  "Key_Dead_Circumflex" ,
        TTkK.Key_Dead_Tilde               :  "Key_Dead_Tilde" ,
        TTkK.Key_Dead_Macron              :  "Key_Dead_Macron" ,
        TTkK.Key_Dead_Breve               :  "Key_Dead_Breve" ,
        TTkK.Key_Dead_Abovedot            :  "Key_Dead_Abovedot" ,
        TTkK.Key_Dead_Diaeresis           :  "Key_Dead_Diaeresis" ,
        TTkK.Key_Dead_Abovering           :  "Key_Dead_Abovering" ,
        TTkK.Key_Dead_Doubleacute         :  "Key_Dead_Doubleacute" ,
        TTkK.Key_Dead_Caron               :  "Key_Dead_Caron" ,
        TTkK.Key_Dead_Cedilla             :  "Key_Dead_Cedilla" ,
        TTkK.Key_Dead_Ogonek              :  "Key_Dead_Ogonek" ,
        TTkK.Key_Dead_Iota                :  "Key_Dead_Iota" ,
        TTkK.Key_Dead_Voiced_Sound        :  "Key_Dead_Voiced_Sound" ,
        TTkK.Key_Dead_Semivoiced_Sound    :  "Key_Dead_Semivoiced_Sound" ,
        TTkK.Key_Dead_Belowdot            :  "Key_Dead_Belowdot" ,
        TTkK.Key_Dead_Hook                :  "Key_Dead_Hook" ,
        TTkK.Key_Dead_Horn                :  "Key_Dead_Horn" ,
        TTkK.Key_Dead_Stroke              :  "Key_Dead_Stroke" ,
        TTkK.Key_Dead_Abovecomma          :  "Key_Dead_Abovecomma" ,
        TTkK.Key_Dead_Abovereversedcomma  :  "Key_Dead_Abovereversedcomma" ,
        TTkK.Key_Dead_Doublegrave         :  "Key_Dead_Doublegrave" ,
        TTkK.Key_Dead_Belowring           :  "Key_Dead_Belowring" ,
        TTkK.Key_Dead_Belowmacron         :  "Key_Dead_Belowmacron" ,
        TTkK.Key_Dead_Belowcircumflex     :  "Key_Dead_Belowcircumflex" ,
        TTkK.Key_Dead_Belowtilde          :  "Key_Dead_Belowtilde" ,
        TTkK.Key_Dead_Belowbreve          :  "Key_Dead_Belowbreve" ,
        TTkK.Key_Dead_Belowdiaeresis      :  "Key_Dead_Belowdiaeresis" ,
        TTkK.Key_Dead_Invertedbreve       :  "Key_Dead_Invertedbreve" ,
        TTkK.Key_Dead_Belowcomma          :  "Key_Dead_Belowcomma" ,
        TTkK.Key_Dead_Currency            :  "Key_Dead_Currency" ,
        TTkK.Key_Dead_a                   :  "Key_Dead_a" ,
        TTkK.Key_Dead_A                   :  "Key_Dead_A" ,
        TTkK.Key_Dead_e                   :  "Key_Dead_e" ,
        TTkK.Key_Dead_E                   :  "Key_Dead_E" ,
        TTkK.Key_Dead_i                   :  "Key_Dead_i" ,
        TTkK.Key_Dead_I                   :  "Key_Dead_I" ,
        TTkK.Key_Dead_o                   :  "Key_Dead_o" ,
        TTkK.Key_Dead_O                   :  "Key_Dead_O" ,
        TTkK.Key_Dead_u                   :  "Key_Dead_u" ,
        TTkK.Key_Dead_U                   :  "Key_Dead_U" ,
        TTkK.Key_Dead_Small_Schwa         :  "Key_Dead_Small_Schwa" ,
        TTkK.Key_Dead_Capital_Schwa       :  "Key_Dead_Capital_Schwa" ,
        TTkK.Key_Dead_Greek               :  "Key_Dead_Greek" ,
        TTkK.Key_Dead_Lowline             :  "Key_Dead_Lowline" ,
        TTkK.Key_Dead_Aboveverticalline   :  "Key_Dead_Aboveverticalline" ,
        TTkK.Key_Dead_Belowverticalline   :  "Key_Dead_Belowverticalline" ,
        TTkK.Key_Dead_Longsolidusoverlay  :  "Key_Dead_Longsolidusoverlay" ,
        TTkK.Key_Back                     :  "Key_Back" ,
        TTkK.Key_Forward                  :  "Key_Forward" ,
        TTkK.Key_Stop                     :  "Key_Stop" ,
        TTkK.Key_Refresh                  :  "Key_Refresh" ,
        TTkK.Key_VolumeDown               :  "Key_VolumeDown" ,
        TTkK.Key_VolumeMute               :  "Key_VolumeMute" ,
        TTkK.Key_VolumeUp                 :  "Key_VolumeUp" ,
        TTkK.Key_BassBoost                :  "Key_BassBoost" ,
        TTkK.Key_BassUp                   :  "Key_BassUp" ,
        TTkK.Key_BassDown                 :  "Key_BassDown" ,
        TTkK.Key_TrebleUp                 :  "Key_TrebleUp" ,
        TTkK.Key_TrebleDown               :  "Key_TrebleDown" ,
        TTkK.Key_MediaPlay                :  "Key_MediaPlay" ,
        TTkK.Key_MediaStop                :  "Key_MediaStop" ,
        TTkK.Key_MediaPrevious            :  "Key_MediaPrevious" ,
        TTkK.Key_MediaNext                :  "Key_MediaNext" ,
        TTkK.Key_MediaRecord              :  "Key_MediaRecord" ,
        TTkK.Key_MediaPause               :  "Key_MediaPause" ,
        TTkK.Key_MediaTogglePlayPause     :  "Key_MediaTogglePlayPause" ,
        TTkK.Key_HomePage                 :  "Key_HomePage" ,
        TTkK.Key_Favorites                :  "Key_Favorites" ,
        TTkK.Key_Search                   :  "Key_Search" ,
        TTkK.Key_Standby                  :  "Key_Standby" ,
        TTkK.Key_OpenUrl                  :  "Key_OpenUrl" ,
        TTkK.Key_LaunchMail               :  "Key_LaunchMail" ,
        TTkK.Key_LaunchMedia              :  "Key_LaunchMedia" ,
        TTkK.Key_Launch0                  :  "Key_Launch0" ,
        TTkK.Key_Launch1                  :  "Key_Launch1" ,
        TTkK.Key_Launch2                  :  "Key_Launch2" ,
        TTkK.Key_Launch3                  :  "Key_Launch3" ,
        TTkK.Key_Launch4                  :  "Key_Launch4" ,
        TTkK.Key_Launch5                  :  "Key_Launch5" ,
        TTkK.Key_Launch6                  :  "Key_Launch6" ,
        TTkK.Key_Launch7                  :  "Key_Launch7" ,
        TTkK.Key_Launch8                  :  "Key_Launch8" ,
        TTkK.Key_Launch9                  :  "Key_Launch9" ,
        TTkK.Key_LaunchA                  :  "Key_LaunchA" ,
        TTkK.Key_LaunchB                  :  "Key_LaunchB" ,
        TTkK.Key_LaunchC                  :  "Key_LaunchC" ,
        TTkK.Key_LaunchD                  :  "Key_LaunchD" ,
        TTkK.Key_LaunchE                  :  "Key_LaunchE" ,
        TTkK.Key_LaunchF                  :  "Key_LaunchF" ,
        TTkK.Key_LaunchG                  :  "Key_LaunchG" ,
        TTkK.Key_LaunchH                  :  "Key_LaunchH" ,
        TTkK.Key_MonBrightnessUp          :  "Key_MonBrightnessUp" ,
        TTkK.Key_MonBrightnessDown        :  "Key_MonBrightnessDown" ,
        TTkK.Key_KeyboardLightOnOff       :  "Key_KeyboardLightOnOff" ,
        TTkK.Key_KeyboardBrightnessUp     :  "Key_KeyboardBrightnessUp" ,
        TTkK.Key_KeyboardBrightnessDown   :  "Key_KeyboardBrightnessDown" ,
        TTkK.Key_PowerOff                 :  "Key_PowerOff" ,
        TTkK.Key_WakeUp                   :  "Key_WakeUp" ,
        TTkK.Key_Eject                    :  "Key_Eject" ,
        TTkK.Key_ScreenSaver              :  "Key_ScreenSaver" ,
        TTkK.Key_WWW                      :  "Key_WWW" ,
        TTkK.Key_Memo                     :  "Key_Memo" ,
        TTkK.Key_LightBulb                :  "Key_LightBulb" ,
        TTkK.Key_Shop                     :  "Key_Shop" ,
        TTkK.Key_History                  :  "Key_History" ,
        TTkK.Key_AddFavorite              :  "Key_AddFavorite" ,
        TTkK.Key_HotLinks                 :  "Key_HotLinks" ,
        TTkK.Key_BrightnessAdjust         :  "Key_BrightnessAdjust" ,
        TTkK.Key_Finance                  :  "Key_Finance" ,
        TTkK.Key_Community                :  "Key_Community" ,
        TTkK.Key_AudioRewind              :  "Key_AudioRewind" ,
        TTkK.Key_BackForward              :  "Key_BackForward" ,
        TTkK.Key_ApplicationLeft          :  "Key_ApplicationLeft" ,
        TTkK.Key_ApplicationRight         :  "Key_ApplicationRight" ,
        TTkK.Key_Book                     :  "Key_Book" ,
        TTkK.Key_CD                       :  "Key_CD" ,
        TTkK.Key_Calculator               :  "Key_Calculator" ,
        TTkK.Key_ToDoList                 :  "Key_ToDoList" ,
        TTkK.Key_ClearGrab                :  "Key_ClearGrab" ,
        TTkK.Key_Close                    :  "Key_Close" ,
        TTkK.Key_Copy                     :  "Key_Copy" ,
        TTkK.Key_Cut                      :  "Key_Cut" ,
        TTkK.Key_Display                  :  "Key_Display" ,
        TTkK.Key_DOS                      :  "Key_DOS" ,
        TTkK.Key_Documents                :  "Key_Documents" ,
        TTkK.Key_Excel                    :  "Key_Excel" ,
        TTkK.Key_Explorer                 :  "Key_Explorer" ,
        TTkK.Key_Game                     :  "Key_Game" ,
        TTkK.Key_Go                       :  "Key_Go" ,
        TTkK.Key_iTouch                   :  "Key_iTouch" ,
        TTkK.Key_LogOff                   :  "Key_LogOff" ,
        TTkK.Key_Market                   :  "Key_Market" ,
        TTkK.Key_Meeting                  :  "Key_Meeting" ,
        TTkK.Key_MenuKB                   :  "Key_MenuKB" ,
        TTkK.Key_MenuPB                   :  "Key_MenuPB" ,
        TTkK.Key_MySites                  :  "Key_MySites" ,
        TTkK.Key_News                     :  "Key_News" ,
        TTkK.Key_OfficeHome               :  "Key_OfficeHome" ,
        TTkK.Key_Option                   :  "Key_Option" ,
        TTkK.Key_Paste                    :  "Key_Paste" ,
        TTkK.Key_Phone                    :  "Key_Phone" ,
        TTkK.Key_Calendar                 :  "Key_Calendar" ,
        TTkK.Key_Reply                    :  "Key_Reply" ,
        TTkK.Key_Reload                   :  "Key_Reload" ,
        TTkK.Key_RotateWindows            :  "Key_RotateWindows" ,
        TTkK.Key_RotationPB               :  "Key_RotationPB" ,
        TTkK.Key_RotationKB               :  "Key_RotationKB" ,
        TTkK.Key_Save                     :  "Key_Save" ,
        TTkK.Key_Send                     :  "Key_Send" ,
        TTkK.Key_Spell                    :  "Key_Spell" ,
        TTkK.Key_SplitScreen              :  "Key_SplitScreen" ,
        TTkK.Key_Support                  :  "Key_Support" ,
        TTkK.Key_TaskPane                 :  "Key_TaskPane" ,
        TTkK.Key_Terminal                 :  "Key_Terminal" ,
        TTkK.Key_Tools                    :  "Key_Tools" ,
        TTkK.Key_Travel                   :  "Key_Travel" ,
        TTkK.Key_Video                    :  "Key_Video" ,
        TTkK.Key_Word                     :  "Key_Word" ,
        TTkK.Key_Xfer                     :  "Key_Xfer" ,
        TTkK.Key_ZoomIn                   :  "Key_ZoomIn" ,
        TTkK.Key_ZoomOut                  :  "Key_ZoomOut" ,
        TTkK.Key_Away                     :  "Key_Away" ,
        TTkK.Key_Messenger                :  "Key_Messenger" ,
        TTkK.Key_WebCam                   :  "Key_WebCam" ,
        TTkK.Key_MailForward              :  "Key_MailForward" ,
        TTkK.Key_Pictures                 :  "Key_Pictures" ,
        TTkK.Key_Music                    :  "Key_Music" ,
        TTkK.Key_Battery                  :  "Key_Battery" ,
        TTkK.Key_Bluetooth                :  "Key_Bluetooth" ,
        TTkK.Key_WLAN                     :  "Key_WLAN" ,
        TTkK.Key_UWB                      :  "Key_UWB" ,
        TTkK.Key_AudioForward             :  "Key_AudioForward" ,
        TTkK.Key_AudioRepeat              :  "Key_AudioRepeat" ,
        TTkK.Key_AudioRandomPlay          :  "Key_AudioRandomPlay" ,
        TTkK.Key_Subtitle                 :  "Key_Subtitle" ,
        TTkK.Key_AudioCycleTrack          :  "Key_AudioCycleTrack" ,
        TTkK.Key_Time                     :  "Key_Time" ,
        TTkK.Key_Hibernate                :  "Key_Hibernate" ,
        TTkK.Key_View                     :  "Key_View" ,
        TTkK.Key_TopMenu                  :  "Key_TopMenu" ,
        TTkK.Key_PowerDown                :  "Key_PowerDown" ,
        TTkK.Key_Suspend                  :  "Key_Suspend" ,
        TTkK.Key_ContrastAdjust           :  "Key_ContrastAdjust" ,
        TTkK.Key_TouchpadToggle           :  "Key_TouchpadToggle" ,
        TTkK.Key_TouchpadOn               :  "Key_TouchpadOn" ,
        TTkK.Key_TouchpadOff              :  "Key_TouchpadOff" ,
        TTkK.Key_MicMute                  :  "Key_MicMute" ,
        TTkK.Key_Red                      :  "Key_Red" ,
        TTkK.Key_Green                    :  "Key_Green" ,
        TTkK.Key_Yellow                   :  "Key_Yellow" ,
        TTkK.Key_Blue                     :  "Key_Blue" ,
        TTkK.Key_ChannelUp                :  "Key_ChannelUp" ,
        TTkK.Key_ChannelDown              :  "Key_ChannelDown" ,
        TTkK.Key_Guide                    :  "Key_Guide" ,
        TTkK.Key_Info                     :  "Key_Info" ,
        TTkK.Key_Settings                 :  "Key_Settings" ,
        TTkK.Key_MicVolumeUp              :  "Key_MicVolumeUp" ,
        TTkK.Key_MicVolumeDown            :  "Key_MicVolumeDown" ,
        TTkK.Key_New                      :  "Key_New" ,
        TTkK.Key_Open                     :  "Key_Open" ,
        TTkK.Key_Find                     :  "Key_Find" ,
        TTkK.Key_Undo                     :  "Key_Undo" ,
        TTkK.Key_Redo                     :  "Key_Redo" ,
        TTkK.Key_MediaLast                :  "Key_MediaLast" ,
        TTkK.Key_unknown                  :  "Key_unknown" ,
        TTkK.Key_Call                     :  "Key_Call" ,
        TTkK.Key_Camera                   :  "Key_Camera" ,
        TTkK.Key_CameraFocus              :  "Key_CameraFocus" ,
        TTkK.Key_Context1                 :  "Key_Context1" ,
        TTkK.Key_Context2                 :  "Key_Context2" ,
        TTkK.Key_Context3                 :  "Key_Context3" ,
        TTkK.Key_Context4                 :  "Key_Context4" ,
        TTkK.Key_Flip                     :  "Key_Flip" ,
        TTkK.Key_Hangup                   :  "Key_Hangup" ,
        TTkK.Key_No                       :  "Key_No" ,
        TTkK.Key_Select                   :  "Key_Select" ,
        TTkK.Key_Yes                      :  "Key_Yes" ,
        TTkK.Key_ToggleCallHangup         :  "Key_ToggleCallHangup" ,
        TTkK.Key_VoiceDial                :  "Key_VoiceDial" ,
        TTkK.Key_LastNumberRedial         :  "Key_LastNumberRedial" ,
        TTkK.Key_Execute                  :  "Key_Execute" ,
        TTkK.Key_Printer                  :  "Key_Printer" ,
        TTkK.Key_Play                     :  "Key_Play" ,
        TTkK.Key_Sleep                    :  "Key_Sleep" ,
        TTkK.Key_Zoom                     :  "Key_Zoom" ,
        TTkK.Key_Exit                     :  "Key_Exit" ,
        TTkK.Key_Cancel                   :  "Key_Cancel"
    }

def key2str(k):
    return _def_hey_map.get(k, "NONE!!!")