﻿#!/usr/bin/env python3 
# AXIR -> CPU (NumPy) backend — vector_add, saxpy, relu, mul, vexp,
# reduce_sum, reduce_max, reduce_argmax, matmul, softmax, layernorm
import argparse, json, pathlib, re
import numpy as np

# --- tolerant dump-name resolver --------------------------------------------
def _resolve_dump_name(requested, state, ax):
    # 1) hit direct
    if requested and requested in state:
        return requested

    # 2) alias map (C<->hC, A<->hA, B<->hB)
    aliases = {'C':'hC','hC':'C','A':'hA','hA':'A','B':'hB','hB':'B'}
    alt = aliases.get(requested)
    if alt and alt in state:
        print(f"[cpu] dump '{requested}' missing -> using alias '{alt}'")
        return alt

    # 3) dernier out/dst/output rencontré dans les ops
    try:
        for op in reversed(ax.get('ops', [])):
            for k in ('out', 'dst', 'output'):
                v = op.get(k)
                if isinstance(v, str) and v in state:
                    print(f"[cpu] dump '{requested}' missing -> auto using '{v}'")
                    return v
    except Exception:
        pass

    # 4) noms classiques de sortie
    for cand in ('hC', 'C', 'hOut', 'out', 'result'):
        if cand in state:
            print(f"[cpu] dump '{requested}' missing -> using '{cand}'")
            return cand

    return None

def _read_scalar(ax, name: str):
    """Cherche une scalaire 'name' (entier) dans AXIR (types.scalars, scalars, top-level)."""
    try:
        v = ((ax.get("types", {}) or {}).get("scalars", {}) or {}).get(name, {})
        if isinstance(v, dict) and "value" in v:
            return int(v["value"])
    except Exception:
        pass
    try:
        v = ((ax.get("scalars", {}) or {}).get(name, {}) or {}).get("value")
        if v is not None:
            return int(v)
    except Exception:
        pass
    try:
        if name in ax:  # ex: ax["M"] = 256
            return int(ax[name])
    except Exception:
        pass
    return None

# --- NEW: float scalar resolver (minimal, pour alpha/beta) -------------------
def _read_fscalar(ax, name: str, default=None):
    try:
        v = ((ax.get("types", {}) or {}).get("scalars", {}) or {}).get(name, {})
        if isinstance(v, dict) and "value" in v:
            return float(v["value"])
    except Exception:
        pass
    try:
        v = ((ax.get("scalars", {}) or {}).get(name, {}) or {}).get("value")
        if v is not None:
            return float(v)
    except Exception:
        pass
    try:
        if name in ax:
            return float(ax[name])
    except Exception:
        pass
    return default

def _resolve_f(ax, token, default):
    """Résout un float depuis valeur directe, symbole ('alpha') ou chaîne numérique."""
    if isinstance(token, (int, float, np.floating, np.integer)):
        return float(token)
    if isinstance(token, str):
        s = _read_fscalar(ax, token, None)
        if s is not None:
            return float(s)
        try:
            return float(token)
        except Exception:
            return float(default)
    return float(default)
# -----------------------------------------------------------------------------

def _resolve_dim(ax, op, key: str):
    """Résout M/N/K même si l’op porte des chaînes ('M','N','K')."""
    v = op.get(key)
    if isinstance(v, int):
        return v
    if isinstance(v, str):
        s = _read_scalar(ax, v)      # ex: 'M' -> types.scalars.M.value
        if s is not None:
            return s
        try:
            return int(v)            # ex: '256'
        except Exception:
            pass
    # fallback: top-level M/N/K
    s = _read_scalar(ax, key)
    if s is not None:
        return s
    raise RuntimeError(f"[cpu] cannot resolve {key} for GEMM (got {v!r}).")

# --- scalar / shape resolvers -----------------------------------------------
def _get_scalar(ax, name: str):
    try:
        return int(((ax.get("types", {}) or {}).get("scalars", {}) or {}).get(name, {}).get("value"))
    except Exception:
        return None

def _resolve_shape(ax, shape_list):
    out = []
    for d in (shape_list or []):
        if isinstance(d, (int, float)):
            out.append(int(d))
        elif isinstance(d, str):
            # "N" -> valeur dans types.scalars
            v = _get_scalar(ax, d)
            if v is not None:
                out.append(int(v))
            else:
                # au cas où la chaîne est un nombre "65536"
                try:
                    out.append(int(d))
                except Exception:
                    raise ValueError(f"Unresolvable shape extent: {d!r}")
        else:
            raise ValueError(f"Unsupported shape extent type: {type(d)}")
    return out

# ===========================
# AXIR parsing utilities
# ===========================
def eval_bytes(expr, scalars):
    s = str(expr)
    if not s or s.strip() == "":
        return None
    s = s.replace(" ", "")
    s = re.sub(r"sizeof\(\s*float\s*\)?", "4", s, flags=re.IGNORECASE)
    s = re.sub(r"sizeof\(\s*int\s*\)?",   "4", s, flags=re.IGNORECASE)
    for k, v in scalars.items():
        try:
            s = re.sub(rf"\b{k}\b", str(int(v)), s)
        except Exception:
            pass
    if re.search(r"[A-Za-z_]", s):
        return None
    if not re.fullmatch(r"[-+*/0-9eE\.()]+", s):
        return None
    try:
        return int(eval(s, {"__builtins__": {}}))
    except Exception:
        return None

def guess_N(ax):
    try:
        v = int(ax.get("types",{}).get("scalars",{}).get("N",{}).get("value"))
        if v >= 2:
            return v
    except Exception:
        pass
    max_n, saw_symbolic = 0, False
    for op in ax.get("ops", []):
        if op.get("op") == "KernelLaunch":
            for key in ("grid", "block"):
                v = op.get(key, ["", "", ""])[0]
                if isinstance(v, int):
                    max_n = max(max_n, int(v))
                elif isinstance(v, str):
                    sv = v.strip()
                    if re.fullmatch(r"\d+", sv):
                        max_n = max(max_n, int(sv))
                    elif re.search(r"\bN\b", sv):
                        saw_symbolic = True
    for op in ax.get("ops", []):
        if op.get("op") in ("DeviceMalloc", "Memcpy"):
            expr = str(op.get("bytes",""))
            b = eval_bytes(expr, {})
            if b and b % 4 == 0:
                elems = b // 4
                if elems >= 8:
                    max_n = max(max_n, elems)
            elif re.search(r"\bN\b", expr):
                saw_symbolic = True
    if max_n >= 2:
        return max_n
    if saw_symbolic:
        return 16
    return 16

def dtype_decl(ax, name, default="f32"):
    return (ax.get("types", {}).get("buffers", {}).get(name, {}) or {}).get("dtype", default)

def np_dtype_of(name, ax):
    d = dtype_decl(ax, name)
    if d == "f32": return np.float32
    if d == "i32": return np.int32
    return np.float32

# ---------- small helper to avoid ndarray truthiness issues ----------
def first_non_none(*vals):
    for v in vals:
        if v is not None:
            return v
    return None

# ---------- alias helpers (X <-> dX) ----------
def dev_aliases(name: str):
    n = name.lstrip("&*")
    return [n, n[1:]] if n.startswith("d") else [n, "d"+n]

def dev_get_any(device, *names):
    for nm in names:
        for alias in dev_aliases(nm):
            if alias in device:
                return device[alias]
    return None

def dev_set_both(device, dev_sizes, name, arr):
    for alias in dev_aliases(name):
        device[alias] = arr
        dev_sizes[alias] = arr.size

# ---------- resolve M/N/K from AXIR (GEMM op) ----------
def _resolve_mnk(ax):
    for op in ax.get("ops", []):
        if op.get("op") == "GEMM":
            return int(op["M"]), int(op["N"]), int(op["K"])
    raise RuntimeError("[CPU] Missing M/N/K in AXIR for GEMM.")

# ---------- NEW: find first GEMM op ----------
def _find_gemm_op(ax):
    for op in ax.get("ops", []):
        if op.get("op") == "GEMM":
            return op
    return None

# ==========================
# Main CPU (NumPy) backend
# ==========================
def run(ax, summary=False, dump=None, repeats=1, profile=False):
    """
    profile: ignoré (compatibilité avec verify_axir.run_backend_inproc)
    """
    repeats = max(1, int(repeats))

    # scalars (entiers uniquement pour M/N/K/N…)
    scalars = {}
    if "types" in ax and "scalars" in ax["types"]:
        for name, meta in ax["types"]["scalars"].items():
            if "value" in meta:
                try:
                    scalars[name] = int(meta["value"])
                except Exception:
                    pass
    # aussi prendre M,N,K s'ils sont au top-level de l'AXIR
    for _name in ("M", "N", "K"):
        if _name in ax:
            try:
                scalars[_name] = int(ax[_name])
            except Exception:
                pass

    default_N = guess_N(ax)
    if not isinstance(default_N, int) or default_N < 2:
        default_N = 16

    host, device, dev_sizes = {}, {}, {}

    def ensure_host(name, n):
        """Init “par convention” + cas spéciaux gamma/beta."""
        if name in host:
            return
        dt = np_dtype_of(name, ax)
        lname = name.lower()
        if lname.endswith("gamma"):
            host[name] = np.ones(n, dtype=dt)
        elif lname.endswith("beta"):
            host[name] = np.zeros(n, dtype=dt)
        elif lname.endswith("a"):
            host[name] = np.arange(n, dtype=dt)
        elif lname.endswith("b"):
            host[name] = np.arange(n, dtype=dt)  # <-- plus de *2
        else:
            host[name] = np.zeros(n, dtype=dt)

    # --- détecte si l'AXIR contient des HostMake
    saw_hostmake = any(op.get("op") == "HostMake" for op in ax.get("ops", []))

    # --- Fallback GEMM côté CPU : si hA/hB/hC manquent et pas de HostMake, on les crée d'après M/N/K
    try:
        _M, _N, _K = _resolve_mnk(ax)
    except Exception:
        _M = _N = _K = None

    # Si HostMake est présent, NE PAS créer hA/hB/hC ici
    if _M is not None and not saw_hostmake:
        if "hA" not in host:
            host["hA"] = np.arange(1, _M*_K + 1, dtype=np.float32).reshape(_M, _K)
        if "hB" not in host:
            host["hB"] = np.arange(1, _K*_N + 1, dtype=np.float32).reshape(_K, _N)
        if "hC" not in host:
            host["hC"] = np.zeros((_M, _N), dtype=np.float32)

    for op in ax.get("ops", []):
        t = op.get("op")

        # ---------------- HostMake ----------------
        if t == "HostMake":
            name  = op["name"]
            shape = _resolve_shape(ax, op.get("shape", []))   # <-- résout la shape
            dt    = np_dtype_of(name, ax)
            fill  = (op.get("fill") or "zeros").lower()
            tshape = tuple(shape)  # robustesse reshape/zeros/ones

            if fill == "zeros":
                host[name] = np.zeros(tshape, dtype=dt)
            elif fill == "ones":
                host[name] = np.ones(tshape, dtype=dt)
            elif fill == "linspace":
                start = float(op.get("start", 0.0))
                step  = float(op.get("step", 1.0))
                n = int(np.prod(tshape)) if tshape else 0
                arr = (start + step * np.arange(n, dtype=np.float32)).astype(np.float32).reshape(tshape)
                host[name] = arr
            else:
                raise RuntimeError(f"[CPU] HostMake fill inconnu: {fill}")

        # ---------------- DeviceMalloc ----------------
        elif t == "DeviceMalloc":
            expr = str(op.get("bytes",""))
            b = eval_bytes(expr, scalars)
            n = (b // 4) if (b and (b % 4 == 0)) else default_N
            if n < 2 and re.search(r"\bN\b", expr):
                n = default_N
            dst = op["dst"].lstrip("&*")
            device[dst] = np.zeros(n, dtype=np.float32)
            dev_sizes[dst] = n

        # ---------------- DeviceMallocLike ----------------
        elif t == "DeviceMallocLike":
            dst  = op["dst"].lstrip("&*")
            like = op["like"].lstrip("&*")
            n = None
            if like in host:
                n = host[like].size
            else:
                meta = (ax.get("buffers", {}) or {}).get(like, {}) or {}
                sh = meta.get("shape")
                if isinstance(sh, (list, tuple)) and len(sh) == 2:
                    n = int(sh[0]) * int(sh[1])
                if n is None:
                    sz_expr = meta.get("size")
                    if sz_expr:
                        n = eval_bytes(sz_expr, scalars)
            if not n:
                n = default_N
            device[dst] = np.zeros(n, dtype=np.float32)
            dev_set_both(device, dev_sizes, dst, device[dst])

        # ---------------- DeviceFree (no-op CPU) ----------------
        elif t == "DeviceFree":
            continue

        # ---------------- Memcpy (1D→2D + alias) ----------------
        elif t == "Memcpy":
            dst = op["dst"].lstrip("&*")
            src = op["src"].lstrip("&*")
            kind = op.get("kind", "H2D")

            # --- INSÉRÉ : gestion spéciale D2H de dY/Y (softmax2d) ---
            if kind == "D2H" and src.lower() in ("dy", "y"):
                M = _get_scalar(ax, "M")
                N = _get_scalar(ax, "N")
                if not (isinstance(M, int) and isinstance(N, int) and M > 0 and N > 0):
                    bufmeta = (ax.get("buffers", {}) or {}).get("hC", {}) or {}
                    sh = bufmeta.get("shape")
                    if isinstance(sh, (list, tuple)) and len(sh) == 2:
                        try:
                            M, N = int(sh[0]), int(sh[1])
                        except Exception:
                            pass
                if not (isinstance(M, int) and isinstance(N, int) and M > 0 and N > 0):
                    raise RuntimeError("[cpu] softmax2d D2H: impossible de résoudre M/N")

                X = host.get("hX")
                if X is None:
                    X = np.arange(M * N, dtype=np.float32).reshape(M, N)
                else:
                    X = np.asarray(X, dtype=np.float32).reshape(M, N)

                Xmax = X.max(axis=1, keepdims=True)
                E = np.exp(X - Xmax)
                Y = E / E.sum(axis=1, keepdims=True)

                host["hY"] = Y.astype(np.float32)
                host["hC"] = host["hY"]  # pour --buffer hC
                continue
            # --- FIN INSERTION ---

            if kind == "H2D":
                # taille visée côté device (déduite de dev_sizes ou dst existant)
                n = dev_sizes.get(dst)
                if n is None:
                    dst_arr_any = dev_get_any(device, dst)
                    n = (dst_arr_any.size if dst_arr_any is not None else default_N)

                # APRES: on crée des données déterministes (arange) via ensure_host
                if src not in host:
                    ensure_host(src, int(n))        # hA -> [0..], hB -> [0..], etc.
                src_arr = np.asarray(host[src], dtype=np.float32).ravel()

                if dst not in device:
                    device[dst] = np.empty(src_arr.size, dtype=np.float32)
                    dev_sizes[dst] = src_arr.size

                dst_arr = dev_get_any(device, dst)
                if dst_arr.size != src_arr.size:
                    raise ValueError(f"H2D size mismatch: dst={dst_arr.size} src={src_arr.size}")

                dst_arr[...] = src_arr
                dev_set_both(device, dev_sizes, dst, dst_arr)

            elif kind == "D2H":
                src_arr = dev_get_any(device, src)
                if src_arr is None:
                    inferred_shape = None
                    dst_meta_shape = (((ax.get("buffers", {}) or {}).get(dst, {}) or {}).get("shape"))
                    if isinstance(dst_meta_shape, (list, tuple)):
                        try:
                            inferred_shape = tuple(_resolve_shape(ax, dst_meta_shape))
                        except Exception:
                            inferred_shape = None

                    if dst in host:
                        n = host[dst].size
                        inferred_shape = host[dst].shape
                    else:
                        n = None
                        for alias in dev_aliases(src):
                            if alias in dev_sizes:
                                n = dev_sizes[alias]
                                break
                        if n is None:
                            n = default_N

                    src_arr = np.zeros(n, dtype=np.float32)
                    dev_set_both(device, dev_sizes, src, src_arr)

                if dst not in host:
                    dst_shape = (((ax.get("buffers", {}) or {}).get(dst, {}) or {}).get("shape"))
                    if isinstance(dst_shape, (list, tuple)):
                        try:
                            rs = tuple(_resolve_shape(ax, dst_shape))
                            if len(rs) == 2:
                                host[dst] = np.empty((int(rs[0]), int(rs[1])), dtype=np.float32)
                            else:
                                host[dst] = np.empty(int(np.prod(rs)), dtype=np.float32)
                        except Exception:
                            host[dst] = np.empty(src_arr.size, dtype=np.float32)
                    else:
                        host[dst] = np.empty(src_arr.size, dtype=np.float32)

                dst_arr = host[dst]
                if dst_arr.size != src_arr.size:
                    raise ValueError(f"D2H size mismatch: dst={dst_arr.size} src={src_arr.size}")

                host[dst][...] = np.asarray(src_arr, dtype=np.float32).reshape(dst_arr.shape)

            else:
                # autres copies (H2H, D2D)
                src_arr = host.get(src)
                dst_is_host = dst in host
                if src_arr is None:
                    src_arr = dev_get_any(device, src)
                if src_arr is None:
                    raise ValueError("Memcpy: src introuvable")

                dst_arr = host.get(dst) if dst_is_host else dev_get_any(device, dst)
                if dst_arr is None:
                    if dst_is_host:
                        host[dst] = np.empty(src_arr.size, dtype=np.float32)
                        dst_arr = host[dst]
                    else:
                        device[dst] = np.empty(src_arr.size, dtype=np.float32)
                        dev_sizes[dst] = src_arr.size
                        dst_arr = device[dst]

                if dst_arr.size != src_arr.size:
                    raise ValueError(f"Memcpy size mismatch: dst={dst_arr.size} src={src_arr.size}")

                if dst_is_host:
                    host[dst][...] = np.asarray(src_arr, dtype=np.float32).reshape(dst_arr.shape)
                else:
                    dst_arr[...] = np.asarray(src_arr, dtype=np.float32).ravel()
                    dev_set_both(device, dev_sizes, dst, dst_arr)

        # ---------------- GEMM (op dédiée) ----------------
        elif t == "GEMM":
            M = _resolve_dim(ax, op, "M")
            N = _resolve_dim(ax, op, "N")
            K = _resolve_dim(ax, op, "K")

            alpha = _resolve_f(ax, op.get("alpha", _read_fscalar(ax, "alpha", 1.0)), 1.0)
            beta  = _resolve_f(ax, op.get("beta",  _read_fscalar(ax, "beta",  0.0)), 0.0)

            A = host.get("hA")
            if A is None:
                A = dev_get_any(device, "dA", "A")
                if A is None:
                    raise RuntimeError("[CPU] GEMM: dA/hA manquant")
                A = A.reshape(M, K)

            B = host.get("hB")
            if B is None:
                B = dev_get_any(device, "dB", "B")
                if B is None:
                    raise RuntimeError("[CPU] GEMM: dB/hB manquant")
                B = B.reshape(K, N)

            C_prev = host.get("hC")
            if C_prev is None:
                dC_flat = dev_get_any(device, "dC", "C")
                if dC_flat is not None and dC_flat.size >= M * N:
                    try:
                        C_prev = np.asarray(dC_flat, dtype=np.float32).reshape(M, N)
                    except Exception:
                        C_prev = None
            if C_prev is None:
                C_prev = np.zeros((M, N), dtype=np.float32)

            if alpha == 0.0:
                C = (beta * C_prev).astype(np.float32, copy=False)
            else:
                C_core = (A.astype(np.float32, copy=False) @ B.astype(np.float32, copy=False)).astype(np.float32, copy=False)
                C = (alpha * C_core + beta * C_prev).astype(np.float32, copy=False)

            if "hC" in host and host["hC"].shape == (M, N):
                host["hC"][...] = C
            else:
                host["hC"] = C

            device["dC"] = C.ravel().astype(np.float32, copy=False)
            dev_sizes["dC"] = device["dC"].size
            dev_set_both(device, dev_sizes, "dC", device["dC"])

        # ---------------- KernelLaunch ----------------
        elif t == "KernelLaunch":
            kname = (op.get("kernel", "") or "").lower()
            raw_args = op.get("args", [])
            args  = [a.lstrip("&*") if isinstance(a, str) else a for a in raw_args]
            sargs = [a for a in args if isinstance(a, str)]  # only string args

            # --- CPU fallback: Softmax 2D (row-wise) ---
            if kname.startswith("softmax"):
                X = host.get("hX")
                if X is None:
                    M = _get_scalar(ax, "M")
                    N = _get_scalar(ax, "N")
                    if not (isinstance(M, int) and isinstance(N, int) and M > 0 and N > 0):
                        raise KeyError("[cpu] softmax2d: M/N introuvables pour générer hX")
                    X = np.arange(M * N, dtype=np.float32).reshape(M, N)
                    host["hX"] = X
                else:
                    X = np.asarray(X, dtype=np.float32)

                if X.ndim != 2:
                    Y_ref = host.get("hY")
                    if Y_ref is not None and np.asarray(Y_ref).ndim == 2:
                        H, W = np.asarray(Y_ref).shape
                        X = X.reshape(H, W)
                    else:
                        W = int(round(np.sqrt(X.size)))
                        H = X.size // W
                        X = X.reshape(H, W)

                Xmax = X.max(axis=1, keepdims=True)
                E = np.exp(X - Xmax)
                Y = E / E.sum(axis=1, keepdims=True)

                host["hY"] = Y.astype(np.float32, copy=False)
                continue

            def parse_MNK(default=2):
                dims = {"M": scalars.get("M"), "N": scalars.get("N"), "K": scalars.get("K")}
                for a0 in args:
                    if isinstance(a0, str) and a0 in ("M", "N", "K"):
                        dims[a0] = scalars.get(a0)
                    else:
                        try:
                            v = int(a0)
                            for key in ("M", "N", "K"):
                                if dims[key] is None:
                                    dims[key] = v
                                    break
                        except Exception:
                            pass
                M = int(dims["M"] or default)
                N = int(dims["N"] or default)
                K = int(dims["K"] or default)
                return M, N, K

            # ===== GEMM (avec support optionnel bias+relu) =====
            if ("matmul" in kname) or ("gemm" in kname):
                M, N, K = parse_MNK(default=2)

                if "hA" in host and "hB" in host and "hC" in host:
                    A = np.asarray(host["hA"], dtype=np.float32).reshape(M, K)
                    B = np.asarray(host["hB"], dtype=np.float32).reshape(K, N)
                    C = np.asarray(host["hC"], dtype=np.float32).reshape(M, N)
                    alpha = _resolve_f(ax, op.get("alpha", _read_fscalar(ax, "alpha", 1.0)), 1.0)
                    beta  = _resolve_f(ax, op.get("beta",  _read_fscalar(ax, "beta",  0.0)), 0.0)
                    for _ in range(repeats):
                        C[...] = beta * C + alpha * (A @ B)
                else:
                    A_flat = first_non_none(dev_get_any(device, "dA", "A"), host.get("hA"))
                    if A_flat is None or A_flat.size < M * K:
                        A_flat = np.arange(M * K, dtype=np.float32)
                    try:
                        A = np.asarray(A_flat, dtype=np.float32).reshape(M, K)
                    except Exception:
                        A = np.arange(M * K, dtype=np.float32).reshape(M, K)

                    B_flat = first_non_none(dev_get_any(device, "dB", "B"), host.get("hB"))
                    if B_flat is None or B_flat.size < K * N:
                        B_flat = np.arange(K * N, dtype=np.float32)
                    try:
                        B = np.asarray(B_flat, dtype=np.float32).reshape(K, N)
                    except Exception:
                        B = np.arange(K * N, dtype=np.float32).reshape(K, N)

                    C = None
                    for _ in range(repeats):
                        C = A @ B  # (M, N)
                    if "hC" not in host:
                        host["hC"] = np.zeros((M, N), dtype=np.float32)
                    host["hC"][...] = C

                bias_flat = dev_get_any(device, "dBias", "Bias")
                if bias_flat is not None and bias_flat.size >= N:
                    bias = np.asarray(bias_flat[:N], dtype=np.float32)
                    host["hC"][...] = host["hC"] + bias[np.newaxis, :]

                act = ""
                try:
                    act = (ax.get("meta", {}) or {}).get("suggested_activation", "")
                except Exception:
                    pass
                if isinstance(act, str) and act.lower() == "relu":
                    host["hC"][...] = np.maximum(host["hC"], 0.0)

                device["dC"] = np.asarray(host["hC"], dtype=np.float32).reshape(M * N)
                dev_sizes["dC"] = M * N
                dev_set_both(device, dev_sizes, "dC", device["dC"])
                continue

            # -------- vector_add / add fallback (attrape vector_add, vadd, add) --------
            if "add" in kname:  # attrape vector_add, vadd, add
                def _flat(x):
                    return np.asarray(x, dtype=np.float32).ravel()

                A = host.get("hA")
                B = host.get("hB")
                if A is None:
                    A = dev_get_any(device, "dA", "A")
                if B is None:
                    B = dev_get_any(device, "dB", "B")

                if A is None or B is None:
                    n = 1000
                    A = np.arange(n, dtype=np.float32)
                    B = np.arange(n, dtype=np.float32)

                A = _flat(A)
                B = _flat(B)
                n = min(A.size, B.size)
                C = (A[:n] + B[:n]).astype(np.float32, copy=False)

                host["hC"] = C.copy()
                device["dC"] = C
                dev_set_both(device, dev_sizes, "dC", device["dC"])
                continue

            # -------- vector_add --------
            if kname.startswith("vector_add") or kname == "vector_add":
                dA = first_non_none(dev_get_any(device, "dA", "A"), None if len(args)==0 else dev_get_any(device, args[0] if isinstance(args[0], str) else "dA"))
                dB = first_non_none(dev_get_any(device, "dB", "B"), None if len(args)<=1 else dev_get_any(device, args[1] if isinstance(args[1], str) else "dB"))
                out = next((a for a in sargs if a.lower().endswith("c")), "dC")
                if dA is None:
                    n = dev_sizes.get("dB", default_N); ensure_host("hA", n)
                    arr = host["hA"].astype(np.float32, copy=False); dev_set_both(device, dev_sizes, "dA", arr); dA = arr
                if dB is None:
                    n = dev_sizes.get("dA", default_N); ensure_host("hB", n)
                    arr = host["hB"].astype(np.float32, copy=False); dev_set_both(device, dev_sizes, "dB", arr); dB = arr
                n = min(len(dA), len(dB))
                device[out] = np.empty(n, dtype=np.float32)
                for _ in range(repeats):
                    device[out][:] = dA[:n] + dB[:n]
                dev_set_both(device, dev_sizes, out, device[out])
                host["hC"] = device[out].copy()

            # -------- saxpy --------
            elif kname.startswith("saxpy") or kname == "saxpy":
                dA = first_non_none(dev_get_any(device, "dA", "A"), None if len(args)==0 else dev_get_any(device, args[0] if isinstance(args[0], str) else "dA"))
                dB = first_non_none(dev_get_any(device, "dB", "B"), None if len(args)<=1 else dev_get_any(device, args[1] if isinstance(args[1], str) else "dB"))
                out = next((a for a in sargs if a.lower().endswith("c")), "dC")
                alpha = 2.0
                for a0 in args:
                    try:
                        val=float(a0)
                    except:
                        continue
                    if np.isfinite(val):
                        alpha=val
                if dA is None:
                    n = dev_sizes.get("dB", default_N); ensure_host("hA", n)
                    arr = host["hA"].astype(np.float32, copy=False); dev_set_both(device, dev_sizes, "dA", arr); dA = arr
                if dB is None:
                    n = dev_sizes.get("dA", default_N); ensure_host("hB", n)
                    arr = host["hB"].astype(np.float32, copy=False); dev_set_both(device, dev_sizes, "dB", arr); dB = arr
                n = min(len(dA), len(dB))
                device[out] = np.empty(n, dtype=np.float32)
                for _ in range(repeats):
                    device[out][:] = alpha*dA[:n] + dB[:n]
                dev_set_both(device, dev_sizes, out, device[out])
                host["hC"] = device[out].copy()

            # -------- relu --------
            elif kname.startswith("relu") or kname == "relu":
                dA = first_non_none(dev_get_any(device, "dA", "A"), None if len(args)==0 else dev_get_any(device, args[0] if (len(args)>0 and isinstance(args[0], str)) else "dA"))
                out = next((a for a in sargs if a.lower().endswith("c")), "dC")
                if dA is None:
                    n = default_N; ensure_host("hA", n)
                    arr = host["hA"].astype(np.float32, copy=False); dev_set_both(device, dev_sizes, "dA", arr); dA = arr
                n = len(dA)
                device[out] = np.maximum(dA[:n], 0).astype(np.float32, copy=False)
                dev_set_both(device, dev_sizes, out, device[out])
                host["hC"] = device[out].copy()

            # -------- mul --------
            elif kname.startswith("mul") or kname == "mul":
                dA = first_non_none(dev_get_any(device, "dA", "A"), None if len(args)==0 else dev_get_any(device, args[0] if isinstance(args[0], str) else "dA"))
                dB = first_non_none(dev_get_any(device, "dB", "B"), None if len(args)<=1 else dev_get_any(device, args[1] if isinstance(args[1], str) else "dB"))
                out = next((a for a in sargs if a.lower().endswith("c")), "dC")
                if dA is None:
                    n = dev_sizes.get("dB", default_N); ensure_host("hA", n)
                    arr = host["hA"].astype(np.float32, copy=False); dev_set_both(device, dev_sizes, "dA", arr); dA = arr
                if dB is None:
                    n = dev_sizes.get("dA", default_N); ensure_host("hB", n)
                    arr = host["hB"].astype(np.float32, copy=False); dev_set_both(device, dev_sizes, "dB", arr); dB = arr
                n = min(len(dA), len(dB))
                device[out] = (dA[:n] * dB[:n]).astype(np.float32, copy=False)
                dev_set_both(device, dev_sizes, out, device[out])
                host["hC"] = device[out].copy()

            # -------- vexp --------
            elif kname.startswith("exp") or kname == "vexp":
                dA = first_non_none(dev_get_any(device, "dA", "A"), None if len(args)==0 else dev_get_any(device, args[0] if (len(args)>0 and isinstance(args[0], str)) else "dA"))
                out = next((a for a in sargs if a.lower().endswith("c")), "dC")
                if dA is None:
                    n = default_N; ensure_host("hA", n)
                    arr = host["hA"].astype(np.float32, copy=False); dev_set_both(device, dev_sizes, "dA", arr); dA = arr
                n = len(dA)
                device[out] = np.exp(dA[:n]).astype(np.float32, copy=False)
                dev_set_both(device, dev_sizes, out, device[out])
                host["hC"] = device[out].copy()

            # -------- reduce_sum --------
            elif kname.startswith("reduce_sum") or kname == "reduce_sum":
                dA = first_non_none(dev_get_any(device, "dA", "A"), None if len(args)==0 else dev_get_any(device, args[0] if (len(args)>0 and isinstance(args[0], str)) else "dA"))
                out = next((a for a in sargs if "out" in a.lower()), "dOut")
                if dA is None:
                    n = default_N; ensure_host("hA", n)
                    arr = host["hA"].astype(np.float32, copy=False); dev_set_both(device, dev_sizes, "dA", arr); dA = arr
                total = None
                for _ in range(repeats):
                    total = np.float32(np.sum(dA, dtype=np.float64))
                device[out] = np.array([total], dtype=np.float32)
                dev_set_both(device, dev_sizes, out, device[out])
                host["hOut"] = device[out].copy()

            # -------- reduce_max --------
            elif kname.startswith("reduce_max") or kname == "reduce_max":
                dA = first_non_none(dev_get_any(device, "dA", "A"), None if len(args)==0 else dev_get_any(device, args[0] if (len(args)>0 and isinstance(args[0], str)) else "dA"))
                out = next((a for a in sargs if "out" in a.lower()), "dOut")
                if dA is None:
                    n = default_N; ensure_host("hA", n)
                    arr = host["hA"].astype(np.float32, copy=False); dev_set_both(device, dev_sizes, "dA", arr); dA = arr
                m = float(np.max(dA))
                device[out] = np.array([m], dtype=np.float32)
                dev_set_both(device, dev_sizes, out, device[out])
                host["hOut"] = device[out].copy()

            # -------- reduce_argmax --------
            elif kname.startswith("reduce_argmax") or kname == "reduce_argmax":
                dA = first_non_none(dev_get_any(device, "dA", "A"), None if len(args)==0 else dev_get_any(device, args[0] if (len(args)>0 and isinstance(args[0], str)) else "dA"))
                out_val = next((a for a in sargs if "out" in a.lower()), "dOut")
                out_idx = next((a for a in sargs if "idx" in a.lower()), "dIdx")
                if dA is None:
                    n = default_N; ensure_host("hA", n)
                    arr = host["hA"].astype(np.float32, copy=False); dev_set_both(device, dev_sizes, "dA", arr); dA = arr
                idx = int(np.argmax(dA)); val = float(dA[idx])
                device[out_val] = np.array([val], dtype=np.float32)
                device[out_idx] = np.array([idx], dtype=np.int32)
                dev_set_both(device, dev_sizes, out_val, device[out_val])
                for alias in dev_aliases(out_idx):
                    device[alias] = device[out_idx]
                    dev_sizes[alias] = device[out_idx].size
                host["hOut"] = device[out_val].copy(); host["hIdx"] = device[out_idx].copy()

            # -------- softmax (2D, row-wise) --------
            elif kname.startswith("softmax"):
                M, N, _ = parse_MNK()
                dX = next((a for a in sargs if a.lower().endswith("x") or a.lower().endswith("a")), "dX")
                dY = next((a for a in sargs if a.lower().endswith("y") or a.lower().endswith("c")), "dY")
                if dev_get_any(device, dX) is None:
                    X = np.arange(M*N, dtype=np.float32).reshape(M, N)
                    dev_set_both(device, dev_sizes, dX, X.reshape(M*N))
                X = dev_get_any(device, dX).reshape(M, N)
                Y = None
                for _ in range(repeats):
                    mx = X.max(axis=1, keepdims=True)
                    ex = np.exp(X - mx)
                    Y  = ex / ex.sum(axis=1, keepdims=True)
                dev_set_both(device, dev_sizes, dY, Y.astype(np.float32).reshape(M*N))
                host["hY"] = dev_get_any(device, dY).reshape(M*N).copy()

            # -------- layernorm (2D, row-wise) --------
            elif kname.startswith("layernorm"):
                M, N, _ = parse_MNK()
                dX = next((a for a in sargs if a.lower().endswith("x") or a.lower()=="dx"), "dX")
                dY = next((a for a in sargs if a.lower().endswith("y") or a.lower()=="dy"), "dY")
                dG = next((a for a in sargs if "gamma" in a.lower()), "dGamma")
                dB = next((a for a in sargs if "beta"  in a.lower()), "dBeta")
                eps = 1e-5
                if dev_get_any(device, dX) is None:
                    X = np.arange(M*N, dtype=np.float32).reshape(M, N)
                    dev_set_both(device, dev_sizes, dX, X.reshape(M*N))
                if dev_get_any(device, dG) is None:
                    G = np.ones(N, dtype=np.float32)
                    dev_set_both(device, dev_sizes, dG, G.reshape(N))
                if dev_get_any(device, dB) is None:
                    B = np.zeros(N, dtype=np.float32)
                    dev_set_both(device, dev_sizes, dB, B.reshape(N))
                X = dev_get_any(device, dX).reshape(M, N)
                G = dev_get_any(device, dG).reshape(N)
                B = dev_get_any(device, dB).reshape(N)
                Y = None
                for _ in range(repeats):
                    mu  = X.mean(axis=1, keepdims=True)
                    var = X.var(axis=1, keepdims=True)
                    Y = ((X - mu) / np.sqrt(var + eps)) * G + B
                dev_set_both(device, dev_sizes, dY, Y.astype(np.float32).reshape(M*N))
                host["hY"] = dev_get_any(device, dY).reshape(M*N).copy()

    if summary:
        for k, v in host.items():
            print(f"[DTYPE] {k}: {v.dtype.name}, shape={v.shape}")
        h = {k: list(np.asarray(v)[:8]) for k, v in host.items() if hasattr(v, "__array_interface__")}
        print("[SUMMARY] host(head):", h)
        d = {k: list(np.asarray(v).ravel()[:8]) for k, v in device.items() if hasattr(v, "__array_interface__")}
        print("[SUMMARY] device(head):", d)
        return {"ok": True}

    # --- PRIORITIZE hY before checking hC/dC (mini-patch B) ---
    hy = host.get("hY")
    if hy is not None and dump is None:
        return np.asarray(hy, dtype=np.float32)

    # --- inproc contract: return array when possible (no dump requested) ---
    if dump is None:
        hc = host.get("hC")
        if hc is None:
            hc = device.get("dC")
        if hc is not None:
            arr = np.asarray(hc, dtype=np.float32)
            try:
                sh = ((ax.get("buffers", {}) or {}).get("hC") or {}).get("shape")
                if isinstance(sh, (list, tuple)) and len(sh) == 2:
                    arr = arr.reshape(int(sh[0]), int(sh[1]))
            except Exception:
                pass
            return arr

    # --- GEMM fallback: calcule C = A @ B à partir de l'AXIR (sans ops) ---
    def _shape_of(bufname):
        ent = (ax.get("buffers", {}) or {}).get(bufname) or {}
        sh  = ent.get("shape")
        if isinstance(sh, (list, tuple)) and len(sh) == 2:
            return int(sh[0]), int(sh[1])
        return None

    def _get_np(bufname, shape):
        ent  = (ax.get("buffers", {}) or {}).get(bufname) or {}
        data = ent.get("data", ax.get(bufname))
        if data is None:
            return np.arange(int(np.prod(shape)), dtype=np.float32).reshape(shape)
        return np.asarray(data, dtype=np.float32).reshape(shape)

    M=K=N=None
    shA = _shape_of("hA") or _shape_of("A")
    shB = _shape_of("hB") or _shape_of("B")
    if shA: M, K = shA
    if shB:
        K = K if K is not None else int(shB[0])
        N = int(shB[1])
    if not all(v is not None for v in (M, N, K)):
        try:
            M = M or int(ax.get("types",{}).get("scalars",{}).get("M",{}).get("value"))
            N = N or int(ax.get("types",{}).get("scalars",{}).get("N",{}).get("value"))
            K = K or int(ax.get("types",{}).get("scalars",{}).get("K",{}).get("value"))
        except Exception:
            pass
    if all(v is not None for v in (M, N, K)):
        A = _get_np("hA", (M, K)) if _shape_of("hA") or ax.get("hA") is not None else _get_np("A", (M, K))
        B = _get_np("hB", (K, N)) if _shape_of("hB") or ax.get("hB") is not None else _get_np("B", (K, N))

        gemm_op = _find_gemm_op(ax)
        if gemm_op is not None:
            alpha = _resolve_f(ax, gemm_op.get("alpha", _read_fscalar(ax, "alpha", 1.0)), 1.0)
            beta  = _resolve_f(ax, gemm_op.get("beta",  _read_fscalar(ax, "beta",  0.0)), 0.0)
            Cprev = None
            entC  = (ax.get("buffers", {}) or {}).get("hC") or {}
            hasCdata = ("data" in entC) or ("hC" in ax)
            if hasCdata:
                Cprev = _get_np("hC", (M, N))
            if Cprev is None:
                Cprev = np.zeros((M, N), dtype=np.float32)
            Ccore = (A @ B).astype(np.float32, copy=False)
            C_out = (alpha * Ccore + beta * Cprev).astype(np.float32, copy=False)
        else:
            C_out = A @ B

        host["hC"] = np.asarray(C_out, dtype=np.float32)

    # --- Normalisation du retour pour les tests ---
    if dump:
        v = host.get(dump)
        if isinstance(v, np.ndarray):
            return {"dump": v}
        d = dev_get_any(device, dump)
        if d is not None:
            return {"dump": np.asarray(d, dtype=np.float32)}
        # dernier recours tolerant: si rien trouvé mais on a calculé C_out
        if dump.lower() in ("hc", "c") and "hC" in host:
            return {"dump": host["hC"]}
        return {"ok": True}

    # si pas de dump demandé et rien à renvoyer explicitement
    return {"ok": True}

# ============
# CLI entry
# ============
def main():
    ap = argparse.ArgumentParser(
        description="AXIR -> CPU (NumPy) backend — kernels de base + softmax/layernorm"
    )
    ap.add_argument("axir", help="Path to the AXIR JSON")
    ap.add_argument("--summary", action="store_true", help="Print a brief head of host/device buffers")
    ap.add_argument("--dump", help="Name of the buffer to fetch (e.g., hC, dC, hOut, dOut, hY)")
    ap.add_argument("--out", help="Output .npy path for --dump")
    ap.add_argument("--repeat-kernel", type=int, default=1,
                    help="Repeat each KernelLaunch R times before any host copy (default: 1).")
    args = ap.parse_args()

    ax = json.loads(pathlib.Path(args.axir).read_text(encoding="utf-8-sig"))

    # IMPORTANT: ne pas passer dump=... à run(), pour laisser le resolver tolérant
    res = run(ax, summary=args.summary, dump=None, repeats=args.repeat_kernel)

    import numpy as np
    if args.dump:
        state = {}
        if isinstance(res, dict) and "dump" in res and hasattr(res["dump"], "__array_interface__"):
            val = np.asarray(res["dump"])
            state["hC"] = val; state["C"] = val
            state["hY"] = val
            state["dC"] = val.reshape(-1)
            state["hOut"] = val; state["out"] = val; state["result"] = val
        elif hasattr(res, "__array_interface__"):
            val = np.asarray(res)
            state["hC"] = val; state["C"] = val
            state["hY"] = val
            state["dC"] = val.reshape(-1)
            state["hOut"] = val; state["out"] = val; state["result"] = val

        dump_name = args.dump.split(",")[0].strip()
        out_path = args.out or f"verify_cpu_{dump_name}.npy"

        name = _resolve_dump_name(args.dump, state, ax)
        if not name:
            print("[cpu] available buffers:", ", ".join(sorted(state.keys())) )
            raise SystemExit(f"[cpu] dump target not found: {args.dump}")

        arr = np.asarray(state[name])
        np.save(out_path, arr.reshape(arr.shape))
        print(f"[entry_cpu] saved {out_path} shape={arr.shape}")
        return

if __name__ == "__main__":
    main()
