# Contributing to EmberQuant

Thank you for your interest in contributing to EmberQuant! This document provides guidelines and instructions for contributing.

## Code of Conduct

By participating in this project, you agree to maintain a respectful and inclusive environment for all contributors.

## Getting Started

### Development Setup

1. **Clone the repository**
   ```bash
   git clone https://github.com/emberquant/emberquant.git
   cd emberquant
   ```

2. **Create a virtual environment**
   ```bash
   python -m venv venv
   source venv/bin/activate  # On Windows: venv\Scripts\activate
   ```

3. **Install in development mode**
   ```bash
   pip install -e ".[dev]"
   ```

4. **Install pre-commit hooks**
   ```bash
   pre-commit install
   ```

### Running Tests

```bash
# Run all tests
pytest

# Run with coverage
pytest --cov=emberquant --cov-report=html

# Run specific test categories
pytest -m unit
pytest -m integration

# Run a specific test file
pytest tests/unit/test_emberframe.py
```

### Code Quality

We use several tools to maintain code quality:

```bash
# Type checking
mypy src/emberquant

# Linting
ruff check src/emberquant

# Code formatting
black src/emberquant
isort src/emberquant

# Run all checks
pre-commit run --all-files
```

## How to Contribute

### Reporting Bugs

1. Check if the bug has already been reported in [Issues](https://github.com/emberquant/emberquant/issues)
2. If not, create a new issue with:
   - Clear title and description
   - Steps to reproduce
   - Expected vs actual behavior
   - Python version and environment details
   - Code samples if applicable

### Suggesting Features

1. Check [Issues](https://github.com/emberquant/emberquant/issues) and [Discussions](https://github.com/emberquant/emberquant/discussions)
2. Open a new discussion or issue describing:
   - The problem you're trying to solve
   - Your proposed solution
   - Alternative approaches considered
   - How it fits into EmberQuant's vision

### Pull Requests

1. **Fork the repository** and create a new branch
   ```bash
   git checkout -b feature/your-feature-name
   ```

2. **Make your changes**
   - Follow the existing code style
   - Add tests for new functionality
   - Update documentation as needed
   - Ensure all tests pass

3. **Commit your changes**
   ```bash
   git add .
   git commit -m "Add feature: description of your changes"
   ```

4. **Push to your fork**
   ```bash
   git push origin feature/your-feature-name
   ```

5. **Create a Pull Request**
   - Provide a clear title and description
   - Reference any related issues
   - Ensure CI checks pass

## Development Guidelines

### Code Style

- Follow [PEP 8](https://pep8.org/)
- Use type hints for all functions
- Maximum line length: 100 characters
- Use descriptive variable and function names

### Type Hints

All code must include type hints:

```python
def process_data(df: pd.DataFrame, source: str) -> EmberFrame:
    """Process data and return an EmberFrame."""
    ...
```

### Documentation

- Add docstrings to all public classes and functions
- Use Google-style docstrings
- Update README.md for user-facing changes
- Add examples for new features

Example docstring:

```python
def my_function(param1: str, param2: int) -> bool:
    """
    Brief description of the function.

    Longer description if needed.

    Args:
        param1: Description of param1
        param2: Description of param2

    Returns:
        Description of return value

    Raises:
        ValueError: When something goes wrong

    Examples:
        >>> my_function("hello", 42)
        True
    """
    ...
```

### Testing

- Write tests for all new functionality
- Maintain or improve code coverage
- Use descriptive test names
- Organize tests into unit and integration categories

```python
class TestEmberFrame:
    """Test suite for EmberFrame class."""

    def test_from_dataframe_creates_valid_frame(self) -> None:
        """Test that from_dataframe creates a valid EmberFrame."""
        ...

    def test_column_type_inference_detects_dates(self) -> None:
        """Test that date columns are correctly identified."""
        ...
```

### Commit Messages

- Use clear, descriptive commit messages
- Start with a verb in present tense
- Reference issues when applicable

Good examples:
```
Add Benford's law analysis to Auditor agent
Fix column type inference for date columns
Update README with new adapter examples
Closes #123: Implement Xero API integration
```

### Branch Naming

- `feature/` - New features
- `fix/` - Bug fixes
- `docs/` - Documentation updates
- `refactor/` - Code refactoring
- `test/` - Test additions or modifications

## Project Structure

```
emberquant/
├── src/emberquant/        # Main package
│   ├── core/              # Core components
│   ├── agents/            # Agent implementations
│   ├── adapters/          # Data source adapters
│   └── utils/             # Utility functions
├── tests/                 # Test suite
│   ├── unit/              # Unit tests
│   └── integration/       # Integration tests
├── docs/                  # Documentation
└── examples/              # Example scripts
```

## Adding New Components

### Adding a New Agent

1. Create a new file in `src/emberquant/agents/`
2. Inherit from `BaseAgent`
3. Implement `name` property and `execute()` method
4. Add to `__init__.py`
5. Write comprehensive tests
6. Update documentation

### Adding a New Adapter

1. Create a new file in `src/emberquant/adapters/`
2. Inherit from `BaseAdapter`
3. Implement all abstract methods
4. Update `connect()` function to support the new adapter
5. Write tests with mock data
6. Document usage in README

## Release Process

1. Update version in `pyproject.toml`
2. Update `CHANGELOG.md`
3. Create a new release on GitHub
4. CI/CD will automatically publish to PyPI

## Questions?

- Open a [Discussion](https://github.com/emberquant/emberquant/discussions)
- Join our community chat (link TBD)
- Email: hello@emberquant.io

Thank you for contributing to EmberQuant!
