# EmberQuant

> **The Agentic Framework for AccountingTech** - "Claude Code for Accounting"

EmberQuant is an infrastructure layer for enterprises to embed autonomous financial reasoning into their stack. It shifts accounting from syntactic to semantic processing, transforming traditional accounting workflows from calculator-style operations to intelligent, context-aware analysis.

## Vision

Traditional accounting software is a calculator. **EmberQuant is a colleague.**

Instead of manually entering "Debit Cash 100, Credit Revenue 100," EmberQuant understands:
*"Recognize revenue from SaaS contract, confirm ASC 606 alignment, flag LTV decline."*

## Features

- **EmberFrame**: Semantic DataFrame for financial data with automatic type inference
- **EmberGraph**: DAG-based orchestration system for multi-agent financial workflows
- **Intelligent Agents**:
  - **Clerk**: Ingestion and normalization of messy Excel/PDF/API data
  - **Auditor**: Benford's law analysis, variance detection, anomaly flagging
  - More agents coming: Modeler, Controller
- **Universal Adapters**: Connect to QuickBooks, Xero, CSV, Excel, and more
- **Provider-Agnostic AI**: Works with Anthropic Claude, OpenAI, or any LLM via LiteLLM
- **Type-Safe**: Full type hints for better IDE support and fewer bugs

## Installation

```bash
pip install emberquant
```

### Development Installation

```bash
git clone https://github.com/emberquant/emberquant.git
cd emberquant
pip install -e ".[dev]"
```

## Quick Start

### Basic Audit Workflow

```python
import emberquant as eq

# Connect to your data source
ledger = eq.connect("./data/transactions.csv")

# Create an execution plan
plan = eq.plan("Audit Q3 marketing spend")

# Execute the plan
from emberquant.core.planning import execute_plan
results = execute_plan(
    plan,
    inputs={"data": ledger.fetch_ledger(), "source": "transactions.csv", "source_type": "csv"}
)

# Access the audit report
audit_report = results["results"][list(results["results"].keys())[-1]]["audit_report"]
print(audit_report.summary)
for finding in audit_report.findings:
    print(f"  [{finding.severity}] {finding.description}")
```

### Working with EmberFrame

```python
import emberquant as eq
import pandas as pd

# Create an EmberFrame from a DataFrame
df = pd.read_csv("ledger.csv")
frame = eq.EmberFrame.from_dataframe(
    df=df,
    source="ledger.csv",
    source_type="csv",
    infer_types=True
)

# EmberFrame automatically infers semantic column types
print(f"Date column: {frame.get_column_by_type(eq.ColumnType.DATE)}")
print(f"Amount columns: {frame.get_all_columns_by_type(eq.ColumnType.AMOUNT)}")

# Filter by date range
from datetime import datetime
q1_data = frame.filter_by_date_range(
    start_date=datetime(2024, 1, 1),
    end_date=datetime(2024, 3, 31)
)

# Aggregate by category
summary = frame.aggregate_by_category()
print(summary)
```

### Using EmberGraph for Custom Workflows

```python
import emberquant as eq

# Create a custom workflow graph
graph = eq.EmberGraph()

# Add agents to the graph
clerk = eq.ClerkAgent()
auditor = eq.AuditorAgent()

# Define the workflow
clerk_id = graph.add_node(
    agent=clerk,
    inputs={"data": df, "source": "data.csv", "source_type": "csv"}
)

auditor_id = graph.add_node(
    agent=auditor,
    dependencies=[clerk_id]  # Auditor depends on Clerk
)

# Execute the graph
results = graph.execute(verbose=True)

# View execution summary
summary = graph.get_execution_summary()
print(f"Completed {summary['completed']}/{summary['total_nodes']} tasks")
print(f"Total duration: {summary['total_duration_seconds']:.2f}s")
```

### Connecting to Accounting Systems

#### QuickBooks

```python
import emberquant as eq

# Connect to QuickBooks (mock implementation in Phase 1)
ledger = eq.connect(
    {
        "access_token": "your_oauth_token",
        "realm_id": "your_company_id"
    },
    adapter_type="quickbooks"
)

data = ledger.fetch_ledger(start_date="2024-01-01", end_date="2024-03-31")
```

#### Xero

```python
import emberquant as eq

ledger = eq.connect(
    {
        "access_token": "your_oauth_token",
        "tenant_id": "your_tenant_id"
    },
    adapter_type="xero"
)

data = ledger.fetch_transactions(start_date="2024-01-01")
```

#### CSV/Excel

```python
import emberquant as eq

# Auto-detects file format
ledger = eq.connect("./financial_data.xlsx")
data = ledger.fetch_ledger()
```

## Architecture

EmberQuant is built around the **EmberGraph** - a directed acyclic graph (DAG) of financial agents:

```
┌─────────────────────────────────────────────────────┐
│                    EmberGraph                       │
│                                                     │
│  ┌──────────┐    ┌──────────┐    ┌──────────┐    │
│  │  Clerk   │───▶│ Auditor  │───▶│Controller│    │
│  │ (Ingest) │    │(Analyze) │    │(Deliver) │    │
│  └──────────┘    └──────────┘    └──────────┘    │
│                        │                           │
│                        ▼                           │
│                  ┌──────────┐                      │
│                  │ Modeler  │                      │
│                  │(Forecast)│                      │
│                  └──────────┘                      │
└─────────────────────────────────────────────────────┘
```

### Core Components

1. **EmberFrame**: Semantic DataFrame that understands financial concepts
2. **Agents**: Specialized AI agents for different accounting tasks
3. **Adapters**: Connectors for external accounting systems
4. **EmberGraph**: Orchestration engine for multi-step workflows

## Command-Line Interface

EmberQuant includes a CLI for common tasks:

```bash
# Display version
emberquant version

# Run an audit on a file
emberquant audit transactions.csv --verbose

# Save audit report
emberquant audit transactions.csv --output report.json

# Display file information
emberquant info ledger.xlsx
```

## Development

### Running Tests

```bash
# Run all tests
pytest

# Run with coverage
pytest --cov=emberquant --cov-report=html

# Run only unit tests
pytest -m unit

# Run only integration tests
pytest -m integration
```

### Type Checking

```bash
mypy src/emberquant
```

### Linting and Formatting

```bash
# Check code style
ruff check src/emberquant

# Format code
black src/emberquant

# Sort imports
isort src/emberquant
```

## Roadmap

### Phase 1 (Current)
- ✅ EmberFrame with semantic type inference
- ✅ Clerk agent for data ingestion
- ✅ Auditor agent with Benford's law and variance analysis
- ✅ Basic adapters (CSV, Excel, mock QuickBooks/Xero)
- ✅ EmberGraph DAG orchestration

### Phase 2 (Next)
- 🔲 Full QuickBooks and Xero API integration
- 🔲 PDF extraction and parsing
- 🔲 LLM-powered anomaly detection
- 🔲 Natural language plan generation

### Phase 3 (Future)
- 🔲 Modeler agent for forecasting
- 🔲 Controller agent for report generation
- 🔲 React component outputs
- 🔲 Real-time monitoring and alerts

## Contributing

We welcome contributions! Please see [CONTRIBUTING.md](CONTRIBUTING.md) for guidelines.

## License

EmberQuant is licensed under the Apache License 2.0. See [LICENSE](LICENSE) for details.

## Support

- **Documentation**: [https://emberquant.readthedocs.io](https://emberquant.readthedocs.io)
- **Issues**: [GitHub Issues](https://github.com/emberquant/emberquant/issues)
- **Discussions**: [GitHub Discussions](https://github.com/emberquant/emberquant/discussions)

## Citation

If you use EmberQuant in your research or business, please cite:

```bibtex
@software{emberquant2024,
  title = {EmberQuant: The Agentic Framework for AccountingTech},
  author = {EmberQuant Team},
  year = {2024},
  url = {https://github.com/emberquant/emberquant}
}
```

---

**EmberQuant** - Transforming accounting from calculation to cognition.
