"""
Base adapter interface for connecting to external accounting systems.
"""

from __future__ import annotations

from abc import ABC, abstractmethod
from typing import Any, Dict, Optional

import pandas as pd
from pydantic import BaseModel


class ConnectionConfig(BaseModel):
    """Base configuration for adapters."""

    adapter_type: str
    credentials: Dict[str, Any] = {}
    options: Dict[str, Any] = {}


class BaseAdapter(ABC):
    """
    Abstract base class for all data source adapters.

    Adapters provide a unified interface for connecting to different
    accounting systems (QuickBooks, Xero, CSV files, etc.).
    """

    def __init__(self, config: ConnectionConfig) -> None:
        """
        Initialize the adapter.

        Args:
            config: Configuration for the connection
        """
        self.config = config
        self._connected = False

    @abstractmethod
    def connect(self) -> None:
        """Establish connection to the data source."""
        ...

    @abstractmethod
    def disconnect(self) -> None:
        """Close connection to the data source."""
        ...

    @abstractmethod
    def fetch_ledger(
        self, start_date: Optional[str] = None, end_date: Optional[str] = None
    ) -> pd.DataFrame:
        """
        Fetch general ledger data.

        Args:
            start_date: Optional start date (ISO format)
            end_date: Optional end date (ISO format)

        Returns:
            DataFrame with ledger data
        """
        ...

    @abstractmethod
    def fetch_transactions(
        self, start_date: Optional[str] = None, end_date: Optional[str] = None
    ) -> pd.DataFrame:
        """
        Fetch transaction data.

        Args:
            start_date: Optional start date (ISO format)
            end_date: Optional end date (ISO format)

        Returns:
            DataFrame with transaction data
        """
        ...

    @property
    def is_connected(self) -> bool:
        """Check if adapter is connected."""
        return self._connected

    def __enter__(self) -> BaseAdapter:
        """Context manager entry."""
        self.connect()
        return self

    def __exit__(self, exc_type: Any, exc_val: Any, exc_tb: Any) -> None:
        """Context manager exit."""
        self.disconnect()
