"""
CSV/Excel file adapter for local file-based accounting data.
"""

from __future__ import annotations

from pathlib import Path
from typing import Optional, Dict, List, Any

import pandas as pd

from emberquant.adapters.base import BaseAdapter, ConnectionConfig


class CSVAdapter(BaseAdapter):
    """
    Adapter for CSV and Excel files.

    Supports reading local accounting data files, including multi-sheet Excel files.
    """

    def __init__(self, config: ConnectionConfig) -> None:
        """
        Initialize CSV adapter.

        Args:
            config: Configuration with file_path in options
        """
        super().__init__(config)
        self.file_path = Path(config.options.get("file_path", ""))
        self.sheet_name = config.options.get("sheet_name", 0)  # Default to first sheet
        self.all_sheets = config.options.get("all_sheets", False)  # Read all sheets

        if not self.file_path:
            raise ValueError("file_path must be specified in config.options")

    def connect(self) -> None:
        """Validate that the file exists."""
        if not self.file_path.exists():
            raise FileNotFoundError(f"File not found: {self.file_path}")
        self._connected = True

    def disconnect(self) -> None:
        """No-op for file-based adapter."""
        self._connected = False

    def fetch_ledger(
        self, start_date: Optional[str] = None, end_date: Optional[str] = None
    ) -> pd.DataFrame:
        """
        Read ledger data from file.

        Args:
            start_date: Optional start date filter
            end_date: Optional end date filter

        Returns:
            DataFrame with ledger data
        """
        df = self._read_file()

        # Apply date filtering if date column exists
        if start_date or end_date:
            date_cols = [col for col in df.columns if "date" in col.lower()]
            if date_cols:
                date_col = date_cols[0]
                df[date_col] = pd.to_datetime(df[date_col])

                if start_date:
                    df = df[df[date_col] >= start_date]
                if end_date:
                    df = df[df[date_col] <= end_date]

        return df

    def fetch_transactions(
        self, start_date: Optional[str] = None, end_date: Optional[str] = None
    ) -> pd.DataFrame:
        """
        Read transaction data from file.

        For CSV adapter, this is the same as fetch_ledger.

        Args:
            start_date: Optional start date filter
            end_date: Optional end date filter

        Returns:
            DataFrame with transaction data
        """
        return self.fetch_ledger(start_date, end_date)

    def _read_file(self) -> pd.DataFrame:
        """Read the file based on extension."""
        suffix = self.file_path.suffix.lower()

        if suffix == ".csv":
            return pd.read_csv(self.file_path)
        elif suffix in [".xlsx", ".xls"]:
            if self.all_sheets:
                # Read all sheets and concatenate
                excel_file = pd.ExcelFile(self.file_path)
                dfs = []
                for sheet in excel_file.sheet_names:
                    df = pd.read_excel(self.file_path, sheet_name=sheet)
                    df['_sheet_name'] = sheet  # Add sheet identifier
                    dfs.append(df)
                return pd.concat(dfs, ignore_index=True)
            else:
                return pd.read_excel(self.file_path, sheet_name=self.sheet_name)
        else:
            raise ValueError(f"Unsupported file format: {suffix}")

    def get_sheet_names(self) -> List[str]:
        """
        Get list of sheet names for Excel files.

        Returns:
            List of sheet names, empty list for CSV files
        """
        suffix = self.file_path.suffix.lower()
        if suffix in [".xlsx", ".xls"]:
            excel_file = pd.ExcelFile(self.file_path)
            return excel_file.sheet_names
        return []
