"""
QuickBooks API adapter (mock implementation for demonstration).

In production, this would use the QuickBooks API via OAuth.
"""

from __future__ import annotations

from datetime import datetime
from typing import Optional

import pandas as pd

from emberquant.adapters.base import BaseAdapter, ConnectionConfig


class QuickBooksAdapter(BaseAdapter):
    """
    Adapter for QuickBooks Online API.

    This is a mock implementation. In production, it would use the
    QuickBooks OAuth flow and API client.
    """

    def __init__(self, config: ConnectionConfig) -> None:
        """
        Initialize QuickBooks adapter.

        Args:
            config: Configuration with OAuth token or credentials
        """
        super().__init__(config)
        self.access_token = config.credentials.get("access_token")
        self.refresh_token = config.credentials.get("refresh_token")
        self.realm_id = config.credentials.get("realm_id")  # Company ID

    def connect(self) -> None:
        """
        Establish connection to QuickBooks API.

        In production, this would validate the OAuth token and refresh if needed.
        """
        if not self.access_token:
            raise ValueError("QuickBooks access_token required in credentials")

        # Mock connection
        print(f"[QuickBooks Mock] Connecting to realm {self.realm_id}...")
        self._connected = True

    def disconnect(self) -> None:
        """Close the connection."""
        self._connected = False

    def fetch_ledger(
        self, start_date: Optional[str] = None, end_date: Optional[str] = None
    ) -> pd.DataFrame:
        """
        Fetch general ledger from QuickBooks.

        Args:
            start_date: Start date (ISO format)
            end_date: End date (ISO format)

        Returns:
            DataFrame with ledger data
        """
        if not self._connected:
            raise RuntimeError("Not connected to QuickBooks")

        # Mock data - in production, this would call the QuickBooks API
        print(f"[QuickBooks Mock] Fetching ledger from {start_date} to {end_date}...")

        mock_data = {
            "transaction_id": ["TXN001", "TXN002", "TXN003", "TXN004", "TXN005"],
            "date": [
                "2024-01-15",
                "2024-01-20",
                "2024-02-01",
                "2024-02-15",
                "2024-03-01",
            ],
            "account": [
                "Revenue:SaaS",
                "Revenue:SaaS",
                "Expenses:Marketing",
                "Revenue:SaaS",
                "Expenses:Salaries",
            ],
            "debit": [0, 0, 5000, 0, 15000],
            "credit": [10000, 8000, 0, 12000, 0],
            "description": [
                "Monthly subscription - Acme Corp",
                "Monthly subscription - TechCo",
                "Facebook Ads Campaign",
                "Annual subscription - StartupXYZ",
                "Monthly payroll",
            ],
            "category": ["Revenue", "Revenue", "Marketing", "Revenue", "Payroll"],
        }

        df = pd.DataFrame(mock_data)
        df["date"] = pd.to_datetime(df["date"])

        # Apply date filters
        if start_date:
            df = df[df["date"] >= start_date]
        if end_date:
            df = df[df["date"] <= end_date]

        return df

    def fetch_transactions(
        self, start_date: Optional[str] = None, end_date: Optional[str] = None
    ) -> pd.DataFrame:
        """
        Fetch transactions from QuickBooks.

        Args:
            start_date: Start date (ISO format)
            end_date: End date (ISO format)

        Returns:
            DataFrame with transaction data
        """
        return self.fetch_ledger(start_date, end_date)
