"""
Xero API adapter (mock implementation for demonstration).

In production, this would use the Xero API via OAuth.
"""

from __future__ import annotations

from typing import Optional

import pandas as pd

from emberquant.adapters.base import BaseAdapter, ConnectionConfig


class XeroAdapter(BaseAdapter):
    """
    Adapter for Xero Accounting API.

    This is a mock implementation. In production, it would use the
    Xero OAuth flow and API client.
    """

    def __init__(self, config: ConnectionConfig) -> None:
        """
        Initialize Xero adapter.

        Args:
            config: Configuration with OAuth token or credentials
        """
        super().__init__(config)
        self.access_token = config.credentials.get("access_token")
        self.tenant_id = config.credentials.get("tenant_id")

    def connect(self) -> None:
        """
        Establish connection to Xero API.

        In production, this would validate the OAuth token and refresh if needed.
        """
        if not self.access_token:
            raise ValueError("Xero access_token required in credentials")

        # Mock connection
        print(f"[Xero Mock] Connecting to tenant {self.tenant_id}...")
        self._connected = True

    def disconnect(self) -> None:
        """Close the connection."""
        self._connected = False

    def fetch_ledger(
        self, start_date: Optional[str] = None, end_date: Optional[str] = None
    ) -> pd.DataFrame:
        """
        Fetch general ledger from Xero.

        Args:
            start_date: Start date (ISO format)
            end_date: End date (ISO format)

        Returns:
            DataFrame with ledger data
        """
        if not self._connected:
            raise RuntimeError("Not connected to Xero")

        # Mock data - in production, this would call the Xero API
        print(f"[Xero Mock] Fetching ledger from {start_date} to {end_date}...")

        mock_data = {
            "transaction_id": ["XRO001", "XRO002", "XRO003", "XRO004"],
            "date": ["2024-01-10", "2024-01-25", "2024-02-05", "2024-02-20"],
            "account": [
                "Sales:Products",
                "Sales:Services",
                "Expenses:Office",
                "Sales:Products",
            ],
            "amount": [15000, 8500, -2000, 20000],
            "description": [
                "Product sale - Enterprise package",
                "Consulting services - Q1",
                "Office supplies purchase",
                "Product sale - Premium tier",
            ],
            "contact": ["BigCorp Ltd", "Consulting Inc", "Office Depot", "SmallBiz Co"],
            "category": ["Sales", "Services", "Expenses", "Sales"],
        }

        df = pd.DataFrame(mock_data)
        df["date"] = pd.to_datetime(df["date"])

        # Apply date filters
        if start_date:
            df = df[df["date"] >= start_date]
        if end_date:
            df = df[df["date"] <= end_date]

        return df

    def fetch_transactions(
        self, start_date: Optional[str] = None, end_date: Optional[str] = None
    ) -> pd.DataFrame:
        """
        Fetch transactions from Xero.

        Args:
            start_date: Start date (ISO format)
            end_date: End date (ISO format)

        Returns:
            DataFrame with transaction data
        """
        return self.fetch_ledger(start_date, end_date)
