"""
Base agent class for all EmberQuant agents.
"""

from __future__ import annotations

from abc import ABC, abstractmethod
from typing import Any, Dict, Optional

from pydantic import BaseModel


class AgentConfig(BaseModel):
    """Base configuration for agents."""

    llm_provider: Optional[str] = None
    llm_model: Optional[str] = None
    temperature: float = 0.1
    max_tokens: int = 4000
    verbose: bool = False


class BaseAgent(ABC):
    """
    Abstract base class for all EmberQuant agents.

    All agents must implement the execute() method and provide a name.
    """

    def __init__(self, config: Optional[AgentConfig] = None) -> None:
        """
        Initialize the agent.

        Args:
            config: Optional configuration for the agent
        """
        self.config = config or AgentConfig()

    @property
    @abstractmethod
    def name(self) -> str:
        """Return the agent's name."""
        ...

    @abstractmethod
    def execute(self, inputs: Dict[str, Any]) -> Dict[str, Any]:
        """
        Execute the agent's main task.

        Args:
            inputs: Dictionary of input data and parameters

        Returns:
            Dictionary of output data and results
        """
        ...

    def _log(self, message: str) -> None:
        """Log a message if verbose mode is enabled."""
        if self.config.verbose:
            print(f"[{self.name}] {message}")
