"""
Controller agent for generating financial reports and deliverables.
"""

from __future__ import annotations

from dataclasses import dataclass
from datetime import datetime
from pathlib import Path
from typing import Any, Dict, List, Optional

import json
import pandas as pd
from pydantic import BaseModel, Field

from emberquant.agents.base import BaseAgent
from emberquant.core.emberframe import EmberFrame


class NarrativeReport(BaseModel):
    """Narrative financial report."""

    title: str = Field(description="Report title")
    summary: str = Field(description="Executive summary")
    sections: List[Dict[str, Any]] = Field(description="Report sections")
    generated_at: str = Field(description="Generation timestamp")
    metadata: Dict[str, Any] = Field(description="Report metadata")


@dataclass
class ControllerConfig:
    """Configuration for Controller agent."""

    include_charts: bool = False  # Future: Generate charts
    output_format: str = "json"  # json, markdown, html
    narrative_style: str = "professional"  # professional, technical, executive
    verbose: bool = False


class ControllerAgent(BaseAgent):
    """
    Agent for generating financial reports and deliverables.

    Capabilities:
    - Narrative report generation
    - JSON data exports
    - Excel workbook creation
    - Markdown reports
    - Financial summaries
    """

    def __init__(self, config: Optional[ControllerConfig] = None) -> None:
        """
        Initialize Controller agent.

        Args:
            config: Optional configuration for report generation
        """
        super().__init__()
        self.config = config or ControllerConfig()

    @property
    def name(self) -> str:
        """Return the agent name."""
        return "Controller"

    def execute(self, inputs: Dict[str, Any]) -> Dict[str, Any]:
        """
        Execute report generation tasks.

        Expected inputs:
            - emberframe: EmberFrame to report on
            - audit_report: Optional audit report from Auditor
            - modeling_results: Optional modeling results from Modeler
            - output_type: Type of output (narrative, json, excel, markdown)
            - output_path: Optional path to save output

        Returns:
            Dictionary with generated reports
        """
        self._log("Starting report generation...")

        emberframe = inputs.get("emberframe")
        if emberframe is None:
            raise ValueError("No EmberFrame provided to Controller agent")

        if not isinstance(emberframe, EmberFrame):
            raise ValueError(f"Expected EmberFrame, got {type(emberframe)}")

        output_type = inputs.get("output_type", "all")
        output_path = inputs.get("output_path", None)

        results = {}

        if output_type in ["all", "narrative"]:
            narrative = self._generate_narrative_report(inputs)
            results["narrative_report"] = narrative

            if output_path:
                self._save_narrative(narrative, output_path)

        if output_type in ["all", "json"]:
            json_export = self._generate_json_export(inputs)
            results["json_export"] = json_export

            if output_path:
                json_path = Path(output_path).with_suffix(".json")
                with open(json_path, "w") as f:
                    json.dump(json_export, f, indent=2, default=str)
                self._log(f"Saved JSON export to {json_path}")

        if output_type in ["all", "excel"]:
            excel_data = self._generate_excel_export(inputs)
            results["excel_data"] = excel_data

            if output_path:
                excel_path = Path(output_path).with_suffix(".xlsx")
                self._save_excel(excel_data, excel_path, emberframe)

        if output_type in ["all", "markdown"]:
            markdown = self._generate_markdown_report(inputs)
            results["markdown_report"] = markdown

            if output_path:
                md_path = Path(output_path).with_suffix(".md")
                with open(md_path, "w", encoding="utf-8") as f:
                    f.write(markdown)
                self._log(f"Saved Markdown report to {md_path}")

        self._log("Report generation complete")

        return {
            "status": "success",
            "reports": results,
            "emberframe": emberframe,
        }

    def _generate_narrative_report(self, inputs: Dict[str, Any]) -> NarrativeReport:
        """
        Generate a narrative financial report.

        Args:
            inputs: Input data including emberframe and analysis results

        Returns:
            Narrative report
        """
        emberframe = inputs["emberframe"]
        audit_report = inputs.get("audit_report")
        modeling_results = inputs.get("modeling_results")

        df = emberframe.data

        # Generate summary
        total_transactions = len(df)
        date_col = next((col for col in df.columns if "date" in col.lower()), None)

        if date_col:
            df[date_col] = pd.to_datetime(df[date_col], errors="coerce")
            min_date = df[date_col].min()
            max_date = df[date_col].max()
            period = f"{min_date.strftime('%Y-%m-%d')} to {max_date.strftime('%Y-%m-%d')}"
        else:
            period = "unknown period"

        # Calculate financial totals
        amount_totals = {}
        for col in df.columns:
            if any(x in col.lower() for x in ["amount", "debit", "credit"]):
                total = df[col].sum()
                if pd.notna(total):
                    amount_totals[col] = float(total)

        summary = (
            f"Financial analysis for {period}. "
            f"Analyzed {total_transactions} transactions. "
        )

        if amount_totals:
            summary += f"Total activity: ${sum(amount_totals.values()):,.2f}."

        # Build sections
        sections = []

        # Overview section
        sections.append({
            "title": "Overview",
            "content": summary,
            "data": {
                "total_transactions": total_transactions,
                "period": period,
                "amount_totals": amount_totals,
            },
        })

        # Audit section
        if audit_report:
            audit_summary = (
                f"Audit analysis identified {len(audit_report.get('findings', []))} findings. "
            )
            sections.append({
                "title": "Audit Results",
                "content": audit_summary,
                "data": audit_report,
            })

        # Modeling section
        if modeling_results:
            modeling_summary = "Financial modeling completed. "
            if "forecast" in modeling_results:
                forecast = modeling_results["forecast"]
                modeling_summary += f"Forecast generated for {forecast.get('horizon_days', 0)} days. "

            sections.append({
                "title": "Financial Modeling",
                "content": modeling_summary,
                "data": modeling_results,
            })

        # Category breakdown
        category_col = next(
            (col for col in df.columns if "category" in col.lower()), None
        )
        if category_col:
            category_summary = df.groupby(category_col).size().to_dict()
            sections.append({
                "title": "Category Breakdown",
                "content": f"Transactions distributed across {len(category_summary)} categories.",
                "data": category_summary,
            })

        return NarrativeReport(
            title=f"Financial Report - {period}",
            summary=summary,
            sections=sections,
            generated_at=datetime.now().isoformat(),
            metadata={
                "source": emberframe.metadata.source,
                "source_type": emberframe.metadata.source_type,
                "rows": total_transactions,
            },
        )

    def _generate_json_export(self, inputs: Dict[str, Any]) -> Dict[str, Any]:
        """
        Generate JSON export of all financial data.

        Args:
            inputs: Input data

        Returns:
            Dictionary for JSON export
        """
        emberframe = inputs["emberframe"]

        export = {
            "metadata": {
                "source": emberframe.metadata.source,
                "source_type": emberframe.metadata.source_type,
                "exported_at": datetime.now().isoformat(),
                "total_rows": len(emberframe.data),
            },
            "data": emberframe.data.to_dict(orient="records"),
            "column_types": {
                col: str(dtype) for col, dtype in emberframe.data.dtypes.items()
            },
        }

        # Include analysis results if available
        if "audit_report" in inputs:
            export["audit_report"] = inputs["audit_report"]

        if "modeling_results" in inputs:
            export["modeling_results"] = inputs["modeling_results"]

        return export

    def _generate_excel_export(self, inputs: Dict[str, Any]) -> Dict[str, pd.DataFrame]:
        """
        Generate Excel workbook data.

        Args:
            inputs: Input data

        Returns:
            Dictionary of sheet names to DataFrames
        """
        emberframe = inputs["emberframe"]
        sheets = {}

        # Main data sheet
        sheets["Transactions"] = emberframe.data

        # Summary sheet
        summary_data = {
            "Metric": ["Total Rows", "Source", "Source Type"],
            "Value": [
                len(emberframe.data),
                emberframe.metadata.source,
                emberframe.metadata.source_type,
            ],
        }
        sheets["Summary"] = pd.DataFrame(summary_data)

        # Category summary if available
        category_col = next(
            (col for col in emberframe.data.columns if "category" in col.lower()), None
        )
        if category_col:
            amount_col = next(
                (col for col in emberframe.data.columns if "amount" in col.lower()), None
            )
            if amount_col:
                category_summary = (
                    emberframe.data.groupby(category_col)[amount_col]
                    .agg(["count", "sum", "mean"])
                    .reset_index()
                )
                sheets["Category Summary"] = category_summary

        return sheets

    def _generate_markdown_report(self, inputs: Dict[str, Any]) -> str:
        """
        Generate Markdown formatted report.

        Args:
            inputs: Input data

        Returns:
            Markdown string
        """
        narrative = self._generate_narrative_report(inputs)

        md = f"# {narrative.title}\n\n"
        md += f"**Generated:** {narrative.generated_at}\n\n"
        md += f"## Executive Summary\n\n{narrative.summary}\n\n"

        for section in narrative.sections:
            md += f"## {section['title']}\n\n"
            md += f"{section['content']}\n\n"

            # Add data as table if simple enough
            if isinstance(section.get("data"), dict):
                data = section["data"]
                if len(data) < 20:  # Only for small datasets
                    md += "| Key | Value |\n"
                    md += "|-----|-------|\n"
                    for key, value in data.items():
                        md += f"| {key} | {value} |\n"
                    md += "\n"

        md += "---\n\n"
        md += "*Generated by EmberQuant Controller Agent*\n"

        return md

    def _save_narrative(self, narrative: NarrativeReport, path: str) -> None:
        """Save narrative report to file."""
        output_path = Path(path).with_suffix(".json")
        with open(output_path, "w", encoding="utf-8") as f:
            json.dump(narrative.dict(), f, indent=2, default=str)
        self._log(f"Saved narrative report to {output_path}")

    def _save_excel(
        self, sheets: Dict[str, pd.DataFrame], path: Path, emberframe: EmberFrame
    ) -> None:
        """Save Excel workbook."""
        with pd.ExcelWriter(path, engine="openpyxl") as writer:
            for sheet_name, df in sheets.items():
                df.to_excel(writer, sheet_name=sheet_name, index=False)
        self._log(f"Saved Excel workbook to {path}")
