"""
Command-line interface for EmberQuant.
"""

from pathlib import Path
from typing import Optional

import typer
from rich.console import Console
from rich.table import Table

app = typer.Typer(
    name="emberquant",
    help="EmberQuant: The Agentic Framework for AccountingTech",
    add_completion=False,
)
console = Console()


@app.command()
def version() -> None:
    """Display EmberQuant version."""
    from emberquant import __version__

    console.print(f"EmberQuant version: {__version__}", style="bold green")


@app.command()
def audit(
    file_path: Path = typer.Argument(..., help="Path to the financial data file"),
    output: Optional[Path] = typer.Option(None, help="Path to save the audit report"),
    verbose: bool = typer.Option(False, "--verbose", "-v", help="Enable verbose output"),
) -> None:
    """
    Run an audit on a financial data file.

    This command loads data from a CSV or Excel file, runs the Clerk and Auditor
    agents, and displays the audit results.
    """
    console.print(f"[bold]EmberQuant Audit[/bold]\n", style="cyan")
    console.print(f"Loading data from: {file_path}")

    try:
        import emberquant as eq

        # Connect to the file
        adapter = eq.connect(str(file_path))

        # Fetch the data
        data = adapter.fetch_ledger()
        console.print(f"Loaded {len(data)} rows")

        # Create execution plan
        execution_plan = eq.plan("Audit financial data", verbose=verbose)

        # Execute the plan
        from emberquant.core.planning import execute_plan

        results = execute_plan(
            execution_plan,
            inputs={"data": data, "source": str(file_path), "source_type": "csv"},
            verbose=verbose,
        )

        # Extract audit report
        audit_report = None
        for node_results in results["results"].values():
            if "audit_report" in node_results:
                audit_report = node_results["audit_report"]
                break

        if audit_report:
            # Display results
            console.print("\n[bold]Audit Results[/bold]", style="green")
            console.print(f"Total Rows: {audit_report.total_rows}")
            console.print(f"Total Amount: ${audit_report.total_amount:,.2f}")
            console.print(f"\n{audit_report.summary}\n")

            if audit_report.findings:
                table = Table(title="Findings")
                table.add_column("Severity", style="cyan")
                table.add_column("Category", style="magenta")
                table.add_column("Description", style="white")

                for finding in audit_report.findings:
                    table.add_row(
                        finding.severity.upper(),
                        finding.category,
                        finding.description,
                    )

                console.print(table)

            # Save report if requested
            if output:
                import json

                with open(output, "w") as f:
                    json.dump(audit_report.model_dump(), f, indent=2, default=str)
                console.print(f"\nReport saved to: {output}", style="green")

        else:
            console.print("[red]No audit report generated[/red]")

    except Exception as e:
        console.print(f"[red]Error: {e}[/red]")
        raise typer.Exit(1)


@app.command()
def info(file_path: Path = typer.Argument(..., help="Path to the financial data file")) -> None:
    """
    Display information about a financial data file.
    """
    console.print(f"[bold]File Information[/bold]\n", style="cyan")

    try:
        import emberquant as eq

        # Connect and load data
        adapter = eq.connect(str(file_path))
        data = adapter.fetch_ledger()

        # Create EmberFrame
        from emberquant.agents.clerk import ClerkAgent

        clerk = ClerkAgent()
        result = clerk.execute(
            {"data": data, "source": str(file_path), "source_type": "csv"}
        )

        emberframe = result["emberframe"]

        # Display info
        console.print(f"Source: {emberframe.metadata.source}")
        console.print(f"Rows: {emberframe.metadata.row_count}")
        console.print(f"Columns: {emberframe.metadata.column_count}\n")

        # Column info table
        table = Table(title="Columns")
        table.add_column("Column Name", style="cyan")
        table.add_column("Semantic Type", style="magenta")
        table.add_column("Confidence", style="green")

        for col in emberframe.metadata.columns:
            table.add_row(
                col.name,
                col.semantic_type.value,
                f"{col.confidence:.2%}",
            )

        console.print(table)

        # Preview data
        console.print("\n[bold]Data Preview[/bold]")
        console.print(emberframe.head())

    except Exception as e:
        console.print(f"[red]Error: {e}[/red]")
        raise typer.Exit(1)


if __name__ == "__main__":
    app()
