"""
Connection management for EmberQuant.

Provides the connect() function for establishing connections to various
accounting data sources.
"""

from __future__ import annotations

from typing import Any, Dict, Union

from emberquant.adapters.base import BaseAdapter, ConnectionConfig
from emberquant.adapters.csv_adapter import CSVAdapter
from emberquant.adapters.quickbooks_adapter import QuickBooksAdapter
from emberquant.adapters.xero_adapter import XeroAdapter
from emberquant.core.emberframe import EmberFrame


def connect(
    source: Union[str, Dict[str, Any]],
    adapter_type: str = "auto",
    **kwargs: Any,
) -> EmberFrame:
    """
    Connect to a data source and return an adapter.

    This is the main entry point for establishing connections to accounting
    systems. It automatically selects the appropriate adapter based on the
    source type or explicit adapter_type parameter.

    Args:
        source: Either a file path (str) or a configuration dict with credentials
        adapter_type: Type of adapter to use. Options:
            - "auto": Auto-detect from source
            - "csv": CSV/Excel files
            - "quickbooks": QuickBooks Online
            - "xero": Xero Accounting
        **kwargs: Additional options passed to the adapter

    Returns:
        Connected adapter instance

    Examples:
        # Connect to a CSV file
        ledger = eq.connect("./data/transactions.csv")

        # Connect to QuickBooks
        ledger = eq.connect(
            {"access_token": "...", "realm_id": "..."},
            adapter_type="quickbooks"
        )

        # Connect to Xero
        ledger = eq.connect(
            {"access_token": "...", "tenant_id": "..."},
            adapter_type="xero"
        )
    """
    # Auto-detect adapter type from source
    if adapter_type == "auto":
        if isinstance(source, str):
            if source.endswith((".csv", ".xlsx", ".xls")):
                adapter_type = "csv"
            else:
                raise ValueError(
                    f"Cannot auto-detect adapter type for source: {source}. "
                    "Please specify adapter_type explicitly."
                )
        elif isinstance(source, dict):
            # Try to infer from dict structure
            if "realm_id" in source:
                adapter_type = "quickbooks"
            elif "tenant_id" in source:
                adapter_type = "xero"
            else:
                raise ValueError(
                    "Cannot auto-detect adapter type from credentials dict. "
                    "Please specify adapter_type explicitly."
                )
        else:
            raise ValueError(f"Unsupported source type: {type(source)}")

    # Create configuration
    if adapter_type == "csv":
        if not isinstance(source, str):
            raise ValueError("CSV adapter requires a file path as source")

        config = ConnectionConfig(
            adapter_type="csv",
            options={"file_path": source, **kwargs},
        )
        adapter_class = CSVAdapter

    elif adapter_type == "quickbooks":
        if not isinstance(source, dict):
            raise ValueError("QuickBooks adapter requires credentials dict as source")

        config = ConnectionConfig(
            adapter_type="quickbooks",
            credentials=source,
            options=kwargs,
        )
        adapter_class = QuickBooksAdapter

    elif adapter_type == "xero":
        if not isinstance(source, dict):
            raise ValueError("Xero adapter requires credentials dict as source")

        config = ConnectionConfig(
            adapter_type="xero",
            credentials=source,
            options=kwargs,
        )
        adapter_class = XeroAdapter

    else:
        raise ValueError(
            f"Unsupported adapter type: {adapter_type}. "
            f"Supported types: csv, quickbooks, xero"
        )

    # Create and connect the adapter
    adapter = adapter_class(config)
    adapter.connect()

    # Fetch data and return as EmberFrame
    data = adapter.fetch_ledger()

    # Determine source name
    if isinstance(source, str):
        source_name = source
    else:
        source_name = adapter_type

    emberframe = EmberFrame.from_dataframe(
        df=data,
        source=source_name,
        source_type=adapter_type,
        infer_types=True
    )

    return emberframe
