"""
Planning and execution system for EmberQuant.

Provides the plan() function for creating execution plans from natural
language descriptions.
"""

from __future__ import annotations

from typing import Any, Dict, Optional

from emberquant.agents.auditor import AuditorAgent
from emberquant.agents.clerk import ClerkAgent
from emberquant.agents.modeler import ModelerAgent
from emberquant.agents.controller import ControllerAgent
from emberquant.core.embергraph import EmberGraph, ExecutionPlan


def plan(
    task_description: str,
    verbose: bool = False,
    llm_provider: Optional[str] = None,
    llm_model: Optional[str] = None,
) -> ExecutionPlan:
    """
    Create an execution plan from a natural language task description.

    This function analyzes the task and determines which agents need to run
    and in what order. In Phase 1, it uses rule-based planning. In future
    phases, it will use LLM-powered planning.

    Args:
        task_description: Natural language description of the task
        verbose: Whether to print planning details
        llm_provider: Optional LLM provider for advanced planning (future)
        llm_model: Optional LLM model for advanced planning (future)

    Returns:
        ExecutionPlan describing the steps to execute

    Examples:
        plan = eq.plan("Audit Q3 marketing spend")
        plan = eq.plan("Analyze revenue trends and detect anomalies")
    """
    if verbose:
        print(f"Planning task: {task_description}")

    # Phase 1: Rule-based planning
    # Analyze keywords to determine which agents to use
    task_lower = task_description.lower()

    nodes = []
    estimated_steps = 0

    # Check if we need data ingestion (Clerk)
    needs_clerk = any(
        keyword in task_lower
        for keyword in ["load", "ingest", "import", "read", "connect", "fetch"]
    )

    # Check if we need auditing (Auditor)
    needs_auditor = any(
        keyword in task_lower
        for keyword in [
            "audit",
            "analyze",
            "detect",
            "anomaly",
            "benford",
            "variance",
            "review",
            "check",
        ]
    )

    # If no specific keywords, assume we need both (common case)
    if not needs_clerk and not needs_auditor:
        needs_clerk = True
        needs_auditor = True

    # Auditor always needs Clerk to provide data
    if needs_auditor and not needs_clerk:
        needs_clerk = True

    # Build the plan
    if needs_clerk:
        nodes.append(
            {
                "agent": "ClerkAgent",
                "description": "Ingest and normalize financial data",
                "dependencies": [],
            }
        )
        estimated_steps += 1

    if needs_auditor:
        dependencies = []
        if needs_clerk:
            dependencies.append("ClerkAgent")

        nodes.append(
            {
                "agent": "AuditorAgent",
                "description": "Perform financial analysis and anomaly detection",
                "dependencies": dependencies,
            }
        )
        estimated_steps += 1

    # Future: Add Modeler and Controller agents
    # For now, we'll note them in the plan structure but not implement

    execution_plan = ExecutionPlan(
        task_description=task_description,
        nodes=nodes,
        estimated_steps=estimated_steps,
    )

    if verbose:
        print(f"Created plan with {estimated_steps} steps:")
        for i, node in enumerate(nodes, 1):
            deps = ", ".join(node["dependencies"]) if node["dependencies"] else "none"
            print(f"  {i}. {node['agent']} (depends on: {deps})")

    return execution_plan


def execute_plan(
    execution_plan: ExecutionPlan,
    inputs: Dict[str, Any],
    verbose: bool = False,
) -> Dict[str, Any]:
    """
    Execute an execution plan.

    Args:
        execution_plan: The plan to execute
        inputs: Initial inputs for the execution
        verbose: Whether to print execution progress

    Returns:
        Dictionary with execution results

    Examples:
        plan = eq.plan("Audit Q3 marketing spend")
        results = execute_plan(plan, {"data": df, "source": "csv"})
    """
    if verbose:
        print(f"Executing plan: {execution_plan.task_description}")

    # Create EmberGraph
    graph = EmberGraph()

    # Map node descriptions to actual agent instances
    agent_map = {
        "ClerkAgent": ClerkAgent(),
        "AuditorAgent": AuditorAgent(),
        "ModelerAgent": ModelerAgent(),
        "ControllerAgent": ControllerAgent(),
    }

    # Build the graph
    node_id_map: Dict[str, str] = {}

    for node_spec in execution_plan.nodes:
        agent_name = node_spec["agent"]
        agent = agent_map.get(agent_name)

        if agent is None:
            raise ValueError(f"Unknown agent: {agent_name}")

        # Map dependency names to node IDs
        dep_ids = [node_id_map[dep] for dep in node_spec["dependencies"]]

        # Add the node
        node_id = graph.add_node(
            agent=agent,
            dependencies=dep_ids,
            inputs=inputs if not dep_ids else {},  # Only root nodes get initial inputs
        )

        node_id_map[agent_name] = node_id

    # Execute the graph
    results = graph.execute(verbose=verbose)

    # Add execution summary
    summary = graph.get_execution_summary()

    return {
        "results": results,
        "summary": summary,
        "graph": graph,
    }
