from abc import ABC, abstractmethod
from typing import Any, Callable, Type

from fastapi import Depends, FastAPI, Request
from pydantic import BaseModel, create_model


class SelectEngine(ABC):

    @abstractmethod
    def select(self, query: str, *params: Any) -> list[Any]:
        pass

    @abstractmethod
    def select_count(self, query: str, *params: Any) -> int:
        pass


# Middleware type definition
Middleware = Callable[[BaseModel, int, int, dict[str, Any], Callable], Any]


def serve_select(
    app: FastAPI,
    query_engine: SelectEngine,
    select_query: str,
    select_param: Callable[[BaseModel, int, int], list[Any]] | None = None,
    count_query: str | None = None,
    count_param: Callable[[BaseModel], list[Any]] | None = None,
    path: str = "/get",
    filter_model: Type[BaseModel] | None = None,
    data_model: Type[BaseModel] | None = None,
    dependencies: list[Any] | None = None,
    middlewares: list[Middleware] | None = None,
):
    """
    Adding fastapi route to app with proper annotation:
    - taking filter_model as parameter
    - return pydantic model with two property:
        - data (data_model)
        - count (int)
    - supports FastAPI dependencies
    - supports middleware system
    """
    if filter_model is None:
        filter_model = create_model("FilterModel")
    if data_model is None:
        data_model = create_model("DataModel")
    if dependencies is None:
        dependencies = []
    if middlewares is None:
        middlewares = []

    class ResponseModel(BaseModel):
        data: list[data_model] | list[Any]
        count: int | None = None

    def create_middleware_chain():
        """Create the middleware chain with the final handler at the end."""

        def final_handler(
            filters: BaseModel,
            limit: int,
            offset: int,
            dependency_results: dict[str, Any],
        ) -> ResponseModel:
            """The final handler that executes the queries."""
            select_args = (
                select_param(filters, limit, offset)
                if select_param is not None
                else [limit, offset]
            )
            data = query_engine.select(select_query, *select_args)
            total = -1
            if count_query:
                count_args = count_param(filters) if count_param is not None else []
                total = query_engine.select_count(count_query, *count_args)
                return ResponseModel(data=data, count=total)
            return ResponseModel(data=data)

        # Build the middleware chain from right to left
        handler = final_handler
        for middleware in reversed(middlewares):
            # Create a closure-safe wrapper
            def make_wrapper(current_middleware, next_handler):
                def wrapper(
                    filters: BaseModel,
                    limit: int,
                    offset: int,
                    dependency_results: dict[str, Any],
                ):
                    return current_middleware(
                        filters,
                        limit,
                        offset,
                        dependency_results,
                        lambda: next_handler(
                            filters, limit, offset, dependency_results
                        ),
                    )

                return wrapper

            handler = make_wrapper(middleware, handler)

        return handler

    # Create dependency wrappers to capture results
    wrapped_dependencies = []
    for dep in dependencies:
        if hasattr(dep, "dependency"):
            # Create a closure-safe wrapper for each dependency
            def create_wrapped_dependency(original_dep):
                async def wrapped_dependency(request: Request):
                    result = await original_dep.dependency()
                    # Store the result in request state
                    if not hasattr(request.state, "dependency_results"):
                        request.state.dependency_results = {}
                    # Use the dependency function name as key
                    dep_name = original_dep.dependency.__name__
                    request.state.dependency_results[dep_name] = result
                    return result

                return wrapped_dependency

            wrapped_dependencies.append(Depends(create_wrapped_dependency(dep)))
        else:
            wrapped_dependencies.append(dep)

    # Create the route with dependencies
    @app.get(path, response_model=ResponseModel, dependencies=wrapped_dependencies)
    def get(
        filters: filter_model = Depends(),
        limit: int = 100,
        offset: int = 0,
        request: Request = None,
    ):
        """
        This route is automatically generated by dbanu
        """
        # Extract dependency results from request state
        dependency_results = {}
        if request and hasattr(request.state, "dependency_results"):
            dependency_results = request.state.dependency_results

        handler = create_middleware_chain()
        return handler(filters, limit, offset, dependency_results)
