# -*- coding: utf-8 -*-
# *******************************************************
#   ____                     _               _
#  / ___|___  _ __ ___   ___| |_   _ __ ___ | |
# | |   / _ \| '_ ` _ \ / _ \ __| | '_ ` _ \| |
# | |__| (_) | | | | | |  __/ |_ _| | | | | | |
#  \____\___/|_| |_| |_|\___|\__(_)_| |_| |_|_|
#
#  Sign up for free at http://www.comet.ml
#  Copyright (C) 2025 Comet ML INC
#  This file can not be copied and/or distributed without the express
#  permission of Comet ML Inc.
# *******************************************************
import sys
from typing import Any, Dict

import pandas as pd

if sys.version_info[:2] >= (3, 8):
    import importlib.metadata as importlib_metadata
else:
    import importlib_metadata


def get_comet_version() -> str:
    try:
        return importlib_metadata.version("comet_mpm")
    except importlib_metadata.PackageNotFoundError:
        return "Please install comet with `pip install comet_mpm`"


def custom_metrics_to_dataframe(data: Dict[str, Any], sql: str) -> pd.DataFrame:
    """
    Convert the given dict into a pandas DataFrame.
    """
    # Collect the dates
    dates = [point["x"] for point in data["data"][0]["data"]]
    df = pd.DataFrame({"timestamp": pd.to_datetime(dates)})

    # For each series, add a column
    sql_set = set()
    series_data = {}
    for series in data["data"]:
        if "data" not in series:
            continue
        # FIXME: only allow one select statement, and get it from parameter
        colname = sql
        sql_set.add(colname)
        predkey = series.get("predicateKey", "predicate")
        y_values = [point["y"] for point in series["data"]]
        key = (predkey, colname)
        series_data[key] = y_values

    series_df = pd.DataFrame(series_data)
    df = pd.concat([df, series_df], axis=1)
    df.set_index("timestamp", inplace=True)
    if len(sql_set) == 1:
        columns = [
            [col.replace("'", "") for col in name][0]
            for i, name in enumerate(df.columns)
        ]
        df.columns = columns
    else:
        columns = [
            [col.replace("'", "") for col in name] for i, name in enumerate(df.columns)
        ]
        df.columns = pd.MultiIndex.from_tuples(
            columns, names=["predicateKey", "columnName"]
        )
    return df
