import numpy as np
import pytest

from mdance.cluster.shine import Shine


@pytest.fixture
def pathways():
    """
    Fixture for pathways. The format is a list of tuples, where each tuple contains 
    a string and a numpy array. The string refers to the pathway number, and the numpy 
    array contains the features.
    """
    frames_all = [
    ('0', np.array([[-74.23699951,  67.61009979],
        [-53.13778687, 113.60979462],
        [-53.13781738, 113.60980225],
        [ 34.90659714,  95.15899658],
        [ 63.34859848,  27.15789986],
        [ 44.89619827,  42.38359833],
        [ 45.58160019,  27.8560009 ],
        [ 54.48529816,  71.10389709],
        [ 55.10719681,   6.53830004],
        [ 61.41569901,  38.7390976 ]])), ('1', np.array([[ -78.74139404,   86.88970184],
        [-124.14131165,  133.28540039],
        [ -62.45608521,  179.37069702],
        [ -94.38858032, -179.26660156],
        [-122.37721252,  122.54660034],
        [ -75.46469116,  166.67678833],
        [ -89.6315918 ,  152.05979919],
        [ -59.07040405,  162.12379456],
        [ -62.35968018,  153.83779907],
        [ -90.17648315,  152.10429382]])), ('2', np.array([[-142.79400635,   95.42210388],
        [ -88.90679932,  131.95779419],
        [-153.45500183,  165.94718933],
        [-149.34820557, -174.05441284],
        [-116.89601135,  167.75419617],
        [-116.65299988,  159.13800049],
        [ -61.93249512,  143.74169922],
        [ -90.13357544,  143.02029419],
        [-118.12361145,  179.47659302],
        [-132.56130981,  176.59658813]])), ('3', np.array([[-142.79400635,   95.42210388],
        [ -88.90679932,  131.95779419],
        [-153.45500183,  165.94718933],
        [-149.34820557, -174.05441284],
        [-116.89601135,  167.75419617],
        [-116.65299988,  159.13800049],
        [ -61.93249512,  143.74169922],
        [ -90.13357544,  143.02029419],
        [-118.12361145,  179.47659302],
        [-132.56130981,  176.59658813]])), ('4', np.array([[-142.79400635,   95.42210388],
        [ -88.90679932,  131.95779419],
        [-153.45500183,  165.94718933],
        [-149.34820557, -174.05441284],
        [-116.89601135,  167.75419617],
        [-116.65299988,  159.13800049],
        [ -61.93249512,  143.74169922],
        [ -90.13357544,  143.02029419],
        [-118.12361145,  179.47659302],
        [-132.56130981,  176.59658813]])), ('5', np.array([[-142.79400635,   95.42210388],
        [ -88.90679932,  131.95779419],
        [-153.45500183,  165.94718933],
        [-149.34820557, -174.05441284],
        [-116.89601135,  167.75419617],
        [-116.65299988,  159.13800049],
        [ -61.93249512,  143.74169922],
        [ -90.13357544,  143.02029419],
        [-118.12361145,  179.47659302],
        [-132.56130981,  176.59658813]])), ('6', np.array([[-142.79400635,   95.42210388],
        [ -88.90679932,  131.95779419],
        [-153.45500183,  165.94718933],
        [-149.34820557, -174.05441284],
        [-116.89601135,  167.75419617],
        [-116.65299988,  159.13800049],
        [ -61.93249512,  143.74169922],
        [ -90.13357544,  143.02029419],
        [-118.12361145,  179.47659302],
        [-132.56130981,  176.59658813]])), ('7', np.array([[-84.03909302,  75.73680115],
        [-99.19448853, 143.54150391],
        [-68.46630859, 137.82501221],
        [ -5.01208496,  97.85250092],
        [ 55.18330002,  41.81550217],
        [ 41.93090057,  44.16210175],
        [ 46.14450073,  27.89189911],
        [ 48.44320297,  59.91820145],
        [ 69.89299774,  21.73629951],
        [ 42.33309937,  49.27910233]])), ('8', np.array([[ -77.67678833,   95.04299927],
        [ -69.0296936 ,  143.3453064 ],
        [ -94.15249634, -172.65719604],
        [ -96.9342041 ,  148.96929932],
        [ -95.92849731,  143.04420471],
        [ -70.60339355,  148.94981384],
        [ -72.47839355,  161.40290833],
        [-101.28518677,  162.35510254],
        [ -97.30999756,  143.41461182],
        [ -80.29470825, -164.96868896]])), ('9', np.array([[ -77.67678833,   95.04299927],
        [ -69.0296936 ,  143.3453064 ],
        [ -94.15249634, -172.65719604],
        [ -96.9342041 ,  148.96929932],
        [ -95.92849731,  143.04420471],
        [ -70.60339355,  148.94981384],
        [ -72.47839355,  161.40290833],
        [-101.28518677,  162.35510254],
        [ -97.30999756,  143.41461182],
        [ -80.29470825, -164.96868896]])), ('10', np.array([[ -77.67678833,   95.04299927],
        [ -69.0296936 ,  143.3453064 ],
        [ -94.15249634, -172.65719604],
        [ -96.9342041 ,  148.96929932],
        [ -95.92849731,  143.04420471],
        [ -70.60339355,  148.94981384],
        [ -72.47839355,  161.40290833],
        [-101.28518677,  162.35510254],
        [ -97.30999756,  143.41461182],
        [ -80.29470825, -164.96868896]])), ('11', np.array([[ -77.67678833,   95.04299927],
        [ -69.0296936 ,  143.3453064 ],
        [ -94.15249634, -172.65719604],
        [ -96.9342041 ,  148.96929932],
        [ -95.92849731,  143.04420471],
        [ -70.60339355,  148.94981384],
        [ -72.47839355,  161.40290833],
        [-101.28518677,  162.35510254],
        [ -97.30999756,  143.41461182],
        [ -80.29470825, -164.96868896]])), ('12', np.array([[ -77.67678833,   95.04299927],
        [ -69.0296936 ,  143.3453064 ],
        [ -94.15249634, -172.65719604],
        [ -96.9342041 ,  148.96929932],
        [ -95.92849731,  143.04420471],
        [ -70.60339355,  148.94981384],
        [ -72.47839355,  161.40290833],
        [-101.28518677,  162.35510254],
        [ -97.30999756,  143.41461182],
        [ -80.29470825, -164.96868896]])), ('13', np.array([[ -77.67678833,   95.04299927],
        [ -69.0296936 ,  143.3453064 ],
        [ -94.15249634, -172.65719604],
        [ -96.9342041 ,  148.96929932],
        [ -95.92849731,  143.04420471],
        [ -70.60339355,  148.94981384],
        [ -72.47839355,  161.40290833],
        [-101.28518677,  162.35510254],
        [ -97.30999756,  143.41461182],
        [ -80.29470825, -164.96868896]])), ('14', np.array([[ -77.67678833,   95.04299927],
        [ -69.0296936 ,  143.3453064 ],
        [ -94.15249634, -172.65719604],
        [ -96.9342041 ,  148.96929932],
        [ -95.92849731,  143.04420471],
        [ -70.60339355,  148.94981384],
        [ -72.47839355,  161.40290833],
        [-101.28518677,  162.35510254],
        [ -97.30999756,  143.41461182],
        [ -80.29470825, -164.96868896]]))
    ]
    return frames_all


def test_process_trajs(pathways):
    """Test the process_trajs method of the Shine class."""
    for sampling in ['diversity', 'quota']:
        mod = Shine(pathways, 'MSD', N_atoms=1 , t=2, criterion='maxclust', 
                    link='ward', merge_scheme='intra', sampling=sampling, 
                    frac=0.25)
        pathway_dict = mod.process_trajs()
        # how many frames are in each trajectory of pathways
        n_frames_pathways = [len(pathway[1]) for pathway in pathways]
        n_frames_dict = [len(pathway_dict[key]) for key in pathway_dict]
        # n_frames_dict should be equal to 0.25 of n_frames_pathways
        assert n_frames_dict == [int(n * 0.25) if n >= 50 else n \
                                 for n in n_frames_pathways]
        assert len(pathway_dict) == 15


def test_shine(pathways):
    """Test the run method of the Shine class to see if the 
    cluster labels are correct."""
    labels_dict = {
        
        0.25: {
            'intra': ['0, 7-11, 13', '1-6, 12, 14'],
            'semi_sum': ['0, 7', '1-6, 8-14'],
            'min': ['0, 7', '1-6, 8-14']
        },
        
        0.5: {
            'intra': ['0, 7-11, 13', '1-6, 12, 14'],
            'semi_sum': ['0, 7', '1-6, 8-14'],
            'min': ['0, 7', '1-6, 8-14']
        }, 
        
        1: {
            'intra': ['0, 7-11, 13', '1-6, 12, 14'],
            'semi_sum': ['0, 7', '1-6, 8-14'],
            'min': ['0, 7', '1-6, 8-14']
        }
    
    }

    for frac in [0.25, 0.5, 1]:
        for merge_scheme in ['intra', 'semi_sum', 'min']:
            mod = Shine(pathways, 'MSD', N_atoms=1 , t=2, criterion='maxclust', 
                        link='ward', merge_scheme=merge_scheme, 
                        sampling='diversity', frac=frac)
            mod.run()
            custom_labels = mod.labels()
            assert len(np.unique(custom_labels)) == 2
            assert custom_labels == labels_dict[frac][merge_scheme]


def test_check_merge_scheme():
    """Test the ``_check_merge_scheme`` method of the Shine class bc
    it should raise an error if the merge_scheme is not in the list."""
    merge_scheme = 'semi-sum'
    with pytest.raises(ValueError):
        mod = Shine(pathways, 'MSD', N_atoms=1 , t=2, criterion='maxclust', 
                        link='ward', merge_scheme=merge_scheme)
        mod._check_merge_scheme()


def test_check_sampling():
    """Test the ``_check_sampling method`` of the Shine class because
    it should raise an error if the sampling is not in the list."""
    sampling = 'Diversity'
    with pytest.raises(ValueError):
        mod = Shine(pathways, 'MSD', N_atoms=1 , t=2, criterion='maxclust', 
                    link='ward', sampling=sampling)
        mod._check_sampling()
    

def test_plot(pathways):
    """Test the plot method of the Shine class to make sure it runs."""
    mod = Shine(pathways, 'MSD', N_atoms=1 , t=2, criterion='maxclust', 
                link='ward', merge_scheme='intra', sampling='diversity', 
                frac=0.25)
    mod.run()
    assert mod.plot() is not None