#!/bin/env python
# -*- coding: utf-8 -*-
##
# setup.py: Installs Python host functionality for azure-quantum.
##
# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License.
##

# IMPORTS #

import setuptools
import os
import glob
import re

# VERSION INFORMATION #
# Our build process sets the PYTHON_VERSION environment variable to a version
# string that is compatible with PEP 440, and so we inherit that version number
# here and propagate that to qsharp/version.py.
#
# To make sure that local builds still work without the same environment
# variables, we'll default to 0.0.1 as a development version.

version = os.environ.get("PYTHON_VERSION", "0.0.1")

with open("./azure/quantum/version.py", "w") as f:
    f.write(
        f"""# Auto-generated file, do not edit.
##
# version.py: Specifies the version of the azure.quantum package.
##
# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License.
##
__version__ = "{version}"
"""
    )

with open("./azure/quantum/_client/_version.py", "w") as f:
    f.write(
        f"""# coding=utf-8
# --------------------------------------------------------------------------
# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See License.txt in the project root for license information.
# Code generated by Microsoft (R) AutoRest Code Generator.
# Changes may cause incorrect behavior and will be lost if the code is regenerated.
# --------------------------------------------------------------------------

VERSION = "{version}"
"""
    )

# DESCRIPTION #
# The long description metadata passed to setuptools is used to populate the
# PyPI page for this package. Thus, we'll generate the description by using the
# same README.md file that we use in the GitHub repo.

with open("./README.md", "r") as fh:
    long_description = fh.read()


# LIST OF REQUIREMENTS #
def read_requirements(requirement_file: str) -> list[str]:
    with open(requirement_file, "r") as fh:
        requirements = fh.readlines()
        requirements = [v.strip(' \n\r') for v in requirements]
        requirements = list(filter(None, requirements))
        return requirements


requirements = read_requirements("requirements.txt")

# LIST OF EXTRA/OPTIONAL REQUIREMENTS #
#
# All `requirements-option.txt` are extra/optional requirements
# Identified by the `option`.
#
# To be used with `pip install azurequantum[option1, option2, ...]`
# Example: `pip install azurequantum[qsharp,qiskit]`
#
# Instead of listing the individual options, you can pass `all`
# Example, `pip install azurequantum[all]`
extras_require = {}
all_requirements = []
for extra_requirement_file in glob.glob("requirements-*.txt"):
    extra_option = re.match(r"requirements-(\w+)\.txt", extra_requirement_file).group(1)
    extra_requirements = read_requirements(extra_requirement_file)
    extras_require[extra_option] = extra_requirements
    all_requirements.extend(extra_requirements)
extras_require["all"] = all_requirements

# SETUPTOOLS INVOCATION #
setuptools.setup(
    name="azure-quantum",
    version=version,
    author="Microsoft",
    description="Python client for Azure Quantum",
    license="MIT License",
    long_description=long_description,
    long_description_content_type="text/markdown",
    url="https://github.com/microsoft/azure-quantum-python",
    packages=setuptools.find_namespace_packages(include=["azure.*"]),
    classifiers=[
        "Programming Language :: Python :: 3",
        "License :: OSI Approved :: MIT License",
        "Operating System :: OS Independent",
    ],
    python_requires=">=3.8",
    install_requires=requirements,
    extras_require=extras_require
)
