import math
from typing import Optional
from langevals_core.base_evaluator import (
    BaseEvaluator,
    EvaluationResult,
    EvaluationResultSkipped,
    EvaluatorEntry,
    SingleEvaluationResult,
)
from ragas import SingleTurnSample
from .lib.common import (
    BaseEvaluator,
    RagasResult,
    capture_cost,
    check_max_tokens,
    clear_context,
    env_vars,
    RagasSettings,
    prepare_llm,
)
from pydantic import Field
from ragas.metrics import Faithfulness
from langchain_core.prompt_values import StringPromptValue


class RagasFaithfulnessEntry(EvaluatorEntry):
    input: Optional[str] = Field(default="")
    output: str
    contexts: list[str]


class RagasFaithfulnessResult(EvaluationResult):
    score: float = Field(
        default=0.0,
        description="A score between 0.0 and 1.0 indicating the faithfulness of the answer.",
    )


class RagasFaithfulnessSettings(RagasSettings):
    autodetect_dont_know: bool = Field(
        default=True,
        description="Whether to autodetect 'I don't know' in the output to avoid failing the evaluation.",
    )


class RagasFaithfulnessEvaluator(
    BaseEvaluator[
        RagasFaithfulnessEntry, RagasFaithfulnessSettings, RagasFaithfulnessResult
    ]
):
    """
    This evaluator assesses the extent to which the generated answer is consistent with the provided context. Higher scores indicate better faithfulness to the context, useful for detecting hallucinations.
    """

    name = "Ragas Faithfulness"
    category = "rag"
    env_vars = env_vars
    default_settings = RagasFaithfulnessSettings()
    docs_url = "https://docs.ragas.io/en/stable/concepts/metrics/available_metrics/faithfulness/"
    is_guardrail = False

    def evaluate(self, entry: RagasFaithfulnessEntry) -> SingleEvaluationResult:
        llm, _ = prepare_llm(self, self.settings)

        contexts = clear_context(entry.contexts)
        skip = check_max_tokens(
            input=entry.input,
            output=entry.output,
            contexts=contexts,
            settings=self.settings,
        )
        if skip:
            return skip

        scorer = Faithfulness()
        scorer.llm = llm

        _original_create_statements = scorer._create_statements
        _original_create_verdicts = scorer._create_verdicts
        ragas_statements = []
        ragas_verdicts = []

        async def _create_statements(self, row: dict, callbacks):
            nonlocal ragas_statements
            statements = await _original_create_statements(row, callbacks)
            for components in statements.sentences:
                ragas_statements += components.simpler_statements
            return statements

        scorer._create_statements = _create_statements.__get__(scorer)

        async def _create_verdicts(self, row: dict, statements: list[str], callbacks):
            nonlocal ragas_verdicts
            verdicts = await _original_create_verdicts(row, statements, callbacks)
            ragas_verdicts += verdicts.statements
            return verdicts

        scorer._create_verdicts = _create_verdicts.__get__(scorer)

        with capture_cost(llm) as cost:
            score = scorer.single_turn_score(
                SingleTurnSample(
                    user_input=entry.input,
                    response=entry.output,
                    retrieved_contexts=contexts,
                )
            )

            if len(ragas_statements) == 0 or math.isnan(score):
                return EvaluationResultSkipped(
                    details="No claims found in the output to measure faitfhulness against context, skipping entry."
                )

            details = "\n\n".join(
                [
                    f'Claim: "{verdict.statement}"\nVerdict: {verdict.reason}\nScore: {verdict.verdict}'
                    for verdict in ragas_verdicts
                ]
            )

            if len(details) == 0:
                details = "\n\n".join(
                    [f'Claim: "{statement}"' for statement in ragas_statements]
                )

            if self.settings.autodetect_dont_know and score < 0.1:
                llm.langchain_llm.temperature = 0  # type: ignore
                is_dont_know = llm.langchain_llm.invoke(
                    input=StringPromptValue(
                        text=f"""
                        You are an LLM evaluator. You are given an output generated by another LLM and you need to determine if it is a 'I don't know', 'I don't have the information', 'I couldn't find anything' or 'I can't help you' statement. If it is, return True, otherwise return False. Return nothing else, only True or False.

                        Generated output:
                        > {entry.output}
                        """
                    ),
                )
                if "true" in is_dont_know.content.lower():  # type: ignore
                    return EvaluationResultSkipped(
                        details="The output seems correctly to be an 'I don't know' statement given the provided contexts. Skipping faithfulness score."
                    )

        return RagasResult(
            score=score,
            cost=cost,
            details=details,
        )
