# -*- coding: utf-8 -*-
################################################################################
#
#  Rattail -- Retail Software Framework
#  Copyright © 2010-2015 Lance Edgar
#
#  This file is part of Rattail.
#
#  Rattail is free software: you can redistribute it and/or modify it under the
#  terms of the GNU Affero General Public License as published by the Free
#  Software Foundation, either version 3 of the License, or (at your option)
#  any later version.
#
#  Rattail is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for
#  more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with Rattail.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################
"""
Core Field Renderers
"""

from __future__ import unicode_literals

import datetime

import formalchemy
from formalchemy.fields import AbstractField
from pyramid.renderers import render


class CustomFieldRenderer(formalchemy.FieldRenderer):
    """
    Base class for renderers which accept customization args, and "fake out"
    FormAlchemy by pretending to still be a renderer factory when in fact it's
    already dealing with a renderer instance.
    """

    def __init__(self, *args, **kwargs):
        if len(args) == 1 and isinstance(args[0], AbstractField):
            super(CustomFieldRenderer, self).__init__(args[0])
        else:
            assert len(args) == 0
            self.init(**kwargs)

    def __call__(self, field):
        super(CustomFieldRenderer, self).__init__(field)
        return self

    def init(self, **kwargs):
        pass

    @property
    def rattail_config(self):
        return self.request.rattail_config


class DateFieldRenderer(CustomFieldRenderer, formalchemy.DateFieldRenderer):
    """
    Date field renderer which uses jQuery UI datepicker widget when rendering
    in edit mode.
    """
    change_year = False

    def init(self, change_year=False):
        self.change_year = change_year

    def render(self, **kwargs):
        kwargs['name'] = self.name
        kwargs['value'] = self.value
        kwargs['change_year'] = self.change_year
        return render('/forms/fields/date.mako', kwargs)

    def deserialize(self):
        value = self._serialized_value()
        if not value:
            return None
        return datetime.datetime.strptime(value, '%Y-%m-%d')

    def _serialized_value(self):
        return self.params.getone(self.name)
