"""PostgreSQL Class module

Base class for interfacing with PostgreSQL to remove repetitive overhead of
psycopg2 use.

THIS IS NOT ASYNC

"""
__author__ = 'Gavin M. Roy'
__email__ = 'gmr@myyearbook.com'
__since__ = '2011-05-27'

import hashlib
import logging
from psycopg2 import connect as pg_connect
from psycopg2 import extensions
from psycopg2 import extras

_CONNECTIONS = None
_LOGGER = logging.getLogger('tinman.clients.pgsql')


def _add_cached_connection(connection_hash, connection):
    """Add the connection to our module level connection dictionary

    :param str connection_hash: Hash generated by _generate_connection
    :param psycopg2._psycopg.connection connection: PostgreSQL connection

    """
    global _CONNECTIONS

    # If we've not initialized the _CONNECTIONS global yet, do so
    if not _CONNECTIONS:
        _CONNECTIONS = dict()

    if connection_hash not in _CONNECTIONS:
        # Assign the connection to the pool
        _CONNECTIONS[connection_hash] = {'connections': 1,
                                         'connection': connection}
        _LOGGER.info('Appended connection %s to module pool', connection_hash)
    else:
        _LOGGER.error('Attempt to append already cached connection: %s',
                      connection_hash)


def _generate_connection_hash(dsn, async=False):
    """Generates a connection hash for the given parameters.

    :param str dsn: DSN for connection
    :param bool async: Connection is setup in async mode
    :returns: str

    """
    # Create our hashlib object
    sha = hashlib.sha1()

    # Update the hashlib sha1 with the value to hash
    sha.update("%s:%s" % (dsn, async))

    # Return the hexdigest of the sha1
    return str(sha.hexdigest())


def _get_cached_connection(connection_hash):
    """Check our global connection stack to see if we already have a
    connection with the same exact connection parameters and use it if so.

    :param str connection_hash: Hash generated by _generate_connection_hash
    :returns: psycopg2._psycopg.connection or None

    """
    global _CONNECTIONS

    # If we've not initialized the _CONNECTIONS global yet, do so
    if not _CONNECTIONS:
        _CONNECTIONS = dict()

    # If our connection_hash is in our global connection pool return it
    if connection_hash in _CONNECTIONS:

        # Increment our use counter
        _CONNECTIONS[connection_hash]['connections'] += 1

        # Return the active connection
        return _CONNECTIONS[connection_hash]['connection']

    # Return None, we don't have it in the pool
    return None


def _free_cached_connection(connection_hash):
    """Decrement our use counter for the hash and if it is the only one, delete
    the cached connection.

    :param str connection_hash: Hash generated by _generate_connection_hash

    """
    global _CONNECTIONS

    # If we've not initialized the _CONNECTIONS global yet, do so
    if not _CONNECTIONS:
        _CONNECTIONS = dict()

    # If our connection_hash is in our global connection pool return it
    if connection_hash in _CONNECTIONS:

        # Increment our use counter
        _CONNECTIONS[connection_hash]['connections'] -= 1

        # Remove the connection_hash object if there are no connections left
        if not _CONNECTIONS[connection_hash]['connections']:
            del _CONNECTIONS[connection_hash]


class PgSQL(object):
    """PostgreSQL connection object.

    Uses a module level cache of connections to reduce overhead.

    """
    def __init__(self, host='localhost', port=5432, dbname=None, user='www',
                 password=None):
        """Connect to a PostgreSQL server using the module wide connection and
        set the isolation level.

        :param str host: PostgreSQL Host
        :param port: PostgreSQL port
        :param str dbname: Dabase name
        :param str user: PostgreSQL user name
        :param str password: User's password

        """
        # Set the DSN
        dsn = "host='%s' port=%i dbname='%s' user='%s'" % \
              (host, port, dbname, user)

        # Add the password if specified
        if password:
            dsn += " password='%s'" % password

        # Generate a connection hash to keep a global instance of this
        # connection with
        self._connection_hash = _generate_connection_hash(dsn)

        # Attempt to get a cached connection from our module level pool
        self._pgsql = _get_cached_connection(self._connection_hash)

        # If we got a result, just log our success in doing so
        if self._pgsql:
            _LOGGER.debug("Re-using cached connection: %s",
                          self._connection_hash)

        # Create a new PostgreSQL connection and cache it
        else:
            # Connect to a PostgreSQL daemon
            _LOGGER.info("Connecting to %s:%i:%s as %s",
                              host, port, dbname, user)
            self._pgsql = pg_connect(dsn)

            # Commit after every command
            self._pgsql.set_isolation_level(
                extensions.ISOLATION_LEVEL_AUTOCOMMIT)
            _LOGGER.info('Connected to PostgreSQL')

            # Add the connection to our module level pool
            _add_cached_connection(self._connection_hash, self._pgsql)

        # Always get a new cursor
        self._cursor = self._pgsql.cursor(cursor_factory=extras.DictCursor)

    def __del__(self):
        """Remove our connection from the stack when we're done."""
        _free_cached_connection(self._connection_hash)

    @property
    def connection(self):
        """Returns the psycopg2 postgresql connection instance"""
        return self._pgsql

    @property
    def cursor(self):
        """Returns the cursor instance

        :returns: psycopg2._psycopg.cursor

        """
        return self._cursor
