"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const cf = require("@aws-cdk/aws-cloudfront");
const s3 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const cdk = require("@aws-cdk/core");
const dotenv = require("dotenv");
const extensions = require("../../extensions");
const resultDotEnv = dotenv.config({ path: __dirname + '/../../../dotenv/cf-authentication-by-oauth2/.env' });
if (resultDotEnv.error) {
    throw resultDotEnv.error;
}
const app = new cdk.App();
const stack = new cdk.Stack(app, 'cf-authentication-by-oauth2-demo');
// create the cloudfront distribution with extension(s)
const OAuth2AuthorizationCodeGrant = new extensions.OAuth2AuthorizationCodeGrant(stack, 'OAuth2AuthorizationCodeGrant', {
    clientDomain: process.env.CLIENT_DOMAIN,
    clientId: process.env.CLIENT_ID,
    clientSecret: process.env.CLIENT_SECRET,
    clientPublicKey: Buffer.from(process.env.CLIENT_PUBLIC_KEY).toString('base64'),
    authorizeUrl: process.env.AUTHORIZE_URL,
    authorizeParams: Buffer.from(process.env.AUTHORIZE_PARAMS).toString('base64'),
    authorizeRedirecturiShouldMatch: process.env.AUTHORIZE_REDIRECTURI_SHOULD_MATCH,
    callbackPath: process.env.CALLBACK_PATH,
    jwtArgorithm: process.env.JWT_ARGORITHM,
    jwtTokenPath: process.env.JWT_TOKEN_PATH,
    debugEnable: process.env.DEBUG_ENABLE,
});
// create Demo S3 Bucket.
const bucket = new s3.Bucket(stack, 'demoBucket', {
    autoDeleteObjects: true,
    removalPolicy: cdk.RemovalPolicy.DESTROY,
    websiteIndexDocument: 'index.html',
    websiteErrorDocument: 'index.html',
});
// create index.html in the demo folder
fs.writeFileSync(path.join(__dirname, 'index.html'), '<h1>Hello CloudFront Extension (OAuth2 Authentication) with CDK!!!</h1><p>You have logged in. Enjoy your private content.</p>');
// Put demo Object to Bucket.
new aws_s3_deployment_1.BucketDeployment(stack, 'BucketDeployment', {
    sources: [aws_s3_deployment_1.Source.asset(path.join(__dirname, './'))],
    destinationBucket: bucket,
    retainOnDelete: false,
});
// CloudFront OriginAccessIdentity for Bucket
const originAccessIdentity = new cf.OriginAccessIdentity(stack, 'OriginAccessIdentity', {
    comment: `CloudFront OriginAccessIdentity for ${bucket.bucketName}`,
});
// CloudfrontWebDistribution
const cloudfrontWebDistribution = new cf.CloudFrontWebDistribution(stack, 'CloudFrontWebDistribution', {
    originConfigs: [
        {
            s3OriginSource: {
                originAccessIdentity,
                s3BucketSource: bucket,
            },
            behaviors: [{
                    isDefaultBehavior: true,
                    lambdaFunctionAssociations: [OAuth2AuthorizationCodeGrant],
                }],
        },
    ],
    priceClass: cf.PriceClass.PRICE_CLASS_ALL,
});
new cdk.CfnOutput(stack, 'distributionDomainName', {
    value: cloudfrontWebDistribution.distributionDomainName,
});
//# sourceMappingURL=data:application/json;base64,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