import base64
import json
from typing import Any

import dill
from rapidfireai.evals.automl.model_config import RFvLLMModelConfig, RFOpenAIAPIModelConfig


def encode_payload(payload: object) -> str:
    """Encode the payload for the database"""
    return base64.b64encode(dill.dumps(payload)).decode("utf-8")


def decode_db_payload(payload: str) -> object:
    """Decode the payload from the database"""
    return dill.loads(base64.b64decode(payload))


def extract_pipeline_config_json(pipeline_config: dict[str, Any]) -> dict[str, Any]:
    """
    Extract JSON-serializable data from a pipeline config dictionary.

    Extracts only serializable parameters (dicts, strings, ints, etc.) and ignores
    functions, classes, and other non-serializable objects. This is used for storing
    a JSON representation in the database for analytics/display purposes.

    The actual pipeline config (with functions and classes) should be stored using
    encode_payload/decode_db_payload in the pipeline_config column.

    Args:
        pipeline_config: Pipeline config dict with keys:
            - "pipeline": RFvLLMModelConfig or RFOpenAIAPIModelConfig instance
            - "batch_size": int
            - "preprocess_fn": function (skipped)
            - "postprocess_fn": function (skipped)
            - "compute_metrics_fn": function (skipped)
            - "accumulate_metrics_fn": function (skipped)
            - "online_strategy_kwargs": dict (optional)

    Returns:
        Dictionary with only JSON-serializable data from the pipeline config
    """
    json_config = {}

    # Extract batch_size if present
    if "batch_size" in pipeline_config:
        json_config["batch_size"] = pipeline_config["batch_size"]

    # Extract online_strategy_kwargs if present
    if "online_strategy_kwargs" in pipeline_config:
        json_config["online_strategy_kwargs"] = pipeline_config[
            "online_strategy_kwargs"
        ]

    # Extract pipeline type and model-specific params
    if "pipeline" in pipeline_config:
        pipeline = pipeline_config["pipeline"]

        # Helper function to extract RAG params
        def extract_rag_params(rag_spec):
            """Extract RAG parameters from rag_spec similar to controller logic."""
            if rag_spec is None:
                return None

            rag_config = {}
            rag_config["search_type"] = getattr(rag_spec, "search_type", None)

            if hasattr(rag_spec, "search_kwargs") and rag_spec.search_kwargs is not None:
                rag_config["k"] = rag_spec.search_kwargs.get("k", None)

            if hasattr(rag_spec, "reranker_kwargs") and rag_spec.reranker_kwargs is not None:
                rag_config["top_n"] = rag_spec.reranker_kwargs.get("top_n", None)

            if hasattr(rag_spec, "text_splitter") and rag_spec.text_splitter is not None:
                rag_config["chunk_size"] = getattr(rag_spec.text_splitter, "_chunk_size", None)
                rag_config["chunk_overlap"] = getattr(rag_spec.text_splitter, "_chunk_overlap", None)

            # Only return rag_config if it has at least one non-None value
            filtered_rag_config = {k: v for k, v in rag_config.items() if v is not None}
            return filtered_rag_config if filtered_rag_config else None

        if isinstance(pipeline, RFvLLMModelConfig):
            json_config["pipeline_type"] = "vllm"

            # Extract model_config (dict)
            if hasattr(pipeline, "model_config") and pipeline.model_config is not None:
                json_config["model_config"] = pipeline.model_config

            # Extract sampling_params from _user_params (original dict, not SamplingParams object)
            if hasattr(pipeline, "_user_params") and "sampling_params" in pipeline._user_params:
                json_config["sampling_params"] = pipeline._user_params["sampling_params"]

            # Extract RAG params if present
            if hasattr(pipeline, "rag") and pipeline.rag is not None:
                rag_config = extract_rag_params(pipeline.rag)
                if rag_config:
                    json_config["rag_config"] = rag_config

        elif isinstance(pipeline, RFOpenAIAPIModelConfig):
            json_config["pipeline_type"] = "openai"

            # Extract client_config (dict)
            if (
                hasattr(pipeline, "client_config")
                and pipeline.client_config is not None
            ):
                json_config["client_config"] = pipeline.client_config

            # Extract model_config (dict)
            if hasattr(pipeline, "model_config") and pipeline.model_config is not None:
                json_config["model_config"] = pipeline.model_config

            # Extract sampling_params using sampling_params_to_dict (extracts from model_config)
            if (
                hasattr(pipeline, "sampling_params")
                and pipeline.sampling_params is not None
            ):
                json_config["sampling_params"] = pipeline.sampling_params_to_dict()

            # Extract rate limiting params
            if hasattr(pipeline, "rpm_limit") and pipeline.rpm_limit is not None:
                json_config["rpm_limit"] = pipeline.rpm_limit
            if hasattr(pipeline, "tpm_limit") and pipeline.tpm_limit is not None:
                json_config["tpm_limit"] = pipeline.tpm_limit
            if (
                hasattr(pipeline, "max_completion_tokens")
                and pipeline.max_completion_tokens is not None
            ):
                json_config["max_completion_tokens"] = pipeline.max_completion_tokens

            # Extract RAG params if present
            if hasattr(pipeline, "rag") and pipeline.rag is not None:
                rag_config = extract_rag_params(pipeline.rag)
                if rag_config:
                    json_config["rag_config"] = rag_config

    # Validate JSON serializability
    try:
        json.dumps(json_config)
    except (TypeError, ValueError) as e:
        raise ValueError(f"Failed to serialize pipeline config to JSON: {e}") from e

    return json_config