//! Types for interacting with Duper's abstract syntax tree.

use std::{
    borrow::Cow,
    collections::HashMap,
    fmt::{Debug, Display},
};

use chumsky::Parser;

use crate::{
    parser::{self, DuperParser},
    visitor::DuperVisitor,
};

/// A Duper identifier: `MyIdentifier(...)`
#[derive(Debug, Clone, Hash, PartialEq, Eq)]
pub struct DuperIdentifier<'a>(pub(crate) Cow<'a, str>);

/// A Duper value.
#[derive(Debug, Clone)]
pub struct DuperValue<'a> {
    /// The identifier of this value.
    pub identifier: Option<DuperIdentifier<'a>>,
    /// The actual value contained here.
    pub inner: DuperInner<'a>,
}

/// The value contained within a [`DuperValue`].
#[derive(Debug, Clone, PartialEq)]
pub enum DuperInner<'a> {
    /// A Duper object: `{...}`
    Object(DuperObject<'a>),
    /// A Duper array: `[...]`
    Array(DuperArray<'a>),
    /// A Duper tuple: `(...)`
    Tuple(DuperTuple<'a>),
    /// A Duper string: `"..."`
    String(DuperString<'a>),
    /// A Duper bytestring: `b"..."`
    Bytes(DuperBytes<'a>),
    /// A Duper Temporal value: `'...'`, `Instant('...')`, `ZonedDateTime('...')`,
    /// `PlainDate('...')`, `PlainTime('...')`, `PlainDateTime('...')`,
    /// `PlainYearMonth('...')`, `PlainMonthDay('...')`, `Duration('...')`
    Temporal(DuperTemporal<'a>),
    /// A Duper integer.
    Integer(i64),
    /// A Duper float.
    Float(f64),
    /// A Duper boolean.
    Boolean(bool),
    /// A Duper null.
    Null,
}

/// A key in a [`DuperObject`].
#[derive(Debug, Clone, Hash, PartialEq, Eq)]
pub struct DuperKey<'a>(pub(crate) Cow<'a, str>);

/// An object (or map) from [`DuperKey`]s to [`DuperValue`]s.
#[derive(Debug, Clone)]
pub struct DuperObject<'a>(pub(crate) Vec<(DuperKey<'a>, DuperValue<'a>)>);

/// An array (or list) of [`DuperValue`]s.
#[derive(Debug, Clone, PartialEq)]
pub struct DuperArray<'a>(pub(crate) Vec<DuperValue<'a>>);

/// A tuple of [`DuperValue`]s.
#[derive(Debug, Clone, PartialEq)]
pub struct DuperTuple<'a>(pub(crate) Vec<DuperValue<'a>>);

/// A string, which may be borrowed or owned.
#[derive(Debug, Clone, Hash, PartialEq, Eq)]
pub struct DuperString<'a>(pub(crate) Cow<'a, str>);

/// A byte sequence, which may be borrowed or owned.
#[derive(Debug, Clone, Hash, PartialEq, Eq)]
pub struct DuperBytes<'a>(pub(crate) Cow<'a, [u8]>);

/// A Duper Temporal, representing one of many date-time values.
#[derive(Debug, Clone, Hash, PartialEq, Eq)]
pub enum DuperTemporal<'a> {
    /// A fixed point, or exact time, without regard to calendar or location.
    Instant(DuperTemporalInner<'a>),
    /// A timezone- and calendar-aware date/time object that represents a real event.
    ZonedDateTime(DuperTemporalInner<'a>),
    /// A calendar date that is not associated with a particular time or timezone.
    PlainDate(DuperTemporalInner<'a>),
    /// A wall-clock time that is not associated with a particular date or timezone.
    PlainTime(DuperTemporalInner<'a>),
    /// A calendar date and wall-clock time duo that does not carry any timezone information.
    PlainDateTime(DuperTemporalInner<'a>),
    /// A date without a day component.
    PlainYearMonth(DuperTemporalInner<'a>),
    /// A date without a year component.
    PlainMonthDay(DuperTemporalInner<'a>),
    /// A length of time, used for date/time arithmetic or differences between Temporal objects.
    Duration(DuperTemporalInner<'a>),
    /// A Temporal object of unspecified type.
    Unspecified(DuperTemporalInner<'a>),
}

/// The inner value of a Duper Temporal, which may be borrowed or owned.
#[derive(Debug, Clone, Hash, PartialEq, Eq)]
pub struct DuperTemporalInner<'a>(pub(crate) Cow<'a, str>);

/// Possible errors generated by [`DuperIdentifier::try_from()`].
#[derive(Debug, Clone)]
pub enum DuperIdentifierTryFromError<'a> {
    /// The identifier was empty.
    EmptyIdentifier,
    /// The identifier contained an invalid character.
    InvalidChar(Cow<'a, str>, usize),
}

/// Possible errors generated by [`DuperObject::try_from()`].
#[derive(Debug, Clone)]
pub enum DuperObjectTryFromError<'a> {
    /// The key was duplicated.
    DuplicateKey(Cow<'a, str>),
}

/// Possible errors generated by [`DuperTemporal::try_from()`].
#[derive(Debug, Clone)]
pub enum DuperTemporalTryFromError<'a> {
    /// The Temporal string was empty.
    EmptyTemporal,
    /// The Temporal value contained an invalid character.
    InvalidChar(Cow<'a, str>, usize),
}

impl<'a> DuperIdentifier<'a> {
    /// Consume this identifier and return the underlying [`Cow<'_, str>`].
    pub fn into_inner(self) -> Cow<'a, str> {
        self.0
    }

    /// Create a valid identifier from the provided [`Cow<'_, str>`], discarding
    /// any invalid characters if necessary.
    pub fn try_from_lossy(value: Cow<'a, str>) -> Result<Self, DuperIdentifierTryFromError<'a>> {
        if value.is_empty() {
            return Err(DuperIdentifierTryFromError::EmptyIdentifier);
        }
        // Eagerly try the non-lossy version first.
        if DuperIdentifier::try_from(value.as_ref()).is_ok() {
            return Ok(DuperIdentifier(value));
        }
        let invalid_char_pos = match parser::identifier_lossy()
            .parse(value.as_ref())
            .into_result()
        {
            Err(errs) => errs[0].span().start,
            Ok(identifier) => return Ok(identifier),
        };
        Err(DuperIdentifierTryFromError::InvalidChar(
            value,
            invalid_char_pos,
        ))
    }

    /// Create a clone of this DuperIdentifier with a static lifetime.
    pub fn static_clone(&self) -> DuperIdentifier<'static> {
        DuperIdentifier(Cow::Owned(self.0.clone().into_owned()))
    }
}

impl<'a> Display for DuperIdentifier<'a> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.write_str(&self.0)
    }
}

impl<'a> AsRef<str> for DuperIdentifier<'a> {
    fn as_ref(&self) -> &str {
        &self.0
    }
}

impl<'a> TryFrom<Cow<'a, str>> for DuperIdentifier<'a> {
    type Error = DuperIdentifierTryFromError<'a>;

    /// Create a valid identifier from the provided [`Cow<'_, str>`], returning
    /// an error if there are invalid characters.
    fn try_from(value: Cow<'a, str>) -> Result<Self, Self::Error> {
        if value.is_empty() {
            return Err(DuperIdentifierTryFromError::EmptyIdentifier);
        }
        let invalid_char_pos = parser::identifier()
            .parse(value.as_ref())
            .into_result()
            .err()
            .map(|errs| errs[0].span().start);
        match invalid_char_pos {
            Some(pos) => Err(DuperIdentifierTryFromError::InvalidChar(value, pos)),
            None => Ok(Self(value)),
        }
    }
}

impl<'a> TryFrom<&'a str> for DuperIdentifier<'a> {
    type Error = DuperIdentifierTryFromError<'a>;

    /// Create a valid identifier from the provided `&str`, returning
    /// an error if there are invalid characters.
    fn try_from(value: &'a str) -> Result<Self, Self::Error> {
        Self::try_from(Cow::Borrowed(value))
    }
}

impl TryFrom<String> for DuperIdentifier<'static> {
    type Error = DuperIdentifierTryFromError<'static>;

    /// Create a valid identifier from the provided [`String`], returning
    /// an error if there are invalid characters.
    fn try_from(value: String) -> Result<Self, Self::Error> {
        Self::try_from(Cow::Owned(value))
    }
}

impl Display for DuperIdentifierTryFromError<'_> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            DuperIdentifierTryFromError::EmptyIdentifier => f.write_str("empty identifier"),
            DuperIdentifierTryFromError::InvalidChar(identifier, pos) => f.write_fmt(format_args!(
                "invalid character in position {pos} of identifier {identifier}"
            )),
        }
    }
}

impl std::error::Error for DuperIdentifierTryFromError<'_> {}

impl<'a> DuperKey<'a> {
    /// Consume this key and return the underlying [`Cow<'_, str>`].
    pub fn into_inner(self) -> Cow<'a, str> {
        self.0
    }
}

impl<'a> AsRef<str> for DuperKey<'a> {
    fn as_ref(&self) -> &str {
        &self.0
    }
}

impl<'a> From<Cow<'a, str>> for DuperKey<'a> {
    fn from(value: Cow<'a, str>) -> Self {
        Self(value)
    }
}

impl<'a> From<&'a str> for DuperKey<'a> {
    fn from(value: &'a str) -> Self {
        Self(Cow::Borrowed(value))
    }
}

impl From<String> for DuperKey<'static> {
    fn from(value: String) -> Self {
        Self(Cow::Owned(value))
    }
}

impl<'a> DuperValue<'a> {
    /// Accepts a [`DuperVisitor`] and visits it with the current value.
    pub fn accept<V: DuperVisitor>(&self, visitor: &mut V) -> V::Value {
        match &self.inner {
            DuperInner::Object(object) => visitor.visit_object(self.identifier.as_ref(), object),
            DuperInner::Array(array) => visitor.visit_array(self.identifier.as_ref(), array),
            DuperInner::Tuple(tuple) => visitor.visit_tuple(self.identifier.as_ref(), tuple),
            DuperInner::String(string) => visitor.visit_string(self.identifier.as_ref(), string),
            DuperInner::Bytes(bytes) => visitor.visit_bytes(self.identifier.as_ref(), bytes),
            DuperInner::Temporal(temporal) => {
                visitor.visit_temporal(self.identifier.as_ref(), temporal)
            }
            DuperInner::Integer(integer) => {
                visitor.visit_integer(self.identifier.as_ref(), *integer)
            }
            DuperInner::Float(float) => visitor.visit_float(self.identifier.as_ref(), *float),
            DuperInner::Boolean(boolean) => {
                visitor.visit_boolean(self.identifier.as_ref(), *boolean)
            }
            DuperInner::Null => visitor.visit_null(self.identifier.as_ref()),
        }
    }
}

impl<'a> TryFrom<&'a str> for DuperValue<'a> {
    type Error = Vec<chumsky::error::Rich<'a, char>>;

    fn try_from(value: &'a str) -> Result<Self, Self::Error> {
        DuperParser::parse_duper_value(value)
    }
}

impl<'a> PartialEq for DuperValue<'a> {
    fn eq(&self, other: &Self) -> bool {
        self.inner == other.inner
    }
}

impl<'a> DuperObject<'a> {
    /// Consume this object and return the underlying [`Vec<(DuperKey<'_>, DuperValue<'_>)>`].
    pub fn into_inner(self) -> Vec<(DuperKey<'a>, DuperValue<'a>)> {
        self.0
    }

    /// Returns `true` if the object contains no elements.
    pub fn is_empty(&self) -> bool {
        self.0.is_empty()
    }

    /// Returns the amount of elements in this object.
    pub fn len(&self) -> usize {
        self.0.len()
    }

    /// Returns an iterator over references to the (key, value) pairs in this
    /// object.
    pub fn iter(&self) -> impl Iterator<Item = &(DuperKey<'a>, DuperValue<'a>)> {
        self.0.iter()
    }
}

impl<'a> TryFrom<Vec<(DuperKey<'a>, DuperValue<'a>)>> for DuperObject<'a> {
    type Error = DuperObjectTryFromError<'a>;

    /// Create a valid object from the provided [`Vec`], returning an error if
    /// a duplicate key is found.
    fn try_from(value: Vec<(DuperKey<'a>, DuperValue<'a>)>) -> Result<Self, Self::Error> {
        let mut keys = std::collections::HashSet::with_capacity(value.len());
        for (key, _) in value.iter() {
            if keys.contains(key) {
                return Err(DuperObjectTryFromError::DuplicateKey(key.0.clone()));
            }
            keys.insert(key);
        }
        Ok(Self(value))
    }
}

impl<'a> PartialEq for DuperObject<'a> {
    fn eq(&self, other: &Self) -> bool {
        if self.0.len() != other.0.len() {
            return false;
        }
        let other_map: HashMap<_, _> = other.0.iter().map(|(k, v)| (k, v)).collect();
        for (k, v) in self.0.iter() {
            match other_map.get(k) {
                Some(v2) => {
                    if v != *v2 {
                        return false;
                    }
                }
                None => return false,
            }
        }
        true
    }
}

impl Display for DuperObjectTryFromError<'_> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            DuperObjectTryFromError::DuplicateKey(key) => {
                f.write_fmt(format_args!("duplicate key {key} in object"))
            }
        }
    }
}

impl std::error::Error for DuperObjectTryFromError<'_> {}

impl<'a> DuperArray<'a> {
    /// Consume this array and return the underlying [`Vec<DuperValue<'_>>`].
    pub fn into_inner(self) -> Vec<DuperValue<'a>> {
        self.0
    }

    /// Returns `true` if the array contains no elements.
    pub fn is_empty(&self) -> bool {
        self.0.is_empty()
    }

    /// Returns the amount of elements in this array.
    pub fn len(&self) -> usize {
        self.0.len()
    }

    /// Returns an iterator over references to the values in this array.
    pub fn iter(&self) -> impl Iterator<Item = &DuperValue<'a>> {
        self.0.iter()
    }

    /// Returns a reference to the value at the given position.
    pub fn get(&self, index: usize) -> Option<&DuperValue<'a>> {
        self.0.get(index)
    }
}

impl<'a> From<Vec<DuperValue<'a>>> for DuperArray<'a> {
    fn from(value: Vec<DuperValue<'a>>) -> Self {
        Self(value)
    }
}

impl<'a> DuperTuple<'a> {
    /// Consume this tuple and return the underlying [`Vec<DuperValue<'_>>`].
    pub fn into_inner(self) -> Vec<DuperValue<'a>> {
        self.0
    }

    /// Returns `true` if the tuple contains no elements.
    pub fn is_empty(&self) -> bool {
        self.0.is_empty()
    }

    /// Returns the amount of elements in this tuple.
    pub fn len(&self) -> usize {
        self.0.len()
    }

    /// Returns an iterator over references to the values in this tuple.
    pub fn iter(&self) -> impl Iterator<Item = &DuperValue<'a>> {
        self.0.iter()
    }

    /// Returns a reference to the value at the given position.
    pub fn get(&self, index: usize) -> Option<&DuperValue<'a>> {
        self.0.get(index)
    }
}

impl<'a> From<Vec<DuperValue<'a>>> for DuperTuple<'a> {
    fn from(value: Vec<DuperValue<'a>>) -> Self {
        Self(value)
    }
}

impl<'a> DuperString<'a> {
    /// Consume this string and return the underlying [`Cow<'_, str>`].
    pub fn into_inner(self) -> Cow<'a, str> {
        self.0
    }

    /// Returns `true` if the string are empty.
    pub fn is_empty(&self) -> bool {
        self.0.is_empty()
    }

    /// Returns the amount of bytes in this value's slice.
    pub fn len(&self) -> usize {
        self.0.len()
    }
}

impl<'a> From<Cow<'a, str>> for DuperString<'a> {
    fn from(value: Cow<'a, str>) -> Self {
        Self(value)
    }
}

impl<'a> From<&'a str> for DuperString<'a> {
    fn from(value: &'a str) -> Self {
        Self(Cow::Borrowed(value))
    }
}

impl From<String> for DuperString<'static> {
    fn from(value: String) -> Self {
        Self(Cow::Owned(value))
    }
}

impl<'a> AsRef<str> for DuperString<'a> {
    fn as_ref(&self) -> &str {
        &self.0
    }
}

impl<'a> DuperBytes<'a> {
    /// Consume these bytes and return the underlying [`Cow<'_, [u8]>`].
    pub fn into_inner(self) -> Cow<'a, [u8]> {
        self.0
    }

    /// Returns `true` if the bytes are empty.
    pub fn is_empty(&self) -> bool {
        self.0.is_empty()
    }

    /// Returns the amount of bytes in this value's slice.
    pub fn len(&self) -> usize {
        self.0.len()
    }
}

impl<'a> From<Cow<'a, [u8]>> for DuperBytes<'a> {
    fn from(value: Cow<'a, [u8]>) -> Self {
        Self(value)
    }
}

impl<'a> From<&'a [u8]> for DuperBytes<'a> {
    fn from(value: &'a [u8]) -> Self {
        Self(Cow::Borrowed(value))
    }
}

impl From<Vec<u8>> for DuperBytes<'static> {
    fn from(value: Vec<u8>) -> Self {
        Self(Cow::Owned(value))
    }
}

impl<'a> AsRef<[u8]> for DuperBytes<'a> {
    fn as_ref(&self) -> &[u8] {
        &self.0
    }
}

impl<'a> DuperTemporal<'a> {
    /// Consume this Temporal value and return the underlying [`Cow<'_, str>`].
    pub fn into_inner(self) -> Cow<'a, str> {
        match self {
            DuperTemporal::Instant(inner)
            | DuperTemporal::ZonedDateTime(inner)
            | DuperTemporal::PlainDate(inner)
            | DuperTemporal::PlainTime(inner)
            | DuperTemporal::PlainDateTime(inner)
            | DuperTemporal::PlainYearMonth(inner)
            | DuperTemporal::PlainMonthDay(inner)
            | DuperTemporal::Duration(inner)
            | DuperTemporal::Unspecified(inner) => inner.0,
        }
    }

    pub fn name(&self) -> &'static str {
        match self {
            DuperTemporal::Instant(_) => "Instant",
            DuperTemporal::ZonedDateTime(_) => "ZonedDateTime",
            DuperTemporal::PlainDate(_) => "PlainDate",
            DuperTemporal::PlainTime(_) => "PlainTime",
            DuperTemporal::PlainDateTime(_) => "PlainDateTime",
            DuperTemporal::PlainYearMonth(_) => "PlainYearMonth",
            DuperTemporal::PlainMonthDay(_) => "PlainMonthDay",
            DuperTemporal::Duration(_) => "Duration",
            DuperTemporal::Unspecified(_) => "Unspecified",
        }
    }

    /// Create a clone of this DuperTemporal with a static lifetime.
    pub fn static_clone(&self) -> DuperTemporal<'static> {
        match self {
            DuperTemporal::Instant(inner) => {
                DuperTemporal::Instant(DuperTemporalInner(Cow::Owned(inner.0.clone().into_owned())))
            }
            DuperTemporal::ZonedDateTime(inner) => DuperTemporal::ZonedDateTime(
                DuperTemporalInner(Cow::Owned(inner.0.clone().into_owned())),
            ),
            DuperTemporal::PlainDate(inner) => DuperTemporal::PlainDate(DuperTemporalInner(
                Cow::Owned(inner.0.clone().into_owned()),
            )),
            DuperTemporal::PlainTime(inner) => DuperTemporal::PlainTime(DuperTemporalInner(
                Cow::Owned(inner.0.clone().into_owned()),
            )),
            DuperTemporal::PlainDateTime(inner) => DuperTemporal::PlainDateTime(
                DuperTemporalInner(Cow::Owned(inner.0.clone().into_owned())),
            ),
            DuperTemporal::PlainYearMonth(inner) => DuperTemporal::PlainYearMonth(
                DuperTemporalInner(Cow::Owned(inner.0.clone().into_owned())),
            ),
            DuperTemporal::PlainMonthDay(inner) => DuperTemporal::PlainMonthDay(
                DuperTemporalInner(Cow::Owned(inner.0.clone().into_owned())),
            ),
            DuperTemporal::Duration(inner) => DuperTemporal::Duration(DuperTemporalInner(
                Cow::Owned(inner.0.clone().into_owned()),
            )),
            DuperTemporal::Unspecified(inner) => DuperTemporal::Unspecified(DuperTemporalInner(
                Cow::Owned(inner.0.clone().into_owned()),
            )),
        }
    }

    /// Create a valid Temporal Instant from the provided [`Cow<'_, str>`],
    /// returning an error if parsing fails.
    ///
    /// The provided string must not contain an identifier, single quotes,
    /// or whitespace.
    pub fn try_instant_from(value: Cow<'a, str>) -> Result<Self, DuperTemporalTryFromError<'a>> {
        if value.is_empty() {
            return Err(DuperTemporalTryFromError::EmptyTemporal);
        }
        let result = match parser::temporal::instant()
            .parse(value.as_ref())
            .into_result()
        {
            Ok(_) => Ok(()),
            Err(errs) => Err(errs[0].span().start),
        };

        match result {
            Ok(_) => Ok(DuperTemporal::Instant(DuperTemporalInner(value))),
            Err(invalid_char_pos) => Err(DuperTemporalTryFromError::InvalidChar(
                value,
                invalid_char_pos,
            )),
        }
    }

    /// Create a valid Temporal ZonedDateTime from the provided [`Cow<'_, str>`],
    /// returning an error if parsing fails.
    ///
    /// The provided string must not contain an identifier, single quotes,
    /// or whitespace.
    pub fn try_zoned_date_time_from(
        value: Cow<'a, str>,
    ) -> Result<Self, DuperTemporalTryFromError<'a>> {
        if value.is_empty() {
            return Err(DuperTemporalTryFromError::EmptyTemporal);
        }
        let result = match parser::temporal::zoned_date_time()
            .parse(value.as_ref())
            .into_result()
        {
            Ok(_) => Ok(()),
            Err(errs) => Err(errs[0].span().start),
        };

        match result {
            Ok(_) => Ok(DuperTemporal::ZonedDateTime(DuperTemporalInner(value))),
            Err(invalid_char_pos) => Err(DuperTemporalTryFromError::InvalidChar(
                value,
                invalid_char_pos,
            )),
        }
    }

    /// Create a valid Temporal PlainDate from the provided [`Cow<'_, str>`],
    /// returning an error if parsing fails.
    ///
    /// The provided string must not contain an identifier, single quotes,
    /// or whitespace.
    pub fn try_plain_date_from(value: Cow<'a, str>) -> Result<Self, DuperTemporalTryFromError<'a>> {
        if value.is_empty() {
            return Err(DuperTemporalTryFromError::EmptyTemporal);
        }
        let result = match parser::temporal::plain_date()
            .parse(value.as_ref())
            .into_result()
        {
            Ok(_) => Ok(()),
            Err(errs) => Err(errs[0].span().start),
        };

        match result {
            Ok(_) => Ok(DuperTemporal::PlainDate(DuperTemporalInner(value))),
            Err(invalid_char_pos) => Err(DuperTemporalTryFromError::InvalidChar(
                value,
                invalid_char_pos,
            )),
        }
    }

    /// Create a valid Temporal PlainTime from the provided [`Cow<'_, str>`],
    /// returning an error if parsing fails.
    ///
    /// The provided string must not contain an identifier, single quotes,
    /// or whitespace.
    pub fn try_plain_time_from(value: Cow<'a, str>) -> Result<Self, DuperTemporalTryFromError<'a>> {
        if value.is_empty() {
            return Err(DuperTemporalTryFromError::EmptyTemporal);
        }
        let result = match parser::temporal::plain_time()
            .parse(value.as_ref())
            .into_result()
        {
            Ok(_) => Ok(()),
            Err(errs) => Err(errs[0].span().start),
        };

        match result {
            Ok(_) => Ok(DuperTemporal::PlainTime(DuperTemporalInner(value))),
            Err(invalid_char_pos) => Err(DuperTemporalTryFromError::InvalidChar(
                value,
                invalid_char_pos,
            )),
        }
    }

    /// Create a valid Temporal PlainDateTime from the provided [`Cow<'_, str>`],
    /// returning an error if parsing fails.
    ///
    /// The provided string must not contain an identifier, single quotes,
    /// or whitespace.
    pub fn try_plain_date_time_from(
        value: Cow<'a, str>,
    ) -> Result<Self, DuperTemporalTryFromError<'a>> {
        if value.is_empty() {
            return Err(DuperTemporalTryFromError::EmptyTemporal);
        }
        let result = match parser::temporal::plain_date_time()
            .parse(value.as_ref())
            .into_result()
        {
            Ok(_) => Ok(()),
            Err(errs) => Err(errs[0].span().start),
        };

        match result {
            Ok(_) => Ok(DuperTemporal::PlainDateTime(DuperTemporalInner(value))),
            Err(invalid_char_pos) => Err(DuperTemporalTryFromError::InvalidChar(
                value,
                invalid_char_pos,
            )),
        }
    }

    /// Create a valid Temporal PlainYearMonth from the provided [`Cow<'_, str>`],
    /// returning an error if parsing fails.
    ///
    /// The provided string must not contain an identifier, single quotes,
    /// or whitespace.
    pub fn try_plain_year_month_from(
        value: Cow<'a, str>,
    ) -> Result<Self, DuperTemporalTryFromError<'a>> {
        if value.is_empty() {
            return Err(DuperTemporalTryFromError::EmptyTemporal);
        }
        let result = match parser::temporal::plain_year_month()
            .parse(value.as_ref())
            .into_result()
        {
            Ok(_) => Ok(()),
            Err(errs) => Err(errs[0].span().start),
        };

        match result {
            Ok(_) => Ok(DuperTemporal::PlainYearMonth(DuperTemporalInner(value))),
            Err(invalid_char_pos) => Err(DuperTemporalTryFromError::InvalidChar(
                value,
                invalid_char_pos,
            )),
        }
    }

    /// Create a valid Temporal PlainMonthDay from the provided [`Cow<'_, str>`],
    /// returning an error if parsing fails.
    ///
    /// The provided string must not contain an identifier, single quotes,
    /// or whitespace.
    pub fn try_plain_month_day_from(
        value: Cow<'a, str>,
    ) -> Result<Self, DuperTemporalTryFromError<'a>> {
        if value.is_empty() {
            return Err(DuperTemporalTryFromError::EmptyTemporal);
        }
        let result = match parser::temporal::plain_month_day()
            .parse(value.as_ref())
            .into_result()
        {
            Ok(_) => Ok(()),
            Err(errs) => Err(errs[0].span().start),
        };

        match result {
            Ok(_) => Ok(DuperTemporal::PlainMonthDay(DuperTemporalInner(value))),
            Err(invalid_char_pos) => Err(DuperTemporalTryFromError::InvalidChar(
                value,
                invalid_char_pos,
            )),
        }
    }

    /// Create a valid Temporal Duration from the provided [`Cow<'_, str>`],
    /// returning an error if parsing fails.
    ///
    /// The provided string must not contain an identifier, single quotes,
    /// or whitespace.
    pub fn try_duration_from(value: Cow<'a, str>) -> Result<Self, DuperTemporalTryFromError<'a>> {
        if value.is_empty() {
            return Err(DuperTemporalTryFromError::EmptyTemporal);
        }
        let result = match parser::temporal::duration()
            .parse(value.as_ref())
            .into_result()
        {
            Ok(_) => Ok(()),
            Err(errs) => Err(errs[0].span().start),
        };

        match result {
            Ok(_) => Ok(DuperTemporal::Duration(DuperTemporalInner(value))),
            Err(invalid_char_pos) => Err(DuperTemporalTryFromError::InvalidChar(
                value,
                invalid_char_pos,
            )),
        }
    }

    /// Create a valid unspecified Temporal value from the provided [`Cow<'_, str>`],
    /// returning an error if parsing fails.
    ///
    /// The provided string must not contain an identifier, single quotes,
    /// or whitespace.
    pub fn try_unspecified_from(
        value: Cow<'a, str>,
    ) -> Result<Self, DuperTemporalTryFromError<'a>> {
        if value.is_empty() {
            return Err(DuperTemporalTryFromError::EmptyTemporal);
        }
        let result = match parser::temporal::unspecified()
            .parse(value.as_ref())
            .into_result()
        {
            Ok(_) => Ok(()),
            Err(errs) => Err(errs[0].span().start),
        };

        match result {
            Ok(_) => Ok(DuperTemporal::Unspecified(DuperTemporalInner(value))),
            Err(invalid_char_pos) => Err(DuperTemporalTryFromError::InvalidChar(
                value,
                invalid_char_pos,
            )),
        }
    }
}

impl<'a> AsRef<str> for DuperTemporal<'a> {
    fn as_ref(&self) -> &str {
        match self {
            DuperTemporal::Instant(inner)
            | DuperTemporal::ZonedDateTime(inner)
            | DuperTemporal::PlainDate(inner)
            | DuperTemporal::PlainTime(inner)
            | DuperTemporal::PlainDateTime(inner)
            | DuperTemporal::PlainYearMonth(inner)
            | DuperTemporal::PlainMonthDay(inner)
            | DuperTemporal::Duration(inner)
            | DuperTemporal::Unspecified(inner) => inner.as_ref(),
        }
    }
}

impl<'a> DuperTemporalInner<'a> {
    /// Consume this inner Temporal value and return the underlying [`Cow<'_, str>`].
    pub fn into_inner(self) -> Cow<'a, str> {
        self.0
    }
}

impl<'a> AsRef<str> for DuperTemporalInner<'a> {
    fn as_ref(&self) -> &str {
        &self.0
    }
}

impl Display for DuperTemporalTryFromError<'_> {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        match self {
            DuperTemporalTryFromError::EmptyTemporal => f.write_str("empty Temporal value"),
            DuperTemporalTryFromError::InvalidChar(temporal, pos) => f.write_fmt(format_args!(
                "invalid character in position {pos} of Temporal value {temporal}"
            )),
        }
    }
}

impl std::error::Error for DuperTemporalTryFromError<'_> {}

#[cfg(test)]
mod ast_tests {
    use std::borrow::Cow;

    use crate::DuperIdentifier;

    #[test]
    fn valid_identifiers() {
        let input = "Regular";
        assert!(
            matches!(DuperIdentifier::try_from(Cow::Borrowed(input)).unwrap(),
                DuperIdentifier(Cow::Borrowed(str)) if str == input
            )
        );
        assert!(matches!(
            DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).unwrap(),
            DuperIdentifier(Cow::Borrowed(str)) if str == input
        ));

        let input = "ConcatWords";
        assert!(
            matches!(DuperIdentifier::try_from(Cow::Borrowed(input)).unwrap(),
                DuperIdentifier(Cow::Borrowed(str)) if str == input
            )
        );
        assert!(matches!(
            DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).unwrap(),
            DuperIdentifier(Cow::Borrowed(str)) if str == input
        ));

        let input = "SCREAMING";
        assert!(
            matches!(DuperIdentifier::try_from(Cow::Borrowed(input)).unwrap(),
                DuperIdentifier(Cow::Borrowed(str)) if str == input
            )
        );
        assert!(matches!(
            DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).unwrap(),
            DuperIdentifier(Cow::Borrowed(str)) if str == input
        ));

        let input = "Numbered123";
        assert!(
            matches!(DuperIdentifier::try_from(Cow::Borrowed(input)).unwrap(),
                DuperIdentifier(Cow::Borrowed(str)) if str == input
            )
        );
        assert!(matches!(
            DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).unwrap(),
            DuperIdentifier(Cow::Borrowed(str)) if str == input
        ));

        let input = "Upper_Snake_case";
        assert!(
            matches!(DuperIdentifier::try_from(Cow::Borrowed(input)).unwrap(),
                DuperIdentifier(Cow::Borrowed(str)) if str == input
            )
        );
        assert!(matches!(
            DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).unwrap(),
            DuperIdentifier(Cow::Borrowed(str)) if str == input
        ));

        let input = "Upper-Kebab-case";
        assert!(
            matches!(DuperIdentifier::try_from(Cow::Borrowed(input)).unwrap(),
                DuperIdentifier(Cow::Borrowed(str)) if str == input
            )
        );
        assert!(matches!(
            DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).unwrap(),
            DuperIdentifier(Cow::Borrowed(str)) if str == input
        ));

        let input = "IPv4Address";
        assert!(
            matches!(DuperIdentifier::try_from(Cow::Borrowed(input)).unwrap(),
                DuperIdentifier(Cow::Borrowed(str)) if str == input
            )
        );
        assert!(matches!(
            DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).unwrap(),
            DuperIdentifier(Cow::Borrowed(str)) if str == input
        ));

        let input = "ISO639-3";
        assert!(
            matches!(DuperIdentifier::try_from(Cow::Borrowed(input)).unwrap(),
                DuperIdentifier(Cow::Borrowed(str)) if str == input
            )
        );
        assert!(matches!(
            DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).unwrap(),
            DuperIdentifier(Cow::Borrowed(str)) if str == input
        ));
    }

    #[test]
    fn lossy_identifiers() {
        let input = "NoÜnicodeÇharacters";
        assert!(DuperIdentifier::try_from(Cow::Borrowed(input)).is_err());
        assert!(matches!(
            DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).unwrap(),
            DuperIdentifier(Cow::Owned(str)) if str == "Nonicodeharacters"
        ));

        let input = "noStartingLowercase";
        assert!(DuperIdentifier::try_from(Cow::Borrowed(input)).is_err());
        assert!(matches!(
            DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).unwrap(),
            DuperIdentifier(Cow::Owned(str)) if str == "NoStartingLowercase"
        ));

        let input = "No Space";
        assert!(DuperIdentifier::try_from(Cow::Borrowed(input)).is_err());
        assert!(matches!(
            DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).unwrap(),
            DuperIdentifier(Cow::Owned(str)) if str == "NoSpace"
        ));

        let input = "NoEndingHyphen-";
        assert!(DuperIdentifier::try_from(Cow::Borrowed(input)).is_err());
        assert!(matches!(
            DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).unwrap(),
            DuperIdentifier(Cow::Owned(str)) if str == "NoEndingHyphen"
        ));

        let input = "NoEndingUnderscore_";
        assert!(DuperIdentifier::try_from(Cow::Borrowed(input)).is_err());
        assert!(matches!(
            DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).unwrap(),
            DuperIdentifier(Cow::Owned(str)) if str == "NoEndingUnderscore"
        ));

        let input = "No-_HyphenThenUnderscore";
        assert!(DuperIdentifier::try_from(Cow::Borrowed(input)).is_err());
        assert!(matches!(
            DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).unwrap(),
            DuperIdentifier(Cow::Owned(str)) if str == "No-HyphenThenUnderscore"
        ));

        let input = "No_-UnderscoreThenHyphen";
        assert!(DuperIdentifier::try_from(Cow::Borrowed(input)).is_err());
        assert!(matches!(
            DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).unwrap(),
            DuperIdentifier(Cow::Owned(str)) if str == "No_UnderscoreThenHyphen"
        ));
    }

    #[test]
    fn invalid_identifiers() {
        let input = "";
        assert!(DuperIdentifier::try_from(Cow::Borrowed(input)).is_err());
        assert!(DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).is_err());

        let input = "1NoStartingNumber";
        assert!(DuperIdentifier::try_from(Cow::Borrowed(input)).is_err());
        assert!(DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).is_err());

        let input = "-NoStartingHyphen";
        assert!(DuperIdentifier::try_from(Cow::Borrowed(input)).is_err());
        assert!(DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).is_err());

        let input = "_NoStartingUnderscore";
        assert!(DuperIdentifier::try_from(Cow::Borrowed(input)).is_err());
        assert!(DuperIdentifier::try_from_lossy(Cow::Borrowed(input)).is_err());
    }
}
