from __future__ import annotations

from typing import Tuple

import numpy as np
import pandas as pd


def _ensure_dataframe(x: pd.Series | pd.DataFrame) -> pd.DataFrame:
    if isinstance(x, pd.Series):
        return x.to_frame()
    return x


def compute_cagr(
    prices: pd.Series | pd.DataFrame,
    periods_per_year: int = 252,
) -> pd.Series:
    """Compute CAGR for each column of a price/equity DataFrame.

    Returns a Series indexed by column name with CAGR as a float in [−1, inf).
    """
    df = _ensure_dataframe(prices).dropna(how="all")
    if df.shape[0] < 2:
        return pd.Series({c: np.nan for c in df.columns})

    def _cagr(col: pd.Series) -> float:
        col = col.dropna()
        if len(col) < 2:
            return np.nan
        start, end = float(col.iloc[0]), float(col.iloc[-1])
        years = (len(col) - 1) / periods_per_year
        if years <= 0 or start <= 0:
            return np.nan
        return (end / start) ** (1.0 / years) - 1.0

    return df.apply(_cagr, axis=0)


def compute_max_drawdown(prices: pd.Series | pd.DataFrame) -> pd.Series:
    """Compute max drawdown magnitude for each column of a price/equity DataFrame.

    Returns positive magnitudes (e.g., 0.35 means -35% at worst).
    """
    df = _ensure_dataframe(prices).dropna(how="all")
    if df.shape[0] < 1:
        return pd.Series({c: np.nan for c in df.columns})

    def _mdd(col: pd.Series) -> float:
        col = col.dropna().astype(float)
        if len(col) == 0:
            return np.nan
        running_max = col.cummax()
        drawdown = col / running_max - 1.0
        return float((-drawdown.min()))  # positive magnitude

    return df.apply(_mdd, axis=0)


def compute_metrics(
    prices: pd.DataFrame,
    periods_per_year: int = 252,
) -> pd.DataFrame:
    """Compute core metrics from a close-price/equity DataFrame.

    Input: wide DataFrame with index as datetime and columns as symbols.
    Output: DataFrame with index as symbols and columns [cagr, max_drawdown].
    """
    cagr = compute_cagr(prices, periods_per_year)
    mdd = compute_max_drawdown(prices)
    out = pd.DataFrame({"cagr": cagr, "max_drawdown": mdd})
    out.index.name = "symbol"
    return out

