from __future__ import annotations

import pandas as pd


def calc_returns(prices: pd.DataFrame) -> pd.DataFrame:
    """Compute simple returns from price DataFrame.

    Assumes prices indexed by date, columns are symbols.
    """
    returns = prices.sort_index().astype(float).pct_change()
    returns = returns.fillna(0.0)
    return returns


def equity_curve_from_weights(prices: pd.DataFrame, weights: pd.DataFrame, start_value: float = 1.0) -> pd.Series:
    """Compute portfolio equity curve using previous-day weights (no lookahead).

    - `prices`: wide DataFrame of close prices
    - `weights`: wide DataFrame of target weights per day and symbol
    Returns a Series named 'PORT' with the equity curve.
    """
    rets = calc_returns(prices)
    w = (
        weights.reindex(index=rets.index, columns=rets.columns, fill_value=0.0)
        .astype(float)
        .shift(1)
        .fillna(0.0)
    )
    port_rets = (w * rets).sum(axis=1)
    equity = (1.0 + port_rets).cumprod() * float(start_value)
    equity.name = "PORT"
    return equity

