"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IpamAllocation = exports.IpamPool = exports.IpamScope = exports.Ipam = exports.IpamPoolAddressFamily = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
var IpamPoolAddressFamily;
(function (IpamPoolAddressFamily) {
    IpamPoolAddressFamily["IPV4"] = "ipv4";
    IpamPoolAddressFamily["IPV6"] = "ipv6";
})(IpamPoolAddressFamily = exports.IpamPoolAddressFamily || (exports.IpamPoolAddressFamily = {}));
/**
 * Creates an IPAM.
 *
 * PAM is a VPC feature that you can use to automate your IP address management workflows including
 * assigning, tracking, troubleshooting, and auditing IP addresses across AWS Regions and accounts
 * throughout your AWS Organization. For more information, see What is IPAM? in the Amazon VPC IPAM
 * User Guide.
 *
 * @see https://docs.aws.amazon.com/vpc/latest/ipam/what-is-it-ipam.html
 */
class Ipam extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        let operatingRegions = [];
        if (props.operatingRegions) {
            for (const region of props.operatingRegions) {
                operatingRegions.push({
                    regionName: region,
                });
            }
        }
        else {
            operatingRegions.push({
                regionName: aws_cdk_lib_1.Stack.of(this).region,
            });
        }
        this.ipam = new aws_cdk_lib_1.aws_ec2.CfnIPAM(this, 'IPAM', {
            description: props.description,
            operatingRegions,
            tags: props.tags,
        });
        this.ipamId = this.ipam.ref;
        this.privateDefaultScopeId = this.ipam.attrPrivateDefaultScopeId;
        this.publicDefaultScopeId = this.ipam.attrPublicDefaultScopeId;
        this.scopeCount = this.ipam.attrScopeCount;
    }
}
exports.Ipam = Ipam;
_a = JSII_RTTI_SYMBOL_1;
Ipam[_a] = { fqn: "@renovosolutions/cdk-library-aws-ipam.Ipam", version: "0.1.209" };
/**
 * An IPAM Scope
 *
 * In IPAM, a scope is the highest-level container within IPAM. An IPAM contains two default scopes.
 * Each scope represents the IP space for a single network. The private scope is intended for all private
 * IP address space. The public scope is intended for all public IP address space. Scopes enable you to
 * reuse IP addresses across multiple unconnected networks without causing IP address overlap or conflict.
 */
class IpamScope extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.scope = new aws_cdk_lib_1.aws_ec2.CfnIPAMScope(this, 'Scope', {
            description: props.description,
            ipamId: props.ipam.ipamId,
            tags: props.tags,
        });
        this.arn = this.scope.attrArn;
        this.ipamArn = this.scope.attrIpamArn;
        this.ipamScopeId = this.scope.attrIpamScopeId;
        this.isDefault = this.scope.attrIsDefault;
        this.poolCount = this.scope.attrPoolCount;
    }
}
exports.IpamScope = IpamScope;
_b = JSII_RTTI_SYMBOL_1;
IpamScope[_b] = { fqn: "@renovosolutions/cdk-library-aws-ipam.IpamScope", version: "0.1.209" };
/**
 * An IPAM Pool
 *
 * In IPAM, a pool is a collection of contiguous IP addresses CIDRs. Pools enable you to organize your IP addresses
 * according to your routing and security needs. For example, if you have separate routing and security needs for
 * development and production applications, you can create a pool for each.
 */
class IpamPool extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.provisionedCidrs = props.provisionedCidrs ?? [];
        this.pool = new aws_cdk_lib_1.aws_ec2.CfnIPAMPool(this, 'Scope', {
            addressFamily: props.addressFamily,
            ipamScopeId: props.ipamScopeId,
            description: props.description,
            allocationDefaultNetmaskLength: props.allocationDefaultNetmaskLength,
            allocationMaxNetmaskLength: props.allocationMaxNetmaskLength,
            allocationMinNetmaskLength: props.allocationMinNetmaskLength,
            allocationResourceTags: props.allocationResourceTags,
            autoImport: props.autoImport,
            locale: props.locale,
            sourceIpamPoolId: props.sourceIpamPoolId,
            tags: props.tags,
            provisionedCidrs: this.provisionedCidrs,
        });
        this.arn = this.pool.attrArn;
        this.ipamArn = this.pool.attrIpamArn;
        this.ipamPoolId = this.pool.attrIpamPoolId;
        this.ipamScopeArn = this.pool.attrIpamScopeArn;
        this.ipamScopeType = this.pool.attrIpamScopeType;
        this.poolDepth = this.pool.attrPoolDepth;
        this.state = this.pool.attrState;
        this.stateMessage = this.pool.attrStateMessage;
        this.allocationDefaultNetmaskLength = props.allocationDefaultNetmaskLength;
    }
    /**
     * Adds a CIDR to the pool
     *
     * @param cidr The CIDR to add to the pool
     */
    provisionCidr(cidr) {
        this.provisionedCidrs.push({ cidr });
        this.pool.addPropertyOverride('ProvisionedCidrs', {
            cidr,
        });
    }
}
exports.IpamPool = IpamPool;
_c = JSII_RTTI_SYMBOL_1;
IpamPool[_c] = { fqn: "@renovosolutions/cdk-library-aws-ipam.IpamPool", version: "0.1.209" };
/**
 * An IPAM Allocation
 *
 * In IPAM, an allocation is a CIDR assignment from an IPAM pool to another resource or IPAM pool.
 */
class IpamAllocation extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.cidr == undefined && props.netmaskLength == undefined && props.ipamPool.allocationDefaultNetmaskLength == undefined) {
            throw new Error('Either CIDR or NetmaskLength must be specified when no default netmask length is set on the pool');
        }
        else {
            this.allocation = new aws_cdk_lib_1.aws_ec2.CfnIPAMAllocation(this, 'Allocation', {
                ipamPoolId: props.ipamPool.ipamPoolId,
                cidr: props.cidr,
                description: props.description,
                netmaskLength: props.netmaskLength,
            });
            this.ipamPoolAllocationId = this.allocation.attrIpamPoolAllocationId;
        }
    }
}
exports.IpamAllocation = IpamAllocation;
_d = JSII_RTTI_SYMBOL_1;
IpamAllocation[_d] = { fqn: "@renovosolutions/cdk-library-aws-ipam.IpamAllocation", version: "0.1.209" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw2Q0FLcUI7QUFDckIsMkNBQXVDO0FBRXZDLElBQVkscUJBR1g7QUFIRCxXQUFZLHFCQUFxQjtJQUMvQixzQ0FBYSxDQUFBO0lBQ2Isc0NBQWEsQ0FBQTtBQUNmLENBQUMsRUFIVyxxQkFBcUIsR0FBckIsNkJBQXFCLEtBQXJCLDZCQUFxQixRQUdoQztBQTRCRDs7Ozs7Ozs7O0dBU0c7QUFDSCxNQUFhLElBQUssU0FBUSxzQkFBUztJQXVCakMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFnQjtRQUN4RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksZ0JBQWdCLEdBQTZDLEVBQUUsQ0FBQztRQUNwRSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUMxQixLQUFLLE1BQU0sTUFBTSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtnQkFDM0MsZ0JBQWdCLENBQUMsSUFBSSxDQUFDO29CQUNwQixVQUFVLEVBQUUsTUFBTTtpQkFDbkIsQ0FBQyxDQUFDO2FBQ0o7U0FDRjthQUFNO1lBQ0wsZ0JBQWdCLENBQUMsSUFBSSxDQUFDO2dCQUNwQixVQUFVLEVBQUUsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTTthQUNsQyxDQUFDLENBQUM7U0FDSjtRQUVELElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxxQkFBRyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFO1lBQ3hDLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztZQUM5QixnQkFBZ0I7WUFDaEIsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJO1NBQ2pCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxNQUFNLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUM7UUFDNUIsSUFBSSxDQUFDLHFCQUFxQixHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMseUJBQXlCLENBQUM7UUFDakUsSUFBSSxDQUFDLG9CQUFvQixHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsd0JBQXdCLENBQUM7UUFDL0QsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQztJQUM3QyxDQUFDOztBQWpESCxvQkFrREM7OztBQW9CRDs7Ozs7OztHQU9HO0FBQ0gsTUFBYSxTQUFVLFNBQVEsc0JBQVM7SUE0QnRDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBcUI7UUFDN0QsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsS0FBSyxHQUFHLElBQUkscUJBQUcsQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLE9BQU8sRUFBRTtZQUMvQyxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsTUFBTSxFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsTUFBTTtZQUN6QixJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUk7U0FDakIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUM5QixJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUM7UUFDOUMsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQztRQUMxQyxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDO0lBQzVDLENBQUM7O0FBMUNILDhCQTJDQzs7O0FBaUZEOzs7Ozs7R0FNRztBQUNILE1BQWEsUUFBUyxTQUFRLHNCQUFTO0lBK0NyQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQW9CO1FBQzVELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxFQUFFLENBQUM7UUFFckQsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLHFCQUFHLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxPQUFPLEVBQUU7WUFDN0MsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhO1lBQ2xDLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztZQUM5QixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsOEJBQThCLEVBQUUsS0FBSyxDQUFDLDhCQUE4QjtZQUNwRSwwQkFBMEIsRUFBRSxLQUFLLENBQUMsMEJBQTBCO1lBQzVELDBCQUEwQixFQUFFLEtBQUssQ0FBQywwQkFBMEI7WUFDNUQsc0JBQXNCLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtZQUNwRCxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7WUFDNUIsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNO1lBQ3BCLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7WUFDeEMsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJO1lBQ2hCLGdCQUFnQixFQUFFLElBQUksQ0FBQyxnQkFBZ0I7U0FDeEMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLEdBQUcsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQztRQUM3QixJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDO1FBQ3JDLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUM7UUFDM0MsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDO1FBQy9DLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQztRQUNqRCxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDO1FBQ3pDLElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUM7UUFDakMsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDO1FBQy9DLElBQUksQ0FBQyw4QkFBOEIsR0FBRyxLQUFLLENBQUMsOEJBQThCLENBQUM7SUFDN0UsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxhQUFhLENBQUMsSUFBWTtRQUMvQixJQUFJLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUNyQyxJQUFJLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLGtCQUFrQixFQUFFO1lBQ2hELElBQUk7U0FDTCxDQUFDLENBQUM7SUFDTCxDQUFDOztBQXhGSCw0QkF5RkM7OztBQXFDRDs7OztHQUlHO0FBQ0gsTUFBYSxjQUFlLFNBQVEsc0JBQVM7SUFXM0MsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUEwQjtRQUNsRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksS0FBSyxDQUFDLElBQUksSUFBSSxTQUFTLElBQUksS0FBSyxDQUFDLGFBQWEsSUFBSSxTQUFTLElBQUksS0FBSyxDQUFDLFFBQVEsQ0FBQyw4QkFBOEIsSUFBSSxTQUFTLEVBQUU7WUFDN0gsTUFBTSxJQUFJLEtBQUssQ0FBQyxrR0FBa0csQ0FBQyxDQUFDO1NBQ3JIO2FBQU07WUFDTCxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUkscUJBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO2dCQUM5RCxVQUFVLEVBQUUsS0FBSyxDQUFDLFFBQVEsQ0FBQyxVQUFVO2dCQUNyQyxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUk7Z0JBQ2hCLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztnQkFDOUIsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhO2FBQ25DLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxvQkFBb0IsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLHdCQUF3QixDQUFDO1NBQ3RFO0lBQ0gsQ0FBQzs7QUExQkgsd0NBMkJDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHtcbiAgYXdzX2VjMiBhcyBlYzIsXG4gIENmblRhZyxcbiAgSVJlc29sdmFibGUsXG4gIFN0YWNrLFxufSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuZXhwb3J0IGVudW0gSXBhbVBvb2xBZGRyZXNzRmFtaWx5IHtcbiAgSVBWNCA9ICdpcHY0JyxcbiAgSVBWNiA9ICdpcHY2J1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgb2YgdGhlIElQQU1cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJcGFtUHJvcHMge1xuICAvKipcbiAgICogVGhlIGRlc2NyaXB0aW9uIGZvciB0aGUgSVBBTS5cbiAgICovXG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uPzogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIG9wZXJhdGluZyBSZWdpb25zIGZvciBhbiBJUEFNLlxuICAgKlxuICAgKiBPcGVyYXRpbmcgUmVnaW9ucyBhcmUgQVdTIFJlZ2lvbnMgd2hlcmUgdGhlIElQQU0gaXMgYWxsb3dlZCB0byBtYW5hZ2UgSVAgYWRkcmVzcyBDSURScy4gSVBBTSBvbmx5XG4gICAqIGRpc2NvdmVycyBhbmQgbW9uaXRvcnMgcmVzb3VyY2VzIGluIHRoZSBBV1MgUmVnaW9ucyB5b3Ugc2VsZWN0IGFzIG9wZXJhdGluZyBSZWdpb25zLlxuICAgKlxuICAgKiBGb3IgbW9yZSBpbmZvcm1hdGlvbiBhYm91dCBvcGVyYXRpbmcgUmVnaW9ucywgc2VlIENyZWF0ZSBhbiBJUEFNIGluIHRoZSBBbWF6b24gVlBDIElQQU0gVXNlciBHdWlkZS5cbiAgICogQHNlZSBodHRwczovL3ZwYy9sYXRlc3QvaXBhbS9jcmVhdGUtaXBhbS5odG1sXG4gICAqXG4gICAqIEBkZWZhdWx0IFN0YWNrLm9mKHRoaXMpLnJlZ2lvblxuICAgKi9cbiAgcmVhZG9ubHkgb3BlcmF0aW5nUmVnaW9ucz86IHN0cmluZ1tdO1xuICAvKipcbiAgICogVGhlIGtleS92YWx1ZSBjb21iaW5hdGlvbiBvZiB0YWdzIHRvIGFzc2lnbiB0byB0aGUgcmVzb3VyY2UuXG4gICAqL1xuICByZWFkb25seSB0YWdzPzogQ2ZuVGFnW107XG59XG5cbi8qKlxuICogQ3JlYXRlcyBhbiBJUEFNLlxuICpcbiAqIFBBTSBpcyBhIFZQQyBmZWF0dXJlIHRoYXQgeW91IGNhbiB1c2UgdG8gYXV0b21hdGUgeW91ciBJUCBhZGRyZXNzIG1hbmFnZW1lbnQgd29ya2Zsb3dzIGluY2x1ZGluZ1xuICogYXNzaWduaW5nLCB0cmFja2luZywgdHJvdWJsZXNob290aW5nLCBhbmQgYXVkaXRpbmcgSVAgYWRkcmVzc2VzIGFjcm9zcyBBV1MgUmVnaW9ucyBhbmQgYWNjb3VudHNcbiAqIHRocm91Z2hvdXQgeW91ciBBV1MgT3JnYW5pemF0aW9uLiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlIFdoYXQgaXMgSVBBTT8gaW4gdGhlIEFtYXpvbiBWUEMgSVBBTVxuICogVXNlciBHdWlkZS5cbiAqXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS92cGMvbGF0ZXN0L2lwYW0vd2hhdC1pcy1pdC1pcGFtLmh0bWxcbiAqL1xuZXhwb3J0IGNsYXNzIElwYW0gZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuXG4gIC8qKlxuICAgKiBUaGUgdW5kZXJseWluZyBJUEFNIHJlc291cmNlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgaXBhbTogZWMyLkNmbklQQU07XG4gIC8qKlxuICAgKiBUaGUgSUQgb2YgdGhlIHJlc3VsdGluZyBJUEFNIHJlc291cmNlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgaXBhbUlkOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgZGVmYXVsdCBwcml2YXRlIHNjb3BlIElEXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcHJpdmF0ZURlZmF1bHRTY29wZUlkOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgZGVmYXVsdCBwdWJsaWMgc2NvcGUgSURcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwdWJsaWNEZWZhdWx0U2NvcGVJZDogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBzY29wZXMgaW4gdGhpcyBJUEFNXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc2NvcGVDb3VudDogbnVtYmVyO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBJcGFtUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgbGV0IG9wZXJhdGluZ1JlZ2lvbnM6ZWMyLkNmbklQQU0uSXBhbU9wZXJhdGluZ1JlZ2lvblByb3BlcnR5W10gPSBbXTtcbiAgICBpZiAocHJvcHMub3BlcmF0aW5nUmVnaW9ucykge1xuICAgICAgZm9yIChjb25zdCByZWdpb24gb2YgcHJvcHMub3BlcmF0aW5nUmVnaW9ucykge1xuICAgICAgICBvcGVyYXRpbmdSZWdpb25zLnB1c2goe1xuICAgICAgICAgIHJlZ2lvbk5hbWU6IHJlZ2lvbixcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIG9wZXJhdGluZ1JlZ2lvbnMucHVzaCh7XG4gICAgICAgIHJlZ2lvbk5hbWU6IFN0YWNrLm9mKHRoaXMpLnJlZ2lvbixcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHRoaXMuaXBhbSA9IG5ldyBlYzIuQ2ZuSVBBTSh0aGlzLCAnSVBBTScsIHtcbiAgICAgIGRlc2NyaXB0aW9uOiBwcm9wcy5kZXNjcmlwdGlvbixcbiAgICAgIG9wZXJhdGluZ1JlZ2lvbnMsXG4gICAgICB0YWdzOiBwcm9wcy50YWdzLFxuICAgIH0pO1xuXG4gICAgdGhpcy5pcGFtSWQgPSB0aGlzLmlwYW0ucmVmO1xuICAgIHRoaXMucHJpdmF0ZURlZmF1bHRTY29wZUlkID0gdGhpcy5pcGFtLmF0dHJQcml2YXRlRGVmYXVsdFNjb3BlSWQ7XG4gICAgdGhpcy5wdWJsaWNEZWZhdWx0U2NvcGVJZCA9IHRoaXMuaXBhbS5hdHRyUHVibGljRGVmYXVsdFNjb3BlSWQ7XG4gICAgdGhpcy5zY29wZUNvdW50ID0gdGhpcy5pcGFtLmF0dHJTY29wZUNvdW50O1xuICB9XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBvZiBhbiBJUEFNIFNjb3BlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSXBhbVNjb3BlUHJvcHMge1xuICAvKipcbiAgICogVGhlIElQQU0gZm9yIHdoaWNoIHlvdSdyZSBjcmVhdGluZyB0aGUgc2NvcGUuXG4gICAqL1xuICByZWFkb25seSBpcGFtOiBJcGFtO1xuICAvKipcbiAgICogVGhlIGRlc2NyaXB0aW9uIG9mIHRoZSBzY29wZS5cbiAgICovXG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uPzogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIGtleS92YWx1ZSBjb21iaW5hdGlvbiBvZiB0YWdzIHRvIGFzc2lnbiB0byB0aGUgcmVzb3VyY2UuXG4gICAqL1xuICByZWFkb25seSB0YWdzPzogQ2ZuVGFnW107XG59XG5cbi8qKlxuICogQW4gSVBBTSBTY29wZVxuICpcbiAqIEluIElQQU0sIGEgc2NvcGUgaXMgdGhlIGhpZ2hlc3QtbGV2ZWwgY29udGFpbmVyIHdpdGhpbiBJUEFNLiBBbiBJUEFNIGNvbnRhaW5zIHR3byBkZWZhdWx0IHNjb3Blcy5cbiAqIEVhY2ggc2NvcGUgcmVwcmVzZW50cyB0aGUgSVAgc3BhY2UgZm9yIGEgc2luZ2xlIG5ldHdvcmsuIFRoZSBwcml2YXRlIHNjb3BlIGlzIGludGVuZGVkIGZvciBhbGwgcHJpdmF0ZVxuICogSVAgYWRkcmVzcyBzcGFjZS4gVGhlIHB1YmxpYyBzY29wZSBpcyBpbnRlbmRlZCBmb3IgYWxsIHB1YmxpYyBJUCBhZGRyZXNzIHNwYWNlLiBTY29wZXMgZW5hYmxlIHlvdSB0b1xuICogcmV1c2UgSVAgYWRkcmVzc2VzIGFjcm9zcyBtdWx0aXBsZSB1bmNvbm5lY3RlZCBuZXR3b3JrcyB3aXRob3V0IGNhdXNpbmcgSVAgYWRkcmVzcyBvdmVybGFwIG9yIGNvbmZsaWN0LlxuICovXG5leHBvcnQgY2xhc3MgSXBhbVNjb3BlIGV4dGVuZHMgQ29uc3RydWN0IHtcblxuICAvKipcbiAgICogVGhlIHVuZGVybHlpbmcgSVBBTSBTY29wZSByZXNvdXJjZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHNjb3BlOiBlYzIuQ2ZuSVBBTVNjb3BlO1xuICAvKipcbiAgICogVGhlIEFSTiBvZiB0aGUgcmVzdWx0aW5nIElQQU0gU2NvcGUgcmVzb3VyY2VcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhcm46IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgdGhlIElQQU0gdGhpcyBzY29wZSBiZWxvbmdzIHRvXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgaXBhbUFybjogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIElEIG9mIHRoZSByZXN1bHRpbmcgSVBBTSBTY29wZSByZXNvdXJjZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGlwYW1TY29wZUlkOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgd2hldGhlciB0aGUgc2NvcGUgaXMgdGhlIGRlZmF1bHQgc2NvcGUgZm9yIHRoZSBJUEFNXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgaXNEZWZhdWx0OiBJUmVzb2x2YWJsZTtcbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgcG9vbHMgaW4gdGhlIHNjb3BlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcG9vbENvdW50OiBudW1iZXI7XG5cblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogSXBhbVNjb3BlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5zY29wZSA9IG5ldyBlYzIuQ2ZuSVBBTVNjb3BlKHRoaXMsICdTY29wZScsIHtcbiAgICAgIGRlc2NyaXB0aW9uOiBwcm9wcy5kZXNjcmlwdGlvbixcbiAgICAgIGlwYW1JZDogcHJvcHMuaXBhbS5pcGFtSWQsXG4gICAgICB0YWdzOiBwcm9wcy50YWdzLFxuICAgIH0pO1xuXG4gICAgdGhpcy5hcm4gPSB0aGlzLnNjb3BlLmF0dHJBcm47XG4gICAgdGhpcy5pcGFtQXJuID0gdGhpcy5zY29wZS5hdHRySXBhbUFybjtcbiAgICB0aGlzLmlwYW1TY29wZUlkID0gdGhpcy5zY29wZS5hdHRySXBhbVNjb3BlSWQ7XG4gICAgdGhpcy5pc0RlZmF1bHQgPSB0aGlzLnNjb3BlLmF0dHJJc0RlZmF1bHQ7XG4gICAgdGhpcy5wb29sQ291bnQgPSB0aGlzLnNjb3BlLmF0dHJQb29sQ291bnQ7XG4gIH1cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIG9mIGFuIElQQU0gUG9vbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIElwYW1Qb29sUHJvcHMge1xuICAvKipcbiAgICogVGhlIGFkZHJlc3MgZmFtaWx5IG9mIHRoZSBwb29sLCBlaXRoZXIgSVB2NCBvciBJUHY2LlxuICAgKi9cbiAgcmVhZG9ubHkgYWRkcmVzc0ZhbWlseTogSXBhbVBvb2xBZGRyZXNzRmFtaWx5O1xuICAvKipcbiAgICogVGhlIElQQU0gc2NvcGUgdGhpcyBwb29sIGlzIGFzc29jaWF0ZWQgd2l0aC5cbiAgICovXG4gIHJlYWRvbmx5IGlwYW1TY29wZUlkOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgZGVmYXVsdCBuZXRtYXNrIGxlbmd0aCBmb3IgYWxsb2NhdGlvbnMgYWRkZWQgdG8gdGhpcyBwb29sLlxuICAgKlxuICAgKiBJZiwgZm9yIGV4YW1wbGUsIHRoZSBDSURSIGFzc2lnbmVkIHRvIHRoaXMgcG9vbCBpcyAxMC4wLjAuMC84IGFuZCB5b3UgZW50ZXIgMTYgaGVyZSxcbiAgICogbmV3IGFsbG9jYXRpb25zIHdpbGwgZGVmYXVsdCB0byAxMC4wLjAuMC8xNi5cbiAgICovXG4gIHJlYWRvbmx5IGFsbG9jYXRpb25EZWZhdWx0TmV0bWFza0xlbmd0aD86IG51bWJlcjtcbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIG5ldG1hc2sgbGVuZ3RoIHBvc3NpYmxlIGZvciBDSURSIGFsbG9jYXRpb25zIGluIHRoaXMgSVBBTSBwb29sIHRvIGJlIGNvbXBsaWFudC5cbiAgICpcbiAgICogVGhlIG1heGltdW0gbmV0bWFzayBsZW5ndGggbXVzdCBiZSBncmVhdGVyIHRoYW4gdGhlIG1pbmltdW0gbmV0bWFzayBsZW5ndGguXG4gICAqIFBvc3NpYmxlIG5ldG1hc2sgbGVuZ3RocyBmb3IgSVB2NCBhZGRyZXNzZXMgYXJlIDAgLSAzMi4gUG9zc2libGUgbmV0bWFzayBsZW5ndGhzIGZvciBJUHY2IGFkZHJlc3NlcyBhcmUgMCAtIDEyOC5cbiAgICovXG4gIHJlYWRvbmx5IGFsbG9jYXRpb25NYXhOZXRtYXNrTGVuZ3RoPzogbnVtYmVyO1xuICAvKipcbiAgICogVGhlIG1pbmltdW0gbmV0bWFzayBsZW5ndGggcmVxdWlyZWQgZm9yIENJRFIgYWxsb2NhdGlvbnMgaW4gdGhpcyBJUEFNIHBvb2wgdG8gYmUgY29tcGxpYW50LlxuICAgKlxuICAgKiBUaGUgbWluaW11bSBuZXRtYXNrIGxlbmd0aCBtdXN0IGJlIGxlc3MgdGhhbiB0aGUgbWF4aW11bSBuZXRtYXNrIGxlbmd0aC5cbiAgICogUG9zc2libGUgbmV0bWFzayBsZW5ndGhzIGZvciBJUHY0IGFkZHJlc3NlcyBhcmUgMCAtIDMyLiBQb3NzaWJsZSBuZXRtYXNrIGxlbmd0aHMgZm9yIElQdjYgYWRkcmVzc2VzIGFyZSAwIC0gMTI4LlxuICAgKi9cbiAgcmVhZG9ubHkgYWxsb2NhdGlvbk1pbk5ldG1hc2tMZW5ndGg/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBUYWdzIHRoYXQgYXJlIHJlcXVpcmVkIGZvciByZXNvdXJjZXMgdGhhdCB1c2UgQ0lEUnMgZnJvbSB0aGlzIElQQU0gcG9vbC5cbiAgICpcbiAgICogUmVzb3VyY2VzIHRoYXQgZG8gbm90IGhhdmUgdGhlc2UgdGFncyB3aWxsIG5vdCBiZSBhbGxvd2VkIHRvIGFsbG9jYXRlIHNwYWNlIGZyb20gdGhlIHBvb2wuXG4gICAqIElmIHRoZSByZXNvdXJjZXMgaGF2ZSB0aGVpciB0YWdzIGNoYW5nZWQgYWZ0ZXIgdGhleSBoYXZlIGFsbG9jYXRlZCBzcGFjZSBvciBpZiB0aGUgYWxsb2NhdGlvblxuICAgKiB0YWdnaW5nIHJlcXVpcmVtZW50cyBhcmUgY2hhbmdlZCBvbiB0aGUgcG9vbCwgdGhlIHJlc291cmNlIG1heSBiZSBtYXJrZWQgYXMgbm9uY29tcGxpYW50LlxuICAgKi9cbiAgcmVhZG9ubHkgYWxsb2NhdGlvblJlc291cmNlVGFncz86IENmblRhZ1tdO1xuICAvKipcbiAgICogSWYgc2VsZWN0ZWQsIElQQU0gd2lsbCBjb250aW51b3VzbHkgbG9vayBmb3IgcmVzb3VyY2VzIHdpdGhpbiB0aGUgQ0lEUiByYW5nZSBvZiB0aGlzIHBvb2wgYW5kIGF1dG9tYXRpY2FsbHkgaW1wb3J0XG4gICAqIHRoZW0gYXMgYWxsb2NhdGlvbnMgaW50byB5b3VyIElQQU0uIFRoZSBDSURScyB0aGF0IHdpbGwgYmUgYWxsb2NhdGVkIGZvciB0aGVzZSByZXNvdXJjZXMgbXVzdCBub3QgYWxyZWFkeSBiZSBhbGxvY2F0ZWRcbiAgICogdG8gb3RoZXIgcmVzb3VyY2VzIGluIG9yZGVyIGZvciB0aGUgaW1wb3J0IHRvIHN1Y2NlZWQuIElQQU0gd2lsbCBpbXBvcnQgYSBDSURSIHJlZ2FyZGxlc3Mgb2YgaXRzIGNvbXBsaWFuY2Ugd2l0aCB0aGVcbiAgICogcG9vbCdzIGFsbG9jYXRpb24gcnVsZXMsIHNvIGEgcmVzb3VyY2UgbWlnaHQgYmUgaW1wb3J0ZWQgYW5kIHN1YnNlcXVlbnRseSBtYXJrZWQgYXMgbm9uY29tcGxpYW50LiBJZiBJUEFNIGRpc2NvdmVyc1xuICAgKiBtdWx0aXBsZSBDSURScyB0aGF0IG92ZXJsYXAsIElQQU0gd2lsbCBpbXBvcnQgdGhlIGxhcmdlc3QgQ0lEUiBvbmx5LiBJZiBJUEFNIGRpc2NvdmVycyBtdWx0aXBsZSBDSURScyB3aXRoIG1hdGNoaW5nXG4gICAqIENJRFJzLCBJUEFNIHdpbGwgcmFuZG9tbHkgaW1wb3J0IG9uZSBvZiB0aGVtIG9ubHkuXG4gICAqXG4gICAqIEEgbG9jYWxlIG11c3QgYmUgc2V0IG9uIHRoZSBwb29sIGZvciB0aGlzIGZlYXR1cmUgdG8gd29yay5cbiAgICovXG4gIHJlYWRvbmx5IGF1dG9JbXBvcnQ/OiBib29sZWFuO1xuICAvKipcbiAgICogVGhlIGRlc2NyaXB0aW9uIG9mIHRoZSBwb29sLlxuICAgKi9cbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgbG9jYWxlIG9mIHRoZSBJUEFNIHBvb2wuIEluIElQQU0sIHRoZSBsb2NhbGUgaXMgdGhlIEFXUyBSZWdpb24gd2hlcmUgeW91IHdhbnQgdG8gbWFrZSBhbiBJUEFNIHBvb2wgYXZhaWxhYmxlXG4gICAqIGZvciBhbGxvY2F0aW9ucy5Pbmx5IHJlc291cmNlcyBpbiB0aGUgc2FtZSBSZWdpb24gYXMgdGhlIGxvY2FsZSBvZiB0aGUgcG9vbCBjYW4gZ2V0IElQIGFkZHJlc3MgYWxsb2NhdGlvbnMgZnJvbSB0aGUgcG9vbC5cbiAgICogWW91IGNhbiBvbmx5IGFsbG9jYXRlIGEgQ0lEUiBmb3IgYSBWUEMsIGZvciBleGFtcGxlLCBmcm9tIGFuIElQQU0gcG9vbCB0aGF0IHNoYXJlcyBhIGxvY2FsZSB3aXRoIHRoZSBWUEPigJlzIFJlZ2lvbi5cbiAgICogTm90ZSB0aGF0IG9uY2UgeW91IGNob29zZSBhIExvY2FsZSBmb3IgYSBwb29sLCB5b3UgY2Fubm90IG1vZGlmeSBpdC4gSWYgeW91IGNob29zZSBhbiBBV1MgUmVnaW9uIGZvciBsb2NhbGUgdGhhdCBoYXNcbiAgICogbm90IGJlZW4gY29uZmlndXJlZCBhcyBhbiBvcGVyYXRpbmcgUmVnaW9uIGZvciB0aGUgSVBBTSwgeW91J2xsIGdldCBhbiBlcnJvci5cbiAgICovXG4gIHJlYWRvbmx5IGxvY2FsZT86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBDSURScyBwcm92aXNpb25lZCB0byB0aGUgSVBBTSBwb29sLiBBIENJRFIgaXMgYSByZXByZXNlbnRhdGlvbiBvZiBhbiBJUCBhZGRyZXNzIGFuZCBpdHMgYXNzb2NpYXRlZCBuZXR3b3JrIG1hc2tcbiAgICogKG9yIG5ldG1hc2spIGFuZCByZWZlcnMgdG8gYSByYW5nZSBvZiBJUCBhZGRyZXNzZXNcbiAgICovXG4gIHJlYWRvbmx5IHByb3Zpc2lvbmVkQ2lkcnM/OiBlYzIuQ2ZuSVBBTVBvb2wuUHJvdmlzaW9uZWRDaWRyUHJvcGVydHlbXTtcbiAgLyoqXG4gICAqIFRoZSBJRCBvZiB0aGUgc291cmNlIElQQU0gcG9vbC4gWW91IGNhbiB1c2UgdGhpcyBvcHRpb24gdG8gY3JlYXRlIGFuIElQQU0gcG9vbCB3aXRoaW4gYW4gZXhpc3Rpbmcgc291cmNlIHBvb2wuXG4gICAqL1xuICByZWFkb25seSBzb3VyY2VJcGFtUG9vbElkPzogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIGtleS92YWx1ZSBjb21iaW5hdGlvbiBvZiB0YWdzIHRvIGFzc2lnbiB0byB0aGUgcmVzb3VyY2UuXG4gICAqL1xuICByZWFkb25seSB0YWdzPzogQ2ZuVGFnW107XG59XG5cbi8qKlxuICogQW4gSVBBTSBQb29sXG4gKlxuICogSW4gSVBBTSwgYSBwb29sIGlzIGEgY29sbGVjdGlvbiBvZiBjb250aWd1b3VzIElQIGFkZHJlc3NlcyBDSURScy4gUG9vbHMgZW5hYmxlIHlvdSB0byBvcmdhbml6ZSB5b3VyIElQIGFkZHJlc3Nlc1xuICogYWNjb3JkaW5nIHRvIHlvdXIgcm91dGluZyBhbmQgc2VjdXJpdHkgbmVlZHMuIEZvciBleGFtcGxlLCBpZiB5b3UgaGF2ZSBzZXBhcmF0ZSByb3V0aW5nIGFuZCBzZWN1cml0eSBuZWVkcyBmb3JcbiAqIGRldmVsb3BtZW50IGFuZCBwcm9kdWN0aW9uIGFwcGxpY2F0aW9ucywgeW91IGNhbiBjcmVhdGUgYSBwb29sIGZvciBlYWNoLlxuICovXG5leHBvcnQgY2xhc3MgSXBhbVBvb2wgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuXG4gIC8qKlxuICAgKiBUaGUgdW5kZXJseWluZyBJUEFNIFBvb2wgcmVzb3VyY2VcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwb29sOiBlYzIuQ2ZuSVBBTVBvb2w7XG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoZSByZXN1bHRpbmcgSVBBTSBQb29sIHJlc291cmNlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXJuOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoZSBJUEFNIHRoaXMgcG9vbCBiZWxvbmdzIHRvXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgaXBhbUFybjogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIElEIG9mIHRoZSByZXN1bHRpbmcgSVBBTSBQb29sIHJlc291cmNlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgaXBhbVBvb2xJZDogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIEFSTiBvZiB0aGUgc2NvcGUgb2YgdGhlIElQQU0gUG9vbFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGlwYW1TY29wZUFybjogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIElQQU0gc2NvcGUgdHlwZSAocHVibGljIG9yIHByaXZhdGUpIG9mIHRoZSBzY29wZSBvZiB0aGUgSVBBTSBQb29sXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgaXBhbVNjb3BlVHlwZTogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIGRlcHRoIG9mIHBvb2xzIGluIHlvdXIgSVBBTSBwb29sLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHBvb2xEZXB0aDogbnVtYmVyO1xuICAvKipcbiAgICogVGhlIHN0YXRlIG9mIHRoZSBJUEFNIHBvb2wuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc3RhdGU6IHN0cmluZztcbiAgLyoqXG4gICAqIEEgbWVzc2FnZSByZWxhdGVkIHRvIHRoZSBmYWlsZWQgY3JlYXRpb24gb2YgYW4gSVBBTSBwb29sLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHN0YXRlTWVzc2FnZTogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIHByb3Zpc2lvbmVkIENJRFJzIGZvciB0aGlzIHBvb2xcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwcm92aXNpb25lZENpZHJzOiBlYzIuQ2ZuSVBBTVBvb2wuUHJvdmlzaW9uZWRDaWRyUHJvcGVydHlbXTtcbiAgLyoqXG4gICAqIFRoZSBkZWZhdWx0IG5ldG1hc2sgbGVuZ3RoIGZvciBhbGxvY2F0aW9ucyBhZGRlZCB0byB0aGlzIHBvb2wuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYWxsb2NhdGlvbkRlZmF1bHROZXRtYXNrTGVuZ3RoOiBudW1iZXIgfCB1bmRlZmluZWQ7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IElwYW1Qb29sUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5wcm92aXNpb25lZENpZHJzID0gcHJvcHMucHJvdmlzaW9uZWRDaWRycyA/PyBbXTtcblxuICAgIHRoaXMucG9vbCA9IG5ldyBlYzIuQ2ZuSVBBTVBvb2wodGhpcywgJ1Njb3BlJywge1xuICAgICAgYWRkcmVzc0ZhbWlseTogcHJvcHMuYWRkcmVzc0ZhbWlseSxcbiAgICAgIGlwYW1TY29wZUlkOiBwcm9wcy5pcGFtU2NvcGVJZCxcbiAgICAgIGRlc2NyaXB0aW9uOiBwcm9wcy5kZXNjcmlwdGlvbixcbiAgICAgIGFsbG9jYXRpb25EZWZhdWx0TmV0bWFza0xlbmd0aDogcHJvcHMuYWxsb2NhdGlvbkRlZmF1bHROZXRtYXNrTGVuZ3RoLFxuICAgICAgYWxsb2NhdGlvbk1heE5ldG1hc2tMZW5ndGg6IHByb3BzLmFsbG9jYXRpb25NYXhOZXRtYXNrTGVuZ3RoLFxuICAgICAgYWxsb2NhdGlvbk1pbk5ldG1hc2tMZW5ndGg6IHByb3BzLmFsbG9jYXRpb25NaW5OZXRtYXNrTGVuZ3RoLFxuICAgICAgYWxsb2NhdGlvblJlc291cmNlVGFnczogcHJvcHMuYWxsb2NhdGlvblJlc291cmNlVGFncyxcbiAgICAgIGF1dG9JbXBvcnQ6IHByb3BzLmF1dG9JbXBvcnQsXG4gICAgICBsb2NhbGU6IHByb3BzLmxvY2FsZSxcbiAgICAgIHNvdXJjZUlwYW1Qb29sSWQ6IHByb3BzLnNvdXJjZUlwYW1Qb29sSWQsXG4gICAgICB0YWdzOiBwcm9wcy50YWdzLFxuICAgICAgcHJvdmlzaW9uZWRDaWRyczogdGhpcy5wcm92aXNpb25lZENpZHJzLFxuICAgIH0pO1xuXG4gICAgdGhpcy5hcm4gPSB0aGlzLnBvb2wuYXR0ckFybjtcbiAgICB0aGlzLmlwYW1Bcm4gPSB0aGlzLnBvb2wuYXR0cklwYW1Bcm47XG4gICAgdGhpcy5pcGFtUG9vbElkID0gdGhpcy5wb29sLmF0dHJJcGFtUG9vbElkO1xuICAgIHRoaXMuaXBhbVNjb3BlQXJuID0gdGhpcy5wb29sLmF0dHJJcGFtU2NvcGVBcm47XG4gICAgdGhpcy5pcGFtU2NvcGVUeXBlID0gdGhpcy5wb29sLmF0dHJJcGFtU2NvcGVUeXBlO1xuICAgIHRoaXMucG9vbERlcHRoID0gdGhpcy5wb29sLmF0dHJQb29sRGVwdGg7XG4gICAgdGhpcy5zdGF0ZSA9IHRoaXMucG9vbC5hdHRyU3RhdGU7XG4gICAgdGhpcy5zdGF0ZU1lc3NhZ2UgPSB0aGlzLnBvb2wuYXR0clN0YXRlTWVzc2FnZTtcbiAgICB0aGlzLmFsbG9jYXRpb25EZWZhdWx0TmV0bWFza0xlbmd0aCA9IHByb3BzLmFsbG9jYXRpb25EZWZhdWx0TmV0bWFza0xlbmd0aDtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgQ0lEUiB0byB0aGUgcG9vbFxuICAgKlxuICAgKiBAcGFyYW0gY2lkciBUaGUgQ0lEUiB0byBhZGQgdG8gdGhlIHBvb2xcbiAgICovXG4gIHB1YmxpYyBwcm92aXNpb25DaWRyKGNpZHI6IHN0cmluZykge1xuICAgIHRoaXMucHJvdmlzaW9uZWRDaWRycy5wdXNoKHsgY2lkciB9KTtcbiAgICB0aGlzLnBvb2wuYWRkUHJvcGVydHlPdmVycmlkZSgnUHJvdmlzaW9uZWRDaWRycycsIHtcbiAgICAgIGNpZHIsXG4gICAgfSk7XG4gIH1cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIG9mIGFuIElQQU0gQWxsb2NhdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIElwYW1BbGxvY2F0aW9uUHJvcHMge1xuICAvKipcbiAgICogVGhlIElQQU0gcG9vbCBmcm9tIHdoaWNoIHlvdSB3b3VsZCBsaWtlIHRvIGFsbG9jYXRlIGEgQ0lEUi5cbiAgICovXG4gIHJlYWRvbmx5IGlwYW1Qb29sOiBJcGFtUG9vbDtcbiAgLyoqXG4gICAqIFRoZSBDSURSIHlvdSB3b3VsZCBsaWtlIHRvIGFsbG9jYXRlIGZyb20gdGhlIElQQU0gcG9vbC4gTm90ZSB0aGUgZm9sbG93aW5nOlxuICAgKlxuICAgKiBJZiB0aGVyZSBpcyBubyBEZWZhdWx0TmV0bWFza0xlbmd0aCBhbGxvY2F0aW9uIHJ1bGUgc2V0IG9uIHRoZSBwb29sLCB5b3UgbXVzdFxuICAgKiBzcGVjaWZ5IGVpdGhlciB0aGUgTmV0bWFza0xlbmd0aCBvciB0aGUgQ0lEUi5cbiAgICpcbiAgICogSWYgdGhlIERlZmF1bHROZXRtYXNrTGVuZ3RoIGFsbG9jYXRpb24gcnVsZSBpcyBzZXQgb24gdGhlIHBvb2wsIHlvdSBjYW4gc3BlY2lmeVxuICAgKiBlaXRoZXIgdGhlIE5ldG1hc2tMZW5ndGggb3IgdGhlIENJRFIgYW5kIHRoZSBEZWZhdWx0TmV0bWFza0xlbmd0aCBhbGxvY2F0aW9uIHJ1bGVcbiAgICogd2lsbCBiZSBpZ25vcmVkLlxuICAgKi9cbiAgcmVhZG9ubHkgY2lkcj86IHN0cmluZztcbiAgLyoqXG4gICAqIEEgZGVzY3JpcHRpb24gb2YgdGhlIHBvb2wgYWxsb2NhdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgbmV0bWFzayBsZW5ndGggb2YgdGhlIENJRFIgeW91IHdvdWxkIGxpa2UgdG8gYWxsb2NhdGUgZnJvbSB0aGUgSVBBTSBwb29sLiBOb3RlIHRoZSBmb2xsb3dpbmc6XG4gICAqXG4gICAqIElmIHRoZXJlIGlzIG5vIERlZmF1bHROZXRtYXNrTGVuZ3RoIGFsbG9jYXRpb24gcnVsZSBzZXQgb24gdGhlIHBvb2wsIHlvdSBtdXN0IHNwZWNpZnkgZWl0aGVyIHRoZVxuICAgKiBOZXRtYXNrTGVuZ3RoIG9yIHRoZSBDSURSLlxuICAgKlxuICAgKiBJZiB0aGUgRGVmYXVsdE5ldG1hc2tMZW5ndGggYWxsb2NhdGlvbiBydWxlIGlzIHNldCBvbiB0aGUgcG9vbCwgeW91IGNhbiBzcGVjaWZ5IGVpdGhlciB0aGVcbiAgICogTmV0bWFza0xlbmd0aCBvciB0aGUgQ0lEUiBhbmQgdGhlIERlZmF1bHROZXRtYXNrTGVuZ3RoIGFsbG9jYXRpb24gcnVsZSB3aWxsIGJlIGlnbm9yZWQuXG4gICAqL1xuICByZWFkb25seSBuZXRtYXNrTGVuZ3RoPzogbnVtYmVyO1xufVxuXG4vKipcbiAqIEFuIElQQU0gQWxsb2NhdGlvblxuICpcbiAqIEluIElQQU0sIGFuIGFsbG9jYXRpb24gaXMgYSBDSURSIGFzc2lnbm1lbnQgZnJvbSBhbiBJUEFNIHBvb2wgdG8gYW5vdGhlciByZXNvdXJjZSBvciBJUEFNIHBvb2wuXG4gKi9cbmV4cG9ydCBjbGFzcyBJcGFtQWxsb2NhdGlvbiBleHRlbmRzIENvbnN0cnVjdCB7XG5cbiAgLyoqXG4gICAqIFRoZSB1bmRlcmx5aW5nIElQQU0gQWxsb2NhdGlvbiByZXNvdXJjZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGFsbG9jYXRpb246IGVjMi5DZm5JUEFNQWxsb2NhdGlvbjtcbiAgLyoqXG4gICAqIFRoZSBJRCBvZiB0aGUgYWxsb2NhdGlvblxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGlwYW1Qb29sQWxsb2NhdGlvbklkOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IElwYW1BbGxvY2F0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgaWYgKHByb3BzLmNpZHIgPT0gdW5kZWZpbmVkICYmIHByb3BzLm5ldG1hc2tMZW5ndGggPT0gdW5kZWZpbmVkICYmIHByb3BzLmlwYW1Qb29sLmFsbG9jYXRpb25EZWZhdWx0TmV0bWFza0xlbmd0aCA9PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignRWl0aGVyIENJRFIgb3IgTmV0bWFza0xlbmd0aCBtdXN0IGJlIHNwZWNpZmllZCB3aGVuIG5vIGRlZmF1bHQgbmV0bWFzayBsZW5ndGggaXMgc2V0IG9uIHRoZSBwb29sJyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuYWxsb2NhdGlvbiA9IG5ldyBlYzIuQ2ZuSVBBTUFsbG9jYXRpb24odGhpcywgJ0FsbG9jYXRpb24nLCB7XG4gICAgICAgIGlwYW1Qb29sSWQ6IHByb3BzLmlwYW1Qb29sLmlwYW1Qb29sSWQsXG4gICAgICAgIGNpZHI6IHByb3BzLmNpZHIsXG4gICAgICAgIGRlc2NyaXB0aW9uOiBwcm9wcy5kZXNjcmlwdGlvbixcbiAgICAgICAgbmV0bWFza0xlbmd0aDogcHJvcHMubmV0bWFza0xlbmd0aCxcbiAgICAgIH0pO1xuXG4gICAgICB0aGlzLmlwYW1Qb29sQWxsb2NhdGlvbklkID0gdGhpcy5hbGxvY2F0aW9uLmF0dHJJcGFtUG9vbEFsbG9jYXRpb25JZDtcbiAgICB9XG4gIH1cbn1cbiJdfQ==