"""Typed model names for autocomplete-friendly usage."""

from enum import Enum


class Models(str, Enum):
    """All models supported by LiteLLM."""

    # Original models (for backwards compatibility)
    GPT4O = "openai/gpt-4o"
    CLAUDE_3_5 = "anthropic/claude-3.5"
    O4_MINI = "openai/o4-mini"
    O4 = "openai/o4"

    # 1024-X-1024 models
    MODEL_1024_X_1024_50_STEPS_BEDROCK_AMAZON_NOVA_CANVAS_V1_0 = "1024-x-1024/50-steps/bedrock/amazon.nova-canvas-v1:0"
    MODEL_1024_X_1024_50_STEPS_STABILITY_STABLE_DIFFUSION_XL_V1 = "1024-x-1024/50-steps/stability.stable-diffusion-xl-v1"
    MODEL_1024_X_1024_DALL_E_2 = "1024-x-1024/dall-e-2"
    MODEL_1024_X_1024_MAX_STEPS_STABILITY_STABLE_DIFFUSION_XL_V1 = "1024-x-1024/max-steps/stability.stable-diffusion-xl-v1"

    # 256-X-256 models
    MODEL_256_X_256_DALL_E_2 = "256-x-256/dall-e-2"

    # 512-X-512 models
    MODEL_512_X_512_50_STEPS_STABILITY_STABLE_DIFFUSION_XL_V0 = "512-x-512/50-steps/stability.stable-diffusion-xl-v0"
    MODEL_512_X_512_DALL_E_2 = "512-x-512/dall-e-2"
    MODEL_512_X_512_MAX_STEPS_STABILITY_STABLE_DIFFUSION_XL_V0 = "512-x-512/max-steps/stability.stable-diffusion-xl-v0"

    # A16Z-INFRA models
    A16Z_INFRA_LLAMA_2_13B_CHAT_2A7F981751EC7FDF87B5B91AD4DB53683A98082E9FF7BFD12C8CD5EA85980A52 = "a16z-infra/llama-2-13b-chat:2a7f981751ec7fdf87b5b91ad4db53683a98082e9ff7bfd12c8cd5ea85980a52"

    # ANTHROPIC models
    ANTHROPIC_CLAUDE_3_5_HAIKU_20241022_V1_0 = "anthropic.claude-3-5-haiku-20241022-v1:0"
    ANTHROPIC_CLAUDE_3_5_SONNET_20240620_V1_0 = "anthropic.claude-3-5-sonnet-20240620-v1:0"
    ANTHROPIC_CLAUDE_3_5_SONNET_20241022_V2_0 = "anthropic.claude-3-5-sonnet-20241022-v2:0"
    ANTHROPIC_CLAUDE_3_7_SONNET_20240620_V1_0 = "anthropic.claude-3-7-sonnet-20240620-v1:0"
    ANTHROPIC_CLAUDE_3_HAIKU_20240307_V1_0 = "anthropic.claude-3-haiku-20240307-v1:0"
    ANTHROPIC_CLAUDE_3_OPUS_20240229_V1_0 = "anthropic.claude-3-opus-20240229-v1:0"
    ANTHROPIC_CLAUDE_3_SONNET_20240229_V1_0 = "anthropic.claude-3-sonnet-20240229-v1:0"
    ANTHROPIC_CLAUDE_HAIKU_4_5_20251001_V1_0 = "anthropic.claude-haiku-4-5-20251001-v1:0"
    ANTHROPIC_CLAUDE_HAIKU_4_5_20251001 = "anthropic.claude-haiku-4-5@20251001"
    ANTHROPIC_CLAUDE_INSTANT_V1 = "anthropic.claude-instant-v1"
    ANTHROPIC_CLAUDE_V1 = "anthropic.claude-v1"
    ANTHROPIC_CLAUDE_V2_1 = "anthropic.claude-v2:1"
    APAC_ANTHROPIC_CLAUDE_3_5_SONNET_20240620_V1_0 = "apac.anthropic.claude-3-5-sonnet-20240620-v1:0"
    APAC_ANTHROPIC_CLAUDE_3_5_SONNET_20241022_V2_0 = "apac.anthropic.claude-3-5-sonnet-20241022-v2:0"
    APAC_ANTHROPIC_CLAUDE_3_HAIKU_20240307_V1_0 = "apac.anthropic.claude-3-haiku-20240307-v1:0"
    APAC_ANTHROPIC_CLAUDE_3_SONNET_20240229_V1_0 = "apac.anthropic.claude-3-sonnet-20240229-v1:0"
    APAC_ANTHROPIC_CLAUDE_HAIKU_4_5_20251001_V1_0 = "apac.anthropic.claude-haiku-4-5-20251001-v1:0"
    CLAUDE_3_5_HAIKU_20241022 = "claude-3-5-haiku-20241022"
    CLAUDE_3_5_HAIKU_LATEST = "claude-3-5-haiku-latest"
    CLAUDE_3_5_SONNET_20240620 = "claude-3-5-sonnet-20240620"
    CLAUDE_3_5_SONNET_20241022 = "claude-3-5-sonnet-20241022"
    CLAUDE_3_5_SONNET_LATEST = "claude-3-5-sonnet-latest"
    CLAUDE_3_7_SONNET_20250219 = "claude-3-7-sonnet-20250219"
    CLAUDE_3_7_SONNET_LATEST = "claude-3-7-sonnet-latest"
    CLAUDE_3_HAIKU_20240307 = "claude-3-haiku-20240307"
    CLAUDE_3_OPUS_20240229 = "claude-3-opus-20240229"
    CLAUDE_3_OPUS_LATEST = "claude-3-opus-latest"
    CLAUDE_4_OPUS_20250514 = "claude-4-opus-20250514"
    CLAUDE_4_SONNET_20250514 = "claude-4-sonnet-20250514"
    CLAUDE_HAIKU_4_5 = "claude-haiku-4-5"
    CLAUDE_HAIKU_4_5_20251001 = "claude-haiku-4-5-20251001"
    CLAUDE_OPUS_4_1 = "claude-opus-4-1"
    CLAUDE_OPUS_4_1_20250805 = "claude-opus-4-1-20250805"
    CLAUDE_OPUS_4_20250514 = "claude-opus-4-20250514"
    CLAUDE_SONNET_4_20250514 = "claude-sonnet-4-20250514"
    CLAUDE_SONNET_4_5 = "claude-sonnet-4-5"
    CLAUDE_SONNET_4_5_20250929 = "claude-sonnet-4-5-20250929"
    EU_ANTHROPIC_CLAUDE_3_5_HAIKU_20241022_V1_0 = "eu.anthropic.claude-3-5-haiku-20241022-v1:0"
    EU_ANTHROPIC_CLAUDE_3_5_SONNET_20240620_V1_0 = "eu.anthropic.claude-3-5-sonnet-20240620-v1:0"
    EU_ANTHROPIC_CLAUDE_3_5_SONNET_20241022_V2_0 = "eu.anthropic.claude-3-5-sonnet-20241022-v2:0"
    EU_ANTHROPIC_CLAUDE_3_7_SONNET_20250219_V1_0 = "eu.anthropic.claude-3-7-sonnet-20250219-v1:0"
    EU_ANTHROPIC_CLAUDE_3_HAIKU_20240307_V1_0 = "eu.anthropic.claude-3-haiku-20240307-v1:0"
    EU_ANTHROPIC_CLAUDE_3_OPUS_20240229_V1_0 = "eu.anthropic.claude-3-opus-20240229-v1:0"
    EU_ANTHROPIC_CLAUDE_3_SONNET_20240229_V1_0 = "eu.anthropic.claude-3-sonnet-20240229-v1:0"
    EU_ANTHROPIC_CLAUDE_HAIKU_4_5_20251001_V1_0 = "eu.anthropic.claude-haiku-4-5-20251001-v1:0"
    JP_ANTHROPIC_CLAUDE_HAIKU_4_5_20251001_V1_0 = "jp.anthropic.claude-haiku-4-5-20251001-v1:0"
    US_ANTHROPIC_CLAUDE_3_5_HAIKU_20241022_V1_0 = "us.anthropic.claude-3-5-haiku-20241022-v1:0"
    US_ANTHROPIC_CLAUDE_3_5_SONNET_20240620_V1_0 = "us.anthropic.claude-3-5-sonnet-20240620-v1:0"
    US_ANTHROPIC_CLAUDE_3_5_SONNET_20241022_V2_0 = "us.anthropic.claude-3-5-sonnet-20241022-v2:0"
    US_ANTHROPIC_CLAUDE_3_HAIKU_20240307_V1_0 = "us.anthropic.claude-3-haiku-20240307-v1:0"
    US_ANTHROPIC_CLAUDE_3_OPUS_20240229_V1_0 = "us.anthropic.claude-3-opus-20240229-v1:0"
    US_ANTHROPIC_CLAUDE_3_SONNET_20240229_V1_0 = "us.anthropic.claude-3-sonnet-20240229-v1:0"
    US_ANTHROPIC_CLAUDE_HAIKU_4_5_20251001_V1_0 = "us.anthropic.claude-haiku-4-5-20251001-v1:0"

    # ANYSCALE models
    ANYSCALE_HUGGINGFACEH4_ZEPHYR_7B_BETA = "anyscale/HuggingFaceH4/zephyr-7b-beta"
    ANYSCALE_CODELLAMA_CODELLAMA_34B_INSTRUCT_HF = "anyscale/codellama/CodeLlama-34b-Instruct-hf"
    ANYSCALE_CODELLAMA_CODELLAMA_70B_INSTRUCT_HF = "anyscale/codellama/CodeLlama-70b-Instruct-hf"
    ANYSCALE_GOOGLE_GEMMA_7B_IT = "anyscale/google/gemma-7b-it"
    ANYSCALE_META_LLAMA_LLAMA_2_13B_CHAT_HF = "anyscale/meta-llama/Llama-2-13b-chat-hf"
    ANYSCALE_META_LLAMA_LLAMA_2_70B_CHAT_HF = "anyscale/meta-llama/Llama-2-70b-chat-hf"
    ANYSCALE_META_LLAMA_LLAMA_2_7B_CHAT_HF = "anyscale/meta-llama/Llama-2-7b-chat-hf"
    ANYSCALE_META_LLAMA_META_LLAMA_3_70B_INSTRUCT = "anyscale/meta-llama/Meta-Llama-3-70B-Instruct"
    ANYSCALE_META_LLAMA_META_LLAMA_3_8B_INSTRUCT = "anyscale/meta-llama/Meta-Llama-3-8B-Instruct"
    ANYSCALE_MISTRALAI_MISTRAL_7B_INSTRUCT_V0_1 = "anyscale/mistralai/Mistral-7B-Instruct-v0.1"
    ANYSCALE_MISTRALAI_MIXTRAL_8X22B_INSTRUCT_V0_1 = "anyscale/mistralai/Mixtral-8x22B-Instruct-v0.1"
    ANYSCALE_MISTRALAI_MIXTRAL_8X7B_INSTRUCT_V0_1 = "anyscale/mistralai/Mixtral-8x7B-Instruct-v0.1"

    # ASSEMBLYAI models
    ASSEMBLYAI_BEST = "assemblyai/best"
    ASSEMBLYAI_NANO = "assemblyai/nano"

    # AUSTISM models
    AUSTISM_CHRONOS_HERMES_13B = "Austism/chronos-hermes-13b"

    # AZURE models
    AZURE_ADA = "azure/ada"
    AZURE_CODEX_MINI = "azure/codex-mini"
    AZURE_COMMAND_R_PLUS = "azure/command-r-plus"
    AZURE_COMPUTER_USE_PREVIEW = "azure/computer-use-preview"
    AZURE_CONTAINER = "azure/container"
    AZURE_EU_GPT_4O_2024_08_06 = "azure/eu/gpt-4o-2024-08-06"
    AZURE_EU_GPT_4O_2024_11_20 = "azure/eu/gpt-4o-2024-11-20"
    AZURE_EU_GPT_4O_MINI_2024_07_18 = "azure/eu/gpt-4o-mini-2024-07-18"
    AZURE_EU_GPT_4O_MINI_REALTIME_PREVIEW_2024_12_17 = "azure/eu/gpt-4o-mini-realtime-preview-2024-12-17"
    AZURE_EU_GPT_4O_REALTIME_PREVIEW_2024_10_01 = "azure/eu/gpt-4o-realtime-preview-2024-10-01"
    AZURE_EU_GPT_4O_REALTIME_PREVIEW_2024_12_17 = "azure/eu/gpt-4o-realtime-preview-2024-12-17"
    AZURE_EU_O1_2024_12_17 = "azure/eu/o1-2024-12-17"
    AZURE_EU_O1_MINI_2024_09_12 = "azure/eu/o1-mini-2024-09-12"
    AZURE_EU_O1_PREVIEW_2024_09_12 = "azure/eu/o1-preview-2024-09-12"
    AZURE_EU_O3_MINI_2025_01_31 = "azure/eu/o3-mini-2025-01-31"
    AZURE_GLOBAL_STANDARD_GPT_4O_2024_08_06 = "azure/global-standard/gpt-4o-2024-08-06"
    AZURE_GLOBAL_STANDARD_GPT_4O_2024_11_20 = "azure/global-standard/gpt-4o-2024-11-20"
    AZURE_GLOBAL_STANDARD_GPT_4O_MINI = "azure/global-standard/gpt-4o-mini"
    AZURE_GLOBAL_GPT_4O_2024_08_06 = "azure/global/gpt-4o-2024-08-06"
    AZURE_GLOBAL_GPT_4O_2024_11_20 = "azure/global/gpt-4o-2024-11-20"
    AZURE_GPT_3_5_TURBO = "azure/gpt-3.5-turbo"
    AZURE_GPT_3_5_TURBO_0125 = "azure/gpt-3.5-turbo-0125"
    AZURE_GPT_3_5_TURBO_INSTRUCT_0914 = "azure/gpt-3.5-turbo-instruct-0914"
    AZURE_GPT_35_TURBO = "azure/gpt-35-turbo"
    AZURE_GPT_35_TURBO_0125 = "azure/gpt-35-turbo-0125"
    AZURE_GPT_35_TURBO_0301 = "azure/gpt-35-turbo-0301"
    AZURE_GPT_35_TURBO_0613 = "azure/gpt-35-turbo-0613"
    AZURE_GPT_35_TURBO_1106 = "azure/gpt-35-turbo-1106"
    AZURE_GPT_35_TURBO_16K = "azure/gpt-35-turbo-16k"
    AZURE_GPT_35_TURBO_16K_0613 = "azure/gpt-35-turbo-16k-0613"
    AZURE_GPT_35_TURBO_INSTRUCT = "azure/gpt-35-turbo-instruct"
    AZURE_GPT_35_TURBO_INSTRUCT_0914 = "azure/gpt-35-turbo-instruct-0914"
    AZURE_GPT_4 = "azure/gpt-4"
    AZURE_GPT_4_0125_PREVIEW = "azure/gpt-4-0125-preview"
    AZURE_GPT_4_0613 = "azure/gpt-4-0613"
    AZURE_GPT_4_1106_PREVIEW = "azure/gpt-4-1106-preview"
    AZURE_GPT_4_32K = "azure/gpt-4-32k"
    AZURE_GPT_4_32K_0613 = "azure/gpt-4-32k-0613"
    AZURE_GPT_4_TURBO = "azure/gpt-4-turbo"
    AZURE_GPT_4_TURBO_2024_04_09 = "azure/gpt-4-turbo-2024-04-09"
    AZURE_GPT_4_TURBO_VISION_PREVIEW = "azure/gpt-4-turbo-vision-preview"
    AZURE_GPT_4_1 = "azure/gpt-4.1"
    AZURE_GPT_4_1_2025_04_14 = "azure/gpt-4.1-2025-04-14"
    AZURE_GPT_4_1_MINI = "azure/gpt-4.1-mini"
    AZURE_GPT_4_1_MINI_2025_04_14 = "azure/gpt-4.1-mini-2025-04-14"
    AZURE_GPT_4_1_NANO = "azure/gpt-4.1-nano"
    AZURE_GPT_4_1_NANO_2025_04_14 = "azure/gpt-4.1-nano-2025-04-14"
    AZURE_GPT_4_5_PREVIEW = "azure/gpt-4.5-preview"
    AZURE_GPT_4O = "azure/gpt-4o"
    AZURE_GPT_4O_2024_05_13 = "azure/gpt-4o-2024-05-13"
    AZURE_GPT_4O_2024_08_06 = "azure/gpt-4o-2024-08-06"
    AZURE_GPT_4O_2024_11_20 = "azure/gpt-4o-2024-11-20"
    AZURE_GPT_4O_AUDIO_PREVIEW_2024_12_17 = "azure/gpt-4o-audio-preview-2024-12-17"
    AZURE_GPT_4O_MINI = "azure/gpt-4o-mini"
    AZURE_GPT_4O_MINI_2024_07_18 = "azure/gpt-4o-mini-2024-07-18"
    AZURE_GPT_4O_MINI_AUDIO_PREVIEW_2024_12_17 = "azure/gpt-4o-mini-audio-preview-2024-12-17"
    AZURE_GPT_4O_MINI_REALTIME_PREVIEW_2024_12_17 = "azure/gpt-4o-mini-realtime-preview-2024-12-17"
    AZURE_GPT_4O_MINI_TRANSCRIBE = "azure/gpt-4o-mini-transcribe"
    AZURE_GPT_4O_MINI_TTS = "azure/gpt-4o-mini-tts"
    AZURE_GPT_4O_REALTIME_PREVIEW_2024_10_01 = "azure/gpt-4o-realtime-preview-2024-10-01"
    AZURE_GPT_4O_REALTIME_PREVIEW_2024_12_17 = "azure/gpt-4o-realtime-preview-2024-12-17"
    AZURE_GPT_4O_TRANSCRIBE = "azure/gpt-4o-transcribe"
    AZURE_GPT_5 = "azure/gpt-5"
    AZURE_GPT_5_2025_08_07 = "azure/gpt-5-2025-08-07"
    AZURE_GPT_5_CHAT = "azure/gpt-5-chat"
    AZURE_GPT_5_CHAT_LATEST = "azure/gpt-5-chat-latest"
    AZURE_GPT_5_CODEX = "azure/gpt-5-codex"
    AZURE_GPT_5_MINI = "azure/gpt-5-mini"
    AZURE_GPT_5_MINI_2025_08_07 = "azure/gpt-5-mini-2025-08-07"
    AZURE_GPT_5_NANO = "azure/gpt-5-nano"
    AZURE_GPT_5_NANO_2025_08_07 = "azure/gpt-5-nano-2025-08-07"
    AZURE_GPT_5_PRO = "azure/gpt-5-pro"
    AZURE_GPT_IMAGE_1 = "azure/gpt-image-1"
    AZURE_GPT_IMAGE_1_MINI = "azure/gpt-image-1-mini"
    AZURE_HD_1024_X_1024_DALL_E_3 = "azure/hd/1024-x-1024/dall-e-3"
    AZURE_HD_1024_X_1792_DALL_E_3 = "azure/hd/1024-x-1792/dall-e-3"
    AZURE_HD_1792_X_1024_DALL_E_3 = "azure/hd/1792-x-1024/dall-e-3"
    AZURE_HIGH_1024_X_1024_GPT_IMAGE_1 = "azure/high/1024-x-1024/gpt-image-1"
    AZURE_HIGH_1024_X_1024_GPT_IMAGE_1_MINI = "azure/high/1024-x-1024/gpt-image-1-mini"
    AZURE_HIGH_1024_X_1536_GPT_IMAGE_1 = "azure/high/1024-x-1536/gpt-image-1"
    AZURE_HIGH_1024_X_1536_GPT_IMAGE_1_MINI = "azure/high/1024-x-1536/gpt-image-1-mini"
    AZURE_HIGH_1536_X_1024_GPT_IMAGE_1 = "azure/high/1536-x-1024/gpt-image-1"
    AZURE_HIGH_1536_X_1024_GPT_IMAGE_1_MINI = "azure/high/1536-x-1024/gpt-image-1-mini"
    AZURE_LOW_1024_X_1024_GPT_IMAGE_1 = "azure/low/1024-x-1024/gpt-image-1"
    AZURE_LOW_1024_X_1024_GPT_IMAGE_1_MINI = "azure/low/1024-x-1024/gpt-image-1-mini"
    AZURE_LOW_1024_X_1536_GPT_IMAGE_1 = "azure/low/1024-x-1536/gpt-image-1"
    AZURE_LOW_1024_X_1536_GPT_IMAGE_1_MINI = "azure/low/1024-x-1536/gpt-image-1-mini"
    AZURE_LOW_1536_X_1024_GPT_IMAGE_1 = "azure/low/1536-x-1024/gpt-image-1"
    AZURE_LOW_1536_X_1024_GPT_IMAGE_1_MINI = "azure/low/1536-x-1024/gpt-image-1-mini"
    AZURE_MEDIUM_1024_X_1024_GPT_IMAGE_1 = "azure/medium/1024-x-1024/gpt-image-1"
    AZURE_MEDIUM_1024_X_1024_GPT_IMAGE_1_MINI = "azure/medium/1024-x-1024/gpt-image-1-mini"
    AZURE_MEDIUM_1024_X_1536_GPT_IMAGE_1 = "azure/medium/1024-x-1536/gpt-image-1"
    AZURE_MEDIUM_1024_X_1536_GPT_IMAGE_1_MINI = "azure/medium/1024-x-1536/gpt-image-1-mini"
    AZURE_MEDIUM_1536_X_1024_GPT_IMAGE_1 = "azure/medium/1536-x-1024/gpt-image-1"
    AZURE_MEDIUM_1536_X_1024_GPT_IMAGE_1_MINI = "azure/medium/1536-x-1024/gpt-image-1-mini"
    AZURE_MISTRAL_LARGE_2402 = "azure/mistral-large-2402"
    AZURE_MISTRAL_LARGE_LATEST = "azure/mistral-large-latest"
    AZURE_O1 = "azure/o1"
    AZURE_O1_2024_12_17 = "azure/o1-2024-12-17"
    AZURE_O1_MINI = "azure/o1-mini"
    AZURE_O1_MINI_2024_09_12 = "azure/o1-mini-2024-09-12"
    AZURE_O1_PREVIEW = "azure/o1-preview"
    AZURE_O1_PREVIEW_2024_09_12 = "azure/o1-preview-2024-09-12"
    AZURE_O3 = "azure/o3"
    AZURE_O3_2025_04_16 = "azure/o3-2025-04-16"
    AZURE_O3_DEEP_RESEARCH = "azure/o3-deep-research"
    AZURE_O3_MINI = "azure/o3-mini"
    AZURE_O3_MINI_2025_01_31 = "azure/o3-mini-2025-01-31"
    AZURE_O3_PRO = "azure/o3-pro"
    AZURE_O3_PRO_2025_06_10 = "azure/o3-pro-2025-06-10"
    AZURE_O4_MINI = "azure/o4-mini"
    AZURE_O4_MINI_2025_04_16 = "azure/o4-mini-2025-04-16"
    AZURE_SORA_2 = "azure/sora-2"
    AZURE_SORA_2_PRO = "azure/sora-2-pro"
    AZURE_SORA_2_PRO_HIGH_RES = "azure/sora-2-pro-high-res"
    AZURE_SPEECH_AZURE_TTS = "azure/speech/azure-tts"
    AZURE_SPEECH_AZURE_TTS_HD = "azure/speech/azure-tts-hd"
    AZURE_STANDARD_1024_X_1024_DALL_E_2 = "azure/standard/1024-x-1024/dall-e-2"
    AZURE_STANDARD_1024_X_1024_DALL_E_3 = "azure/standard/1024-x-1024/dall-e-3"
    AZURE_STANDARD_1024_X_1792_DALL_E_3 = "azure/standard/1024-x-1792/dall-e-3"
    AZURE_STANDARD_1792_X_1024_DALL_E_3 = "azure/standard/1792-x-1024/dall-e-3"
    AZURE_TEXT_EMBEDDING_3_LARGE = "azure/text-embedding-3-large"
    AZURE_TEXT_EMBEDDING_3_SMALL = "azure/text-embedding-3-small"
    AZURE_TEXT_EMBEDDING_ADA_002 = "azure/text-embedding-ada-002"
    AZURE_TTS_1 = "azure/tts-1"
    AZURE_TTS_1_HD = "azure/tts-1-hd"
    AZURE_US_GPT_4O_2024_08_06 = "azure/us/gpt-4o-2024-08-06"
    AZURE_US_GPT_4O_2024_11_20 = "azure/us/gpt-4o-2024-11-20"
    AZURE_US_GPT_4O_MINI_2024_07_18 = "azure/us/gpt-4o-mini-2024-07-18"
    AZURE_US_GPT_4O_MINI_REALTIME_PREVIEW_2024_12_17 = "azure/us/gpt-4o-mini-realtime-preview-2024-12-17"
    AZURE_US_GPT_4O_REALTIME_PREVIEW_2024_10_01 = "azure/us/gpt-4o-realtime-preview-2024-10-01"
    AZURE_US_GPT_4O_REALTIME_PREVIEW_2024_12_17 = "azure/us/gpt-4o-realtime-preview-2024-12-17"
    AZURE_US_O1_2024_12_17 = "azure/us/o1-2024-12-17"
    AZURE_US_O1_MINI_2024_09_12 = "azure/us/o1-mini-2024-09-12"
    AZURE_US_O1_PREVIEW_2024_09_12 = "azure/us/o1-preview-2024-09-12"
    AZURE_US_O3_MINI_2025_01_31 = "azure/us/o3-mini-2025-01-31"
    AZURE_WHISPER_1 = "azure/whisper-1"

    # AZURE_AI models
    AZURE_AI_COHERE_EMBED_V3_ENGLISH = "azure_ai/Cohere-embed-v3-english"
    AZURE_AI_COHERE_EMBED_V3_MULTILINGUAL = "azure_ai/Cohere-embed-v3-multilingual"
    AZURE_AI_FLUX_1_1_PRO = "azure_ai/FLUX-1.1-pro"
    AZURE_AI_FLUX_1_KONTEXT_PRO = "azure_ai/FLUX.1-Kontext-pro"
    AZURE_AI_LLAMA_3_2_11B_VISION_INSTRUCT = "azure_ai/Llama-3.2-11B-Vision-Instruct"
    AZURE_AI_LLAMA_3_2_90B_VISION_INSTRUCT = "azure_ai/Llama-3.2-90B-Vision-Instruct"
    AZURE_AI_LLAMA_3_3_70B_INSTRUCT = "azure_ai/Llama-3.3-70B-Instruct"
    AZURE_AI_LLAMA_4_MAVERICK_17B_128E_INSTRUCT_FP8 = "azure_ai/Llama-4-Maverick-17B-128E-Instruct-FP8"
    AZURE_AI_LLAMA_4_SCOUT_17B_16E_INSTRUCT = "azure_ai/Llama-4-Scout-17B-16E-Instruct"
    AZURE_AI_MAI_DS_R1 = "azure_ai/MAI-DS-R1"
    AZURE_AI_META_LLAMA_3_70B_INSTRUCT = "azure_ai/Meta-Llama-3-70B-Instruct"
    AZURE_AI_META_LLAMA_3_1_405B_INSTRUCT = "azure_ai/Meta-Llama-3.1-405B-Instruct"
    AZURE_AI_META_LLAMA_3_1_70B_INSTRUCT = "azure_ai/Meta-Llama-3.1-70B-Instruct"
    AZURE_AI_META_LLAMA_3_1_8B_INSTRUCT = "azure_ai/Meta-Llama-3.1-8B-Instruct"
    AZURE_AI_PHI_3_MEDIUM_128K_INSTRUCT = "azure_ai/Phi-3-medium-128k-instruct"
    AZURE_AI_PHI_3_MEDIUM_4K_INSTRUCT = "azure_ai/Phi-3-medium-4k-instruct"
    AZURE_AI_PHI_3_MINI_128K_INSTRUCT = "azure_ai/Phi-3-mini-128k-instruct"
    AZURE_AI_PHI_3_MINI_4K_INSTRUCT = "azure_ai/Phi-3-mini-4k-instruct"
    AZURE_AI_PHI_3_SMALL_128K_INSTRUCT = "azure_ai/Phi-3-small-128k-instruct"
    AZURE_AI_PHI_3_SMALL_8K_INSTRUCT = "azure_ai/Phi-3-small-8k-instruct"
    AZURE_AI_PHI_3_5_MOE_INSTRUCT = "azure_ai/Phi-3.5-MoE-instruct"
    AZURE_AI_PHI_3_5_MINI_INSTRUCT = "azure_ai/Phi-3.5-mini-instruct"
    AZURE_AI_PHI_3_5_VISION_INSTRUCT = "azure_ai/Phi-3.5-vision-instruct"
    AZURE_AI_PHI_4 = "azure_ai/Phi-4"
    AZURE_AI_PHI_4_MINI_INSTRUCT = "azure_ai/Phi-4-mini-instruct"
    AZURE_AI_PHI_4_MINI_REASONING = "azure_ai/Phi-4-mini-reasoning"
    AZURE_AI_PHI_4_MULTIMODAL_INSTRUCT = "azure_ai/Phi-4-multimodal-instruct"
    AZURE_AI_PHI_4_REASONING = "azure_ai/Phi-4-reasoning"
    AZURE_AI_COHERE_RERANK_V3_ENGLISH = "azure_ai/cohere-rerank-v3-english"
    AZURE_AI_COHERE_RERANK_V3_MULTILINGUAL = "azure_ai/cohere-rerank-v3-multilingual"
    AZURE_AI_COHERE_RERANK_V3_5 = "azure_ai/cohere-rerank-v3.5"
    AZURE_AI_DEEPSEEK_R1 = "azure_ai/deepseek-r1"
    AZURE_AI_DEEPSEEK_V3 = "azure_ai/deepseek-v3"
    AZURE_AI_DEEPSEEK_V3_0324 = "azure_ai/deepseek-v3-0324"
    AZURE_AI_DOC_INTELLIGENCE_PREBUILT_DOCUMENT = "azure_ai/doc-intelligence/prebuilt-document"
    AZURE_AI_DOC_INTELLIGENCE_PREBUILT_LAYOUT = "azure_ai/doc-intelligence/prebuilt-layout"
    AZURE_AI_DOC_INTELLIGENCE_PREBUILT_READ = "azure_ai/doc-intelligence/prebuilt-read"
    AZURE_AI_EMBED_V_4_0 = "azure_ai/embed-v-4-0"
    AZURE_AI_GLOBAL_GROK_3 = "azure_ai/global/grok-3"
    AZURE_AI_GLOBAL_GROK_3_MINI = "azure_ai/global/grok-3-mini"
    AZURE_AI_GROK_3 = "azure_ai/grok-3"
    AZURE_AI_GROK_3_MINI = "azure_ai/grok-3-mini"
    AZURE_AI_GROK_4 = "azure_ai/grok-4"
    AZURE_AI_GROK_4_FAST_NON_REASONING = "azure_ai/grok-4-fast-non-reasoning"
    AZURE_AI_GROK_4_FAST_REASONING = "azure_ai/grok-4-fast-reasoning"
    AZURE_AI_GROK_CODE_FAST_1 = "azure_ai/grok-code-fast-1"
    AZURE_AI_JAIS_30B_CHAT = "azure_ai/jais-30b-chat"
    AZURE_AI_JAMBA_INSTRUCT = "azure_ai/jamba-instruct"
    AZURE_AI_MINISTRAL_3B = "azure_ai/ministral-3b"
    AZURE_AI_MISTRAL_DOCUMENT_AI_2505 = "azure_ai/mistral-document-ai-2505"
    AZURE_AI_MISTRAL_LARGE = "azure_ai/mistral-large"
    AZURE_AI_MISTRAL_LARGE_2407 = "azure_ai/mistral-large-2407"
    AZURE_AI_MISTRAL_LARGE_LATEST = "azure_ai/mistral-large-latest"
    AZURE_AI_MISTRAL_MEDIUM_2505 = "azure_ai/mistral-medium-2505"
    AZURE_AI_MISTRAL_NEMO = "azure_ai/mistral-nemo"
    AZURE_AI_MISTRAL_SMALL = "azure_ai/mistral-small"
    AZURE_AI_MISTRAL_SMALL_2503 = "azure_ai/mistral-small-2503"

    # BEDROCK models
    BEDROCK_US_ANTHROPIC_CLAUDE_3_5_HAIKU_20241022_V1_0 = "bedrock/us.anthropic.claude-3-5-haiku-20241022-v1:0"

    # CEREBRAS models
    CEREBRAS_LLAMA_3_3_70B = "cerebras/llama-3.3-70b"
    CEREBRAS_LLAMA3_1_70B = "cerebras/llama3.1-70b"
    CEREBRAS_LLAMA3_1_8B = "cerebras/llama3.1-8b"
    CEREBRAS_OPENAI_GPT_OSS_120B = "cerebras/openai/gpt-oss-120b"
    CEREBRAS_QWEN_3_32B = "cerebras/qwen-3-32b"

    # CLOUDFLARE models
    CLOUDFLARE_CF_META_LLAMA_2_7B_CHAT_FP16 = "cloudflare/@cf/meta/llama-2-7b-chat-fp16"
    CLOUDFLARE_CF_META_LLAMA_2_7B_CHAT_INT8 = "cloudflare/@cf/meta/llama-2-7b-chat-int8"
    CLOUDFLARE_CF_MISTRAL_MISTRAL_7B_INSTRUCT_V0_1 = "cloudflare/@cf/mistral/mistral-7b-instruct-v0.1"
    CLOUDFLARE_HF_THEBLOKE_CODELLAMA_7B_INSTRUCT_AWQ = "cloudflare/@hf/thebloke/codellama-7b-instruct-awq"

    # CODESTRAL models
    CODESTRAL_CODESTRAL_2405 = "codestral/codestral-2405"
    CODESTRAL_CODESTRAL_LATEST = "codestral/codestral-latest"

    # COHERE models
    COHERE_COMMAND_LIGHT_TEXT_V14 = "cohere.command-light-text-v14"
    COHERE_COMMAND_R_PLUS_V1_0 = "cohere.command-r-plus-v1:0"
    COHERE_COMMAND_R_V1_0 = "cohere.command-r-v1:0"
    COHERE_COMMAND_TEXT_V14 = "cohere.command-text-v14"
    COHERE_EMBED_ENGLISH_V3 = "cohere.embed-english-v3"
    COHERE_EMBED_MULTILINGUAL_V3 = "cohere.embed-multilingual-v3"
    COHERE_EMBED_V4_0 = "cohere.embed-v4:0"
    COHERE_RERANK_V3_5_0 = "cohere.rerank-v3-5:0"
    COMMAND = "command"
    COMMAND_A_03_2025 = "command-a-03-2025"
    COMMAND_LIGHT = "command-light"
    COMMAND_NIGHTLY = "command-nightly"
    COMMAND_R = "command-r"
    COMMAND_R_08_2024 = "command-r-08-2024"
    COMMAND_R_PLUS = "command-r-plus"
    COMMAND_R_PLUS_08_2024 = "command-r-plus-08-2024"
    COMMAND_R7B_12_2024 = "command-r7b-12-2024"

    # DAANELSON models
    DAANELSON_FLAN_T5_LARGE_CE962B3F6792A57074A601D3979DB5839697ADD2E4E02696B3CED4C022D4767F = "daanelson/flan-t5-large:ce962b3f6792a57074a601d3979db5839697add2e4e02696b3ced4c022d4767f"

    # DASHSCOPE models
    DASHSCOPE_QWEN_CODER = "dashscope/qwen-coder"
    DASHSCOPE_QWEN_FLASH = "dashscope/qwen-flash"
    DASHSCOPE_QWEN_FLASH_2025_07_28 = "dashscope/qwen-flash-2025-07-28"
    DASHSCOPE_QWEN_MAX = "dashscope/qwen-max"
    DASHSCOPE_QWEN_PLUS = "dashscope/qwen-plus"
    DASHSCOPE_QWEN_PLUS_2025_01_25 = "dashscope/qwen-plus-2025-01-25"
    DASHSCOPE_QWEN_PLUS_2025_04_28 = "dashscope/qwen-plus-2025-04-28"
    DASHSCOPE_QWEN_PLUS_2025_07_14 = "dashscope/qwen-plus-2025-07-14"
    DASHSCOPE_QWEN_PLUS_2025_07_28 = "dashscope/qwen-plus-2025-07-28"
    DASHSCOPE_QWEN_PLUS_2025_09_11 = "dashscope/qwen-plus-2025-09-11"
    DASHSCOPE_QWEN_PLUS_LATEST = "dashscope/qwen-plus-latest"
    DASHSCOPE_QWEN_TURBO = "dashscope/qwen-turbo"
    DASHSCOPE_QWEN_TURBO_2024_11_01 = "dashscope/qwen-turbo-2024-11-01"
    DASHSCOPE_QWEN_TURBO_2025_04_28 = "dashscope/qwen-turbo-2025-04-28"
    DASHSCOPE_QWEN_TURBO_LATEST = "dashscope/qwen-turbo-latest"
    DASHSCOPE_QWEN3_30B_A3B = "dashscope/qwen3-30b-a3b"
    DASHSCOPE_QWEN3_CODER_FLASH = "dashscope/qwen3-coder-flash"
    DASHSCOPE_QWEN3_CODER_FLASH_2025_07_28 = "dashscope/qwen3-coder-flash-2025-07-28"
    DASHSCOPE_QWEN3_CODER_PLUS = "dashscope/qwen3-coder-plus"
    DASHSCOPE_QWEN3_CODER_PLUS_2025_07_22 = "dashscope/qwen3-coder-plus-2025-07-22"
    DASHSCOPE_QWEN3_MAX_PREVIEW = "dashscope/qwen3-max-preview"
    DASHSCOPE_QWQ_PLUS = "dashscope/qwq-plus"

    # DATABRICKS models
    DATABRICKS_DATABRICKS_BGE_LARGE_EN = "databricks/databricks-bge-large-en"
    DATABRICKS_DATABRICKS_CLAUDE_3_7_SONNET = "databricks/databricks-claude-3-7-sonnet"
    DATABRICKS_DATABRICKS_GTE_LARGE_EN = "databricks/databricks-gte-large-en"
    DATABRICKS_DATABRICKS_LLAMA_2_70B_CHAT = "databricks/databricks-llama-2-70b-chat"
    DATABRICKS_DATABRICKS_LLAMA_4_MAVERICK = "databricks/databricks-llama-4-maverick"
    DATABRICKS_DATABRICKS_META_LLAMA_3_1_405B_INSTRUCT = "databricks/databricks-meta-llama-3-1-405b-instruct"
    DATABRICKS_DATABRICKS_META_LLAMA_3_3_70B_INSTRUCT = "databricks/databricks-meta-llama-3-3-70b-instruct"
    DATABRICKS_DATABRICKS_META_LLAMA_3_70B_INSTRUCT = "databricks/databricks-meta-llama-3-70b-instruct"
    DATABRICKS_DATABRICKS_MIXTRAL_8X7B_INSTRUCT = "databricks/databricks-mixtral-8x7b-instruct"
    DATABRICKS_DATABRICKS_MPT_30B_INSTRUCT = "databricks/databricks-mpt-30b-instruct"
    DATABRICKS_DATABRICKS_MPT_7B_INSTRUCT = "databricks/databricks-mpt-7b-instruct"

    # DEEPGRAM models
    DEEPGRAM_BASE = "deepgram/base"
    DEEPGRAM_BASE_CONVERSATIONALAI = "deepgram/base-conversationalai"
    DEEPGRAM_BASE_FINANCE = "deepgram/base-finance"
    DEEPGRAM_BASE_GENERAL = "deepgram/base-general"
    DEEPGRAM_BASE_MEETING = "deepgram/base-meeting"
    DEEPGRAM_BASE_PHONECALL = "deepgram/base-phonecall"
    DEEPGRAM_BASE_VIDEO = "deepgram/base-video"
    DEEPGRAM_BASE_VOICEMAIL = "deepgram/base-voicemail"
    DEEPGRAM_ENHANCED = "deepgram/enhanced"
    DEEPGRAM_ENHANCED_FINANCE = "deepgram/enhanced-finance"
    DEEPGRAM_ENHANCED_GENERAL = "deepgram/enhanced-general"
    DEEPGRAM_ENHANCED_MEETING = "deepgram/enhanced-meeting"
    DEEPGRAM_ENHANCED_PHONECALL = "deepgram/enhanced-phonecall"
    DEEPGRAM_NOVA = "deepgram/nova"
    DEEPGRAM_NOVA_2 = "deepgram/nova-2"
    DEEPGRAM_NOVA_2_ATC = "deepgram/nova-2-atc"
    DEEPGRAM_NOVA_2_AUTOMOTIVE = "deepgram/nova-2-automotive"
    DEEPGRAM_NOVA_2_CONVERSATIONALAI = "deepgram/nova-2-conversationalai"
    DEEPGRAM_NOVA_2_DRIVETHRU = "deepgram/nova-2-drivethru"
    DEEPGRAM_NOVA_2_FINANCE = "deepgram/nova-2-finance"
    DEEPGRAM_NOVA_2_GENERAL = "deepgram/nova-2-general"
    DEEPGRAM_NOVA_2_MEETING = "deepgram/nova-2-meeting"
    DEEPGRAM_NOVA_2_PHONECALL = "deepgram/nova-2-phonecall"
    DEEPGRAM_NOVA_2_VIDEO = "deepgram/nova-2-video"
    DEEPGRAM_NOVA_2_VOICEMAIL = "deepgram/nova-2-voicemail"
    DEEPGRAM_NOVA_3 = "deepgram/nova-3"
    DEEPGRAM_NOVA_3_GENERAL = "deepgram/nova-3-general"
    DEEPGRAM_NOVA_3_MEDICAL = "deepgram/nova-3-medical"
    DEEPGRAM_NOVA_GENERAL = "deepgram/nova-general"
    DEEPGRAM_NOVA_PHONECALL = "deepgram/nova-phonecall"
    DEEPGRAM_WHISPER = "deepgram/whisper"
    DEEPGRAM_WHISPER_BASE = "deepgram/whisper-base"
    DEEPGRAM_WHISPER_LARGE = "deepgram/whisper-large"
    DEEPGRAM_WHISPER_MEDIUM = "deepgram/whisper-medium"
    DEEPGRAM_WHISPER_SMALL = "deepgram/whisper-small"
    DEEPGRAM_WHISPER_TINY = "deepgram/whisper-tiny"

    # DEEPINFRA models
    DEEPINFRA_GRYPHE_MYTHOMAX_L2_13B = "deepinfra/Gryphe/MythoMax-L2-13b"
    DEEPINFRA_NOUSRESEARCH_HERMES_3_LLAMA_3_1_405B = "deepinfra/NousResearch/Hermes-3-Llama-3.1-405B"
    DEEPINFRA_NOUSRESEARCH_HERMES_3_LLAMA_3_1_70B = "deepinfra/NousResearch/Hermes-3-Llama-3.1-70B"
    DEEPINFRA_QWEN_QWQ_32B = "deepinfra/Qwen/QwQ-32B"
    DEEPINFRA_QWEN_QWEN2_5_72B_INSTRUCT = "deepinfra/Qwen/Qwen2.5-72B-Instruct"
    DEEPINFRA_QWEN_QWEN2_5_7B_INSTRUCT = "deepinfra/Qwen/Qwen2.5-7B-Instruct"
    DEEPINFRA_QWEN_QWEN2_5_VL_32B_INSTRUCT = "deepinfra/Qwen/Qwen2.5-VL-32B-Instruct"
    DEEPINFRA_QWEN_QWEN3_14B = "deepinfra/Qwen/Qwen3-14B"
    DEEPINFRA_QWEN_QWEN3_235B_A22B = "deepinfra/Qwen/Qwen3-235B-A22B"
    DEEPINFRA_QWEN_QWEN3_235B_A22B_INSTRUCT_2507 = "deepinfra/Qwen/Qwen3-235B-A22B-Instruct-2507"
    DEEPINFRA_QWEN_QWEN3_235B_A22B_THINKING_2507 = "deepinfra/Qwen/Qwen3-235B-A22B-Thinking-2507"
    DEEPINFRA_QWEN_QWEN3_30B_A3B = "deepinfra/Qwen/Qwen3-30B-A3B"
    DEEPINFRA_QWEN_QWEN3_32B = "deepinfra/Qwen/Qwen3-32B"
    DEEPINFRA_QWEN_QWEN3_CODER_480B_A35B_INSTRUCT = "deepinfra/Qwen/Qwen3-Coder-480B-A35B-Instruct"
    DEEPINFRA_QWEN_QWEN3_CODER_480B_A35B_INSTRUCT_TURBO = "deepinfra/Qwen/Qwen3-Coder-480B-A35B-Instruct-Turbo"
    DEEPINFRA_QWEN_QWEN3_NEXT_80B_A3B_INSTRUCT = "deepinfra/Qwen/Qwen3-Next-80B-A3B-Instruct"
    DEEPINFRA_QWEN_QWEN3_NEXT_80B_A3B_THINKING = "deepinfra/Qwen/Qwen3-Next-80B-A3B-Thinking"
    DEEPINFRA_SAO10K_L3_8B_LUNARIS_V1_TURBO = "deepinfra/Sao10K/L3-8B-Lunaris-v1-Turbo"
    DEEPINFRA_SAO10K_L3_1_70B_EURYALE_V2_2 = "deepinfra/Sao10K/L3.1-70B-Euryale-v2.2"
    DEEPINFRA_SAO10K_L3_3_70B_EURYALE_V2_3 = "deepinfra/Sao10K/L3.3-70B-Euryale-v2.3"
    DEEPINFRA_ALLENAI_OLMOCR_7B_0725_FP8 = "deepinfra/allenai/olmOCR-7B-0725-FP8"
    DEEPINFRA_ANTHROPIC_CLAUDE_3_7_SONNET_LATEST = "deepinfra/anthropic/claude-3-7-sonnet-latest"
    DEEPINFRA_ANTHROPIC_CLAUDE_4_OPUS = "deepinfra/anthropic/claude-4-opus"
    DEEPINFRA_ANTHROPIC_CLAUDE_4_SONNET = "deepinfra/anthropic/claude-4-sonnet"
    DEEPINFRA_DEEPSEEK_AI_DEEPSEEK_R1 = "deepinfra/deepseek-ai/DeepSeek-R1"
    DEEPINFRA_DEEPSEEK_AI_DEEPSEEK_R1_0528 = "deepinfra/deepseek-ai/DeepSeek-R1-0528"
    DEEPINFRA_DEEPSEEK_AI_DEEPSEEK_R1_0528_TURBO = "deepinfra/deepseek-ai/DeepSeek-R1-0528-Turbo"
    DEEPINFRA_DEEPSEEK_AI_DEEPSEEK_R1_DISTILL_LLAMA_70B = "deepinfra/deepseek-ai/DeepSeek-R1-Distill-Llama-70B"
    DEEPINFRA_DEEPSEEK_AI_DEEPSEEK_R1_DISTILL_QWEN_32B = "deepinfra/deepseek-ai/DeepSeek-R1-Distill-Qwen-32B"
    DEEPINFRA_DEEPSEEK_AI_DEEPSEEK_R1_TURBO = "deepinfra/deepseek-ai/DeepSeek-R1-Turbo"
    DEEPINFRA_DEEPSEEK_AI_DEEPSEEK_V3 = "deepinfra/deepseek-ai/DeepSeek-V3"
    DEEPINFRA_DEEPSEEK_AI_DEEPSEEK_V3_0324 = "deepinfra/deepseek-ai/DeepSeek-V3-0324"
    DEEPINFRA_DEEPSEEK_AI_DEEPSEEK_V3_1 = "deepinfra/deepseek-ai/DeepSeek-V3.1"
    DEEPINFRA_DEEPSEEK_AI_DEEPSEEK_V3_1_TERMINUS = "deepinfra/deepseek-ai/DeepSeek-V3.1-Terminus"
    DEEPINFRA_GOOGLE_GEMINI_2_0_FLASH_001 = "deepinfra/google/gemini-2.0-flash-001"
    DEEPINFRA_GOOGLE_GEMINI_2_5_FLASH = "deepinfra/google/gemini-2.5-flash"
    DEEPINFRA_GOOGLE_GEMINI_2_5_PRO = "deepinfra/google/gemini-2.5-pro"
    DEEPINFRA_GOOGLE_GEMMA_3_12B_IT = "deepinfra/google/gemma-3-12b-it"
    DEEPINFRA_GOOGLE_GEMMA_3_27B_IT = "deepinfra/google/gemma-3-27b-it"
    DEEPINFRA_GOOGLE_GEMMA_3_4B_IT = "deepinfra/google/gemma-3-4b-it"
    DEEPINFRA_META_LLAMA_LLAMA_3_2_11B_VISION_INSTRUCT = "deepinfra/meta-llama/Llama-3.2-11B-Vision-Instruct"
    DEEPINFRA_META_LLAMA_LLAMA_3_2_3B_INSTRUCT = "deepinfra/meta-llama/Llama-3.2-3B-Instruct"
    DEEPINFRA_META_LLAMA_LLAMA_3_3_70B_INSTRUCT = "deepinfra/meta-llama/Llama-3.3-70B-Instruct"
    DEEPINFRA_META_LLAMA_LLAMA_3_3_70B_INSTRUCT_TURBO = "deepinfra/meta-llama/Llama-3.3-70B-Instruct-Turbo"
    DEEPINFRA_META_LLAMA_LLAMA_4_MAVERICK_17B_128E_INSTRUCT_FP8 = "deepinfra/meta-llama/Llama-4-Maverick-17B-128E-Instruct-FP8"
    DEEPINFRA_META_LLAMA_LLAMA_4_SCOUT_17B_16E_INSTRUCT = "deepinfra/meta-llama/Llama-4-Scout-17B-16E-Instruct"
    DEEPINFRA_META_LLAMA_LLAMA_GUARD_3_8B = "deepinfra/meta-llama/Llama-Guard-3-8B"
    DEEPINFRA_META_LLAMA_LLAMA_GUARD_4_12B = "deepinfra/meta-llama/Llama-Guard-4-12B"
    DEEPINFRA_META_LLAMA_META_LLAMA_3_8B_INSTRUCT = "deepinfra/meta-llama/Meta-Llama-3-8B-Instruct"
    DEEPINFRA_META_LLAMA_META_LLAMA_3_1_70B_INSTRUCT = "deepinfra/meta-llama/Meta-Llama-3.1-70B-Instruct"
    DEEPINFRA_META_LLAMA_META_LLAMA_3_1_70B_INSTRUCT_TURBO = "deepinfra/meta-llama/Meta-Llama-3.1-70B-Instruct-Turbo"
    DEEPINFRA_META_LLAMA_META_LLAMA_3_1_8B_INSTRUCT = "deepinfra/meta-llama/Meta-Llama-3.1-8B-Instruct"
    DEEPINFRA_META_LLAMA_META_LLAMA_3_1_8B_INSTRUCT_TURBO = "deepinfra/meta-llama/Meta-Llama-3.1-8B-Instruct-Turbo"
    DEEPINFRA_MICROSOFT_WIZARDLM_2_8X22B = "deepinfra/microsoft/WizardLM-2-8x22B"
    DEEPINFRA_MICROSOFT_PHI_4 = "deepinfra/microsoft/phi-4"
    DEEPINFRA_MISTRALAI_MISTRAL_NEMO_INSTRUCT_2407 = "deepinfra/mistralai/Mistral-Nemo-Instruct-2407"
    DEEPINFRA_MISTRALAI_MISTRAL_SMALL_24B_INSTRUCT_2501 = "deepinfra/mistralai/Mistral-Small-24B-Instruct-2501"
    DEEPINFRA_MISTRALAI_MISTRAL_SMALL_3_2_24B_INSTRUCT_2506 = "deepinfra/mistralai/Mistral-Small-3.2-24B-Instruct-2506"
    DEEPINFRA_MISTRALAI_MIXTRAL_8X7B_INSTRUCT_V0_1 = "deepinfra/mistralai/Mixtral-8x7B-Instruct-v0.1"
    DEEPINFRA_MOONSHOTAI_KIMI_K2_INSTRUCT = "deepinfra/moonshotai/Kimi-K2-Instruct"
    DEEPINFRA_MOONSHOTAI_KIMI_K2_INSTRUCT_0905 = "deepinfra/moonshotai/Kimi-K2-Instruct-0905"
    DEEPINFRA_NVIDIA_LLAMA_3_1_NEMOTRON_70B_INSTRUCT = "deepinfra/nvidia/Llama-3.1-Nemotron-70B-Instruct"
    DEEPINFRA_NVIDIA_LLAMA_3_3_NEMOTRON_SUPER_49B_V1_5 = "deepinfra/nvidia/Llama-3.3-Nemotron-Super-49B-v1.5"
    DEEPINFRA_NVIDIA_NVIDIA_NEMOTRON_NANO_9B_V2 = "deepinfra/nvidia/NVIDIA-Nemotron-Nano-9B-v2"
    DEEPINFRA_OPENAI_GPT_OSS_120B = "deepinfra/openai/gpt-oss-120b"
    DEEPINFRA_OPENAI_GPT_OSS_20B = "deepinfra/openai/gpt-oss-20b"
    DEEPINFRA_ZAI_ORG_GLM_4_5 = "deepinfra/zai-org/GLM-4.5"

    # DEEPSEEK models
    DEEPSEEK_CHAT = "deepseek-chat"
    DEEPSEEK_REASONER = "deepseek-reasoner"
    DEEPSEEK_DEEPSEEK_CHAT = "deepseek/deepseek-chat"
    DEEPSEEK_DEEPSEEK_CODER = "deepseek/deepseek-coder"
    DEEPSEEK_DEEPSEEK_R1 = "deepseek/deepseek-r1"
    DEEPSEEK_DEEPSEEK_REASONER = "deepseek/deepseek-reasoner"
    DEEPSEEK_DEEPSEEK_V3 = "deepseek/deepseek-v3"

    # DEFOG models
    DEFOG_SQLCODER = "defog/sqlcoder"

    # ELEVENLABS models
    ELEVENLABS_SCRIBE_V1 = "elevenlabs/scribe_v1"
    ELEVENLABS_SCRIBE_V1_EXPERIMENTAL = "elevenlabs/scribe_v1_experimental"

    # FEATHERLESS_AI models
    FEATHERLESS_AI_FEATHERLESS_AI_QWERKY_72B = "featherless_ai/featherless-ai/Qwerky-72B"
    FEATHERLESS_AI_FEATHERLESS_AI_QWERKY_QWQ_32B = "featherless_ai/featherless-ai/Qwerky-QwQ-32B"

    # FRIENDLIAI models
    FRIENDLIAI_META_LLAMA_3_1_70B_INSTRUCT = "friendliai/meta-llama-3.1-70b-instruct"
    FRIENDLIAI_META_LLAMA_3_1_8B_INSTRUCT = "friendliai/meta-llama-3.1-8b-instruct"

    # GEMINI models
    CHAT_BISON = "chat-bison"
    CHAT_BISON_32K = "chat-bison-32k"
    CHAT_BISON_32K_002 = "chat-bison-32k@002"
    CHAT_BISON_001 = "chat-bison@001"
    CHAT_BISON_002 = "chat-bison@002"
    GEMINI_1_0_PRO = "gemini-1.0-pro"
    GEMINI_1_0_PRO_001 = "gemini-1.0-pro-001"
    GEMINI_1_0_PRO_002 = "gemini-1.0-pro-002"
    GEMINI_1_0_ULTRA = "gemini-1.0-ultra"
    GEMINI_1_0_ULTRA_001 = "gemini-1.0-ultra-001"
    GEMINI_1_5_FLASH = "gemini-1.5-flash"
    GEMINI_1_5_FLASH_001 = "gemini-1.5-flash-001"
    GEMINI_1_5_FLASH_002 = "gemini-1.5-flash-002"
    GEMINI_1_5_FLASH_EXP_0827 = "gemini-1.5-flash-exp-0827"
    GEMINI_1_5_FLASH_PREVIEW_0514 = "gemini-1.5-flash-preview-0514"
    GEMINI_1_5_PRO = "gemini-1.5-pro"
    GEMINI_1_5_PRO_001 = "gemini-1.5-pro-001"
    GEMINI_1_5_PRO_002 = "gemini-1.5-pro-002"
    GEMINI_1_5_PRO_PREVIEW_0215 = "gemini-1.5-pro-preview-0215"
    GEMINI_1_5_PRO_PREVIEW_0409 = "gemini-1.5-pro-preview-0409"
    GEMINI_1_5_PRO_PREVIEW_0514 = "gemini-1.5-pro-preview-0514"
    GEMINI_2_0_FLASH = "gemini-2.0-flash"
    GEMINI_2_0_FLASH_001 = "gemini-2.0-flash-001"
    GEMINI_2_0_FLASH_EXP = "gemini-2.0-flash-exp"
    GEMINI_2_0_FLASH_LITE = "gemini-2.0-flash-lite"
    GEMINI_2_0_FLASH_LITE_001 = "gemini-2.0-flash-lite-001"
    GEMINI_2_0_FLASH_LIVE_PREVIEW_04_09 = "gemini-2.0-flash-live-preview-04-09"
    GEMINI_2_0_FLASH_PREVIEW_IMAGE_GENERATION = "gemini-2.0-flash-preview-image-generation"
    GEMINI_2_0_FLASH_THINKING_EXP = "gemini-2.0-flash-thinking-exp"
    GEMINI_2_0_FLASH_THINKING_EXP_01_21 = "gemini-2.0-flash-thinking-exp-01-21"
    GEMINI_2_0_PRO_EXP_02_05 = "gemini-2.0-pro-exp-02-05"
    GEMINI_2_5_FLASH = "gemini-2.5-flash"
    GEMINI_2_5_FLASH_IMAGE = "gemini-2.5-flash-image"
    GEMINI_2_5_FLASH_IMAGE_PREVIEW = "gemini-2.5-flash-image-preview"
    GEMINI_2_5_FLASH_LITE = "gemini-2.5-flash-lite"
    GEMINI_2_5_FLASH_LITE_PREVIEW_06_17 = "gemini-2.5-flash-lite-preview-06-17"
    GEMINI_2_5_FLASH_LITE_PREVIEW_09_2025 = "gemini-2.5-flash-lite-preview-09-2025"
    GEMINI_2_5_FLASH_PREVIEW_04_17 = "gemini-2.5-flash-preview-04-17"
    GEMINI_2_5_FLASH_PREVIEW_05_20 = "gemini-2.5-flash-preview-05-20"
    GEMINI_2_5_FLASH_PREVIEW_09_2025 = "gemini-2.5-flash-preview-09-2025"
    GEMINI_2_5_PRO = "gemini-2.5-pro"
    GEMINI_2_5_PRO_EXP_03_25 = "gemini-2.5-pro-exp-03-25"
    GEMINI_2_5_PRO_PREVIEW_03_25 = "gemini-2.5-pro-preview-03-25"
    GEMINI_2_5_PRO_PREVIEW_05_06 = "gemini-2.5-pro-preview-05-06"
    GEMINI_2_5_PRO_PREVIEW_06_05 = "gemini-2.5-pro-preview-06-05"
    GEMINI_2_5_PRO_PREVIEW_TTS = "gemini-2.5-pro-preview-tts"
    GEMINI_FLASH_EXPERIMENTAL = "gemini-flash-experimental"
    GEMINI_LIVE_2_5_FLASH_PREVIEW_NATIVE_AUDIO_09_2025 = "gemini-live-2.5-flash-preview-native-audio-09-2025"
    GEMINI_PRO = "gemini-pro"
    GEMINI_PRO_EXPERIMENTAL = "gemini-pro-experimental"
    GEMINI__GEMINI_1_5_FLASH = "gemini/gemini-1.5-flash"
    GEMINI__GEMINI_1_5_FLASH_001 = "gemini/gemini-1.5-flash-001"
    GEMINI__GEMINI_1_5_FLASH_002 = "gemini/gemini-1.5-flash-002"
    GEMINI__GEMINI_1_5_FLASH_8B = "gemini/gemini-1.5-flash-8b"
    GEMINI__GEMINI_1_5_FLASH_8B_EXP_0827 = "gemini/gemini-1.5-flash-8b-exp-0827"
    GEMINI__GEMINI_1_5_FLASH_8B_EXP_0924 = "gemini/gemini-1.5-flash-8b-exp-0924"
    GEMINI__GEMINI_1_5_FLASH_EXP_0827 = "gemini/gemini-1.5-flash-exp-0827"
    GEMINI__GEMINI_1_5_FLASH_LATEST = "gemini/gemini-1.5-flash-latest"
    GEMINI__GEMINI_1_5_PRO = "gemini/gemini-1.5-pro"
    GEMINI__GEMINI_1_5_PRO_001 = "gemini/gemini-1.5-pro-001"
    GEMINI__GEMINI_1_5_PRO_002 = "gemini/gemini-1.5-pro-002"
    GEMINI__GEMINI_1_5_PRO_EXP_0801 = "gemini/gemini-1.5-pro-exp-0801"
    GEMINI__GEMINI_1_5_PRO_EXP_0827 = "gemini/gemini-1.5-pro-exp-0827"
    GEMINI__GEMINI_1_5_PRO_LATEST = "gemini/gemini-1.5-pro-latest"
    GEMINI__GEMINI_2_0_FLASH = "gemini/gemini-2.0-flash"
    GEMINI__GEMINI_2_0_FLASH_001 = "gemini/gemini-2.0-flash-001"
    GEMINI__GEMINI_2_0_FLASH_EXP = "gemini/gemini-2.0-flash-exp"
    GEMINI__GEMINI_2_0_FLASH_LITE = "gemini/gemini-2.0-flash-lite"
    GEMINI__GEMINI_2_0_FLASH_LITE_PREVIEW_02_05 = "gemini/gemini-2.0-flash-lite-preview-02-05"
    GEMINI__GEMINI_2_0_FLASH_LIVE_001 = "gemini/gemini-2.0-flash-live-001"
    GEMINI__GEMINI_2_0_FLASH_PREVIEW_IMAGE_GENERATION = "gemini/gemini-2.0-flash-preview-image-generation"
    GEMINI__GEMINI_2_0_FLASH_THINKING_EXP = "gemini/gemini-2.0-flash-thinking-exp"
    GEMINI__GEMINI_2_0_FLASH_THINKING_EXP_01_21 = "gemini/gemini-2.0-flash-thinking-exp-01-21"
    GEMINI__GEMINI_2_0_PRO_EXP_02_05 = "gemini/gemini-2.0-pro-exp-02-05"
    GEMINI__GEMINI_2_5_FLASH = "gemini/gemini-2.5-flash"
    GEMINI__GEMINI_2_5_FLASH_IMAGE = "gemini/gemini-2.5-flash-image"
    GEMINI__GEMINI_2_5_FLASH_IMAGE_PREVIEW = "gemini/gemini-2.5-flash-image-preview"
    GEMINI__GEMINI_2_5_FLASH_LITE = "gemini/gemini-2.5-flash-lite"
    GEMINI__GEMINI_2_5_FLASH_LITE_PREVIEW_06_17 = "gemini/gemini-2.5-flash-lite-preview-06-17"
    GEMINI__GEMINI_2_5_FLASH_LITE_PREVIEW_09_2025 = "gemini/gemini-2.5-flash-lite-preview-09-2025"
    GEMINI__GEMINI_2_5_FLASH_PREVIEW_04_17 = "gemini/gemini-2.5-flash-preview-04-17"
    GEMINI__GEMINI_2_5_FLASH_PREVIEW_05_20 = "gemini/gemini-2.5-flash-preview-05-20"
    GEMINI__GEMINI_2_5_FLASH_PREVIEW_09_2025 = "gemini/gemini-2.5-flash-preview-09-2025"
    GEMINI__GEMINI_2_5_FLASH_PREVIEW_TTS = "gemini/gemini-2.5-flash-preview-tts"
    GEMINI__GEMINI_2_5_PRO = "gemini/gemini-2.5-pro"
    GEMINI__GEMINI_2_5_PRO_EXP_03_25 = "gemini/gemini-2.5-pro-exp-03-25"
    GEMINI__GEMINI_2_5_PRO_PREVIEW_03_25 = "gemini/gemini-2.5-pro-preview-03-25"
    GEMINI__GEMINI_2_5_PRO_PREVIEW_05_06 = "gemini/gemini-2.5-pro-preview-05-06"
    GEMINI__GEMINI_2_5_PRO_PREVIEW_06_05 = "gemini/gemini-2.5-pro-preview-06-05"
    GEMINI__GEMINI_2_5_PRO_PREVIEW_TTS = "gemini/gemini-2.5-pro-preview-tts"
    GEMINI__GEMINI_EMBEDDING_001 = "gemini/gemini-embedding-001"
    GEMINI__GEMINI_EXP_1114 = "gemini/gemini-exp-1114"
    GEMINI__GEMINI_EXP_1206 = "gemini/gemini-exp-1206"
    GEMINI__GEMINI_FLASH_LATEST = "gemini/gemini-flash-latest"
    GEMINI__GEMINI_FLASH_LITE_LATEST = "gemini/gemini-flash-lite-latest"
    GEMINI__GEMINI_GEMMA_2_27B_IT = "gemini/gemini-gemma-2-27b-it"
    GEMINI__GEMINI_GEMMA_2_9B_IT = "gemini/gemini-gemma-2-9b-it"
    GEMINI__GEMINI_LIVE_2_5_FLASH_PREVIEW_NATIVE_AUDIO_09_2025 = "gemini/gemini-live-2.5-flash-preview-native-audio-09-2025"
    GEMINI__GEMINI_PRO = "gemini/gemini-pro"
    GEMINI__GEMINI_PRO_VISION = "gemini/gemini-pro-vision"
    GEMINI__GEMMA_3_27B_IT = "gemini/gemma-3-27b-it"
    GEMINI__IMAGEN_3_0_FAST_GENERATE_001 = "gemini/imagen-3.0-fast-generate-001"
    GEMINI__IMAGEN_3_0_GENERATE_001 = "gemini/imagen-3.0-generate-001"
    GEMINI__IMAGEN_3_0_GENERATE_002 = "gemini/imagen-3.0-generate-002"
    GEMINI__IMAGEN_4_0_FAST_GENERATE_001 = "gemini/imagen-4.0-fast-generate-001"
    GEMINI__IMAGEN_4_0_GENERATE_001 = "gemini/imagen-4.0-generate-001"
    GEMINI__IMAGEN_4_0_ULTRA_GENERATE_001 = "gemini/imagen-4.0-ultra-generate-001"
    GEMINI__LEARNLM_1_5_PRO_EXPERIMENTAL = "gemini/learnlm-1.5-pro-experimental"
    GEMINI__VEO_2_0_GENERATE_001 = "gemini/veo-2.0-generate-001"
    GEMINI__VEO_3_0_FAST_GENERATE_PREVIEW = "gemini/veo-3.0-fast-generate-preview"
    GEMINI__VEO_3_0_GENERATE_PREVIEW = "gemini/veo-3.0-generate-preview"

    # GROQ models
    GROQ_DEEPSEEK_R1_DISTILL_LLAMA_70B = "groq/deepseek-r1-distill-llama-70b"
    GROQ_DISTIL_WHISPER_LARGE_V3_EN = "groq/distil-whisper-large-v3-en"
    GROQ_GEMMA_7B_IT = "groq/gemma-7b-it"
    GROQ_GEMMA2_9B_IT = "groq/gemma2-9b-it"
    GROQ_LLAMA_3_1_405B_REASONING = "groq/llama-3.1-405b-reasoning"
    GROQ_LLAMA_3_1_70B_VERSATILE = "groq/llama-3.1-70b-versatile"
    GROQ_LLAMA_3_1_8B_INSTANT = "groq/llama-3.1-8b-instant"
    GROQ_LLAMA_3_2_11B_TEXT_PREVIEW = "groq/llama-3.2-11b-text-preview"
    GROQ_LLAMA_3_2_11B_VISION_PREVIEW = "groq/llama-3.2-11b-vision-preview"
    GROQ_LLAMA_3_2_1B_PREVIEW = "groq/llama-3.2-1b-preview"
    GROQ_LLAMA_3_2_3B_PREVIEW = "groq/llama-3.2-3b-preview"
    GROQ_LLAMA_3_2_90B_TEXT_PREVIEW = "groq/llama-3.2-90b-text-preview"
    GROQ_LLAMA_3_2_90B_VISION_PREVIEW = "groq/llama-3.2-90b-vision-preview"
    GROQ_LLAMA_3_3_70B_SPECDEC = "groq/llama-3.3-70b-specdec"
    GROQ_LLAMA_3_3_70B_VERSATILE = "groq/llama-3.3-70b-versatile"
    GROQ_LLAMA_GUARD_3_8B = "groq/llama-guard-3-8b"
    GROQ_LLAMA2_70B_4096 = "groq/llama2-70b-4096"
    GROQ_LLAMA3_GROQ_70B_8192_TOOL_USE_PREVIEW = "groq/llama3-groq-70b-8192-tool-use-preview"
    GROQ_LLAMA3_GROQ_8B_8192_TOOL_USE_PREVIEW = "groq/llama3-groq-8b-8192-tool-use-preview"
    GROQ_META_LLAMA_LLAMA_4_MAVERICK_17B_128E_INSTRUCT = "groq/meta-llama/llama-4-maverick-17b-128e-instruct"
    GROQ_META_LLAMA_LLAMA_4_SCOUT_17B_16E_INSTRUCT = "groq/meta-llama/llama-4-scout-17b-16e-instruct"
    GROQ_MISTRAL_SABA_24B = "groq/mistral-saba-24b"
    GROQ_MIXTRAL_8X7B_32768 = "groq/mixtral-8x7b-32768"
    GROQ_MOONSHOTAI_KIMI_K2_INSTRUCT = "groq/moonshotai/kimi-k2-instruct"
    GROQ_MOONSHOTAI_KIMI_K2_INSTRUCT_0905 = "groq/moonshotai/kimi-k2-instruct-0905"
    GROQ_OPENAI_GPT_OSS_120B = "groq/openai/gpt-oss-120b"
    GROQ_OPENAI_GPT_OSS_20B = "groq/openai/gpt-oss-20b"
    GROQ_PLAYAI_TTS = "groq/playai-tts"
    GROQ_QWEN_QWEN3_32B = "groq/qwen/qwen3-32b"
    GROQ_WHISPER_LARGE_V3 = "groq/whisper-large-v3"
    GROQ_WHISPER_LARGE_V3_TURBO = "groq/whisper-large-v3-turbo"

    # HD models
    HD_1024_X_1024_DALL_E_3 = "hd/1024-x-1024/dall-e-3"
    HD_1024_X_1792_DALL_E_3 = "hd/1024-x-1792/dall-e-3"
    HD_1792_X_1024_DALL_E_3 = "hd/1792-x-1024/dall-e-3"

    # HIGH models
    HIGH_1024_X_1024_GPT_IMAGE_1 = "high/1024-x-1024/gpt-image-1"
    HIGH_1024_X_1536_GPT_IMAGE_1 = "high/1024-x-1536/gpt-image-1"
    HIGH_1536_X_1024_GPT_IMAGE_1 = "high/1536-x-1024/gpt-image-1"

    # HUGGINGFACEH4 models
    HUGGINGFACEH4_STARCHAT_ALPHA = "HuggingFaceH4/starchat-alpha"

    # JINA models
    JINA_RERANKER_V2_BASE_MULTILINGUAL = "jina-reranker-v2-base-multilingual"

    # JOEHOOVER models
    JOEHOOVER_INSTRUCTBLIP_VICUNA13B_C4C54E3C8C97CD50C2D2FEC9BE3B6065563CCF7D43787FB99F84151B867178FE = "joehoover/instructblip-vicuna13b:c4c54e3c8c97cd50c2d2fec9be3b6065563ccf7d43787fb99f84151b867178fe"

    # LAMBDA_AI models
    LAMBDA_AI_DEEPSEEK_LLAMA3_3_70B = "lambda_ai/deepseek-llama3.3-70b"
    LAMBDA_AI_DEEPSEEK_R1_0528 = "lambda_ai/deepseek-r1-0528"
    LAMBDA_AI_DEEPSEEK_R1_671B = "lambda_ai/deepseek-r1-671b"
    LAMBDA_AI_DEEPSEEK_V3_0324 = "lambda_ai/deepseek-v3-0324"
    LAMBDA_AI_HERMES3_405B = "lambda_ai/hermes3-405b"
    LAMBDA_AI_HERMES3_70B = "lambda_ai/hermes3-70b"
    LAMBDA_AI_HERMES3_8B = "lambda_ai/hermes3-8b"
    LAMBDA_AI_LFM_40B = "lambda_ai/lfm-40b"
    LAMBDA_AI_LFM_7B = "lambda_ai/lfm-7b"
    LAMBDA_AI_LLAMA_4_MAVERICK_17B_128E_INSTRUCT_FP8 = "lambda_ai/llama-4-maverick-17b-128e-instruct-fp8"
    LAMBDA_AI_LLAMA_4_SCOUT_17B_16E_INSTRUCT = "lambda_ai/llama-4-scout-17b-16e-instruct"
    LAMBDA_AI_LLAMA3_1_405B_INSTRUCT_FP8 = "lambda_ai/llama3.1-405b-instruct-fp8"
    LAMBDA_AI_LLAMA3_1_70B_INSTRUCT_FP8 = "lambda_ai/llama3.1-70b-instruct-fp8"
    LAMBDA_AI_LLAMA3_1_8B_INSTRUCT = "lambda_ai/llama3.1-8b-instruct"
    LAMBDA_AI_LLAMA3_1_NEMOTRON_70B_INSTRUCT_FP8 = "lambda_ai/llama3.1-nemotron-70b-instruct-fp8"
    LAMBDA_AI_LLAMA3_2_11B_VISION_INSTRUCT = "lambda_ai/llama3.2-11b-vision-instruct"
    LAMBDA_AI_LLAMA3_2_3B_INSTRUCT = "lambda_ai/llama3.2-3b-instruct"
    LAMBDA_AI_LLAMA3_3_70B_INSTRUCT_FP8 = "lambda_ai/llama3.3-70b-instruct-fp8"
    LAMBDA_AI_QWEN25_CODER_32B_INSTRUCT = "lambda_ai/qwen25-coder-32b-instruct"
    LAMBDA_AI_QWEN3_32B_FP8 = "lambda_ai/qwen3-32b-fp8"

    # LOW models
    LOW_1024_X_1024_GPT_IMAGE_1 = "low/1024-x-1024/gpt-image-1"
    LOW_1024_X_1024_GPT_IMAGE_1_MINI = "low/1024-x-1024/gpt-image-1-mini"
    LOW_1024_X_1536_GPT_IMAGE_1 = "low/1024-x-1536/gpt-image-1"
    LOW_1024_X_1536_GPT_IMAGE_1_MINI = "low/1024-x-1536/gpt-image-1-mini"
    LOW_1536_X_1024_GPT_IMAGE_1 = "low/1536-x-1024/gpt-image-1"
    LOW_1536_X_1024_GPT_IMAGE_1_MINI = "low/1536-x-1024/gpt-image-1-mini"

    # MARITALK models
    MARITALK = "maritalk"

    # MAX-X-MAX models
    MAX_X_MAX_50_STEPS_STABILITY_STABLE_DIFFUSION_XL_V0 = "max-x-max/50-steps/stability.stable-diffusion-xl-v0"
    MAX_X_MAX_MAX_STEPS_STABILITY_STABLE_DIFFUSION_XL_V0 = "max-x-max/max-steps/stability.stable-diffusion-xl-v0"

    # MEDIUM models
    MEDIUM_1024_X_1024_GPT_IMAGE_1 = "medium/1024-x-1024/gpt-image-1"
    MEDIUM_1024_X_1024_GPT_IMAGE_1_MINI = "medium/1024-x-1024/gpt-image-1-mini"
    MEDIUM_1024_X_1536_GPT_IMAGE_1 = "medium/1024-x-1536/gpt-image-1"
    MEDIUM_1024_X_1536_GPT_IMAGE_1_MINI = "medium/1024-x-1536/gpt-image-1-mini"
    MEDIUM_1536_X_1024_GPT_IMAGE_1 = "medium/1536-x-1024/gpt-image-1"
    MEDIUM_1536_X_1024_GPT_IMAGE_1_MINI = "medium/1536-x-1024/gpt-image-1-mini"

    # META models
    EU_META_LLAMA3_2_1B_INSTRUCT_V1_0 = "eu.meta.llama3-2-1b-instruct-v1:0"
    EU_META_LLAMA3_2_3B_INSTRUCT_V1_0 = "eu.meta.llama3-2-3b-instruct-v1:0"
    LLAMA2 = "llama2"
    META_LLAMA2_13B_CHAT_V1 = "meta.llama2-13b-chat-v1"
    META_LLAMA2_70B_CHAT_V1 = "meta.llama2-70b-chat-v1"
    META_LLAMA3_1_405B_INSTRUCT_V1_0 = "meta.llama3-1-405b-instruct-v1:0"
    META_LLAMA3_1_70B_INSTRUCT_V1_0 = "meta.llama3-1-70b-instruct-v1:0"
    META_LLAMA3_1_8B_INSTRUCT_V1_0 = "meta.llama3-1-8b-instruct-v1:0"
    META_LLAMA3_2_11B_INSTRUCT_V1_0 = "meta.llama3-2-11b-instruct-v1:0"
    META_LLAMA3_2_1B_INSTRUCT_V1_0 = "meta.llama3-2-1b-instruct-v1:0"
    META_LLAMA3_2_3B_INSTRUCT_V1_0 = "meta.llama3-2-3b-instruct-v1:0"
    META_LLAMA3_2_90B_INSTRUCT_V1_0 = "meta.llama3-2-90b-instruct-v1:0"
    META_LLAMA3_70B_INSTRUCT_V1_0 = "meta.llama3-70b-instruct-v1:0"
    META_LLAMA3_8B_INSTRUCT_V1_0 = "meta.llama3-8b-instruct-v1:0"
    META_CODELLAMA_13B_1C914D844307B0588599B8393480A3BA917B660C7E9DFAE681542B5325F228DB = "meta/codellama-13b:1c914d844307b0588599b8393480a3ba917b660c7e9dfae681542b5325f228db"
    US_META_LLAMA3_1_405B_INSTRUCT_V1_0 = "us.meta.llama3-1-405b-instruct-v1:0"
    US_META_LLAMA3_1_70B_INSTRUCT_V1_0 = "us.meta.llama3-1-70b-instruct-v1:0"
    US_META_LLAMA3_1_8B_INSTRUCT_V1_0 = "us.meta.llama3-1-8b-instruct-v1:0"
    US_META_LLAMA3_2_11B_INSTRUCT_V1_0 = "us.meta.llama3-2-11b-instruct-v1:0"
    US_META_LLAMA3_2_1B_INSTRUCT_V1_0 = "us.meta.llama3-2-1b-instruct-v1:0"
    US_META_LLAMA3_2_3B_INSTRUCT_V1_0 = "us.meta.llama3-2-3b-instruct-v1:0"
    US_META_LLAMA3_2_90B_INSTRUCT_V1_0 = "us.meta.llama3-2-90b-instruct-v1:0"

    # META-LLAMA models
    META_LLAMA_LLAMA_2_13B = "meta-llama/Llama-2-13b"
    META_LLAMA_LLAMA_2_13B_CHAT = "meta-llama/Llama-2-13b-chat"
    META_LLAMA_LLAMA_2_13B_CHAT_HF = "meta-llama/Llama-2-13b-chat-hf"
    META_LLAMA_LLAMA_2_13B_HF = "meta-llama/Llama-2-13b-hf"
    META_LLAMA_LLAMA_2_70B = "meta-llama/Llama-2-70b"
    META_LLAMA_LLAMA_2_70B_CHAT = "meta-llama/Llama-2-70b-chat"
    META_LLAMA_LLAMA_2_70B_CHAT_HF = "meta-llama/Llama-2-70b-chat-hf"
    META_LLAMA_LLAMA_2_70B_HF = "meta-llama/Llama-2-70b-hf"
    META_LLAMA_LLAMA_2_7B = "meta-llama/Llama-2-7b"
    META_LLAMA_LLAMA_2_7B_CHAT = "meta-llama/Llama-2-7b-chat"
    META_LLAMA_LLAMA_2_7B_CHAT_HF = "meta-llama/Llama-2-7b-chat-hf"
    META_LLAMA_LLAMA_2_7B_HF = "meta-llama/Llama-2-7b-hf"

    # META_LLAMA models
    META_LLAMA_LLAMA_3_3_70B_INSTRUCT = "meta_llama/Llama-3.3-70B-Instruct"
    META_LLAMA_LLAMA_3_3_8B_INSTRUCT = "meta_llama/Llama-3.3-8B-Instruct"
    META_LLAMA_LLAMA_4_MAVERICK_17B_128E_INSTRUCT_FP8 = "meta_llama/Llama-4-Maverick-17B-128E-Instruct-FP8"
    META_LLAMA_LLAMA_4_SCOUT_17B_16E_INSTRUCT_FP8 = "meta_llama/Llama-4-Scout-17B-16E-Instruct-FP8"

    # MISTRAL models
    MISTRAL_MISTRAL_7B_INSTRUCT_V0_2 = "mistral.mistral-7b-instruct-v0:2"
    MISTRAL_MISTRAL_LARGE_2402_V1_0 = "mistral.mistral-large-2402-v1:0"
    MISTRAL_MISTRAL_LARGE_2407_V1_0 = "mistral.mistral-large-2407-v1:0"
    MISTRAL_MISTRAL_SMALL_2402_V1_0 = "mistral.mistral-small-2402-v1:0"
    MISTRAL_MIXTRAL_8X7B_INSTRUCT_V0_1 = "mistral.mixtral-8x7b-instruct-v0:1"

    # MOONSHOT models
    MOONSHOT_KIMI_K2_0711_PREVIEW = "moonshot/kimi-k2-0711-preview"
    MOONSHOT_KIMI_LATEST = "moonshot/kimi-latest"
    MOONSHOT_KIMI_LATEST_128K = "moonshot/kimi-latest-128k"
    MOONSHOT_KIMI_LATEST_32K = "moonshot/kimi-latest-32k"
    MOONSHOT_KIMI_LATEST_8K = "moonshot/kimi-latest-8k"
    MOONSHOT_KIMI_THINKING_PREVIEW = "moonshot/kimi-thinking-preview"
    MOONSHOT_MOONSHOT_V1_128K = "moonshot/moonshot-v1-128k"
    MOONSHOT_MOONSHOT_V1_128K_0430 = "moonshot/moonshot-v1-128k-0430"
    MOONSHOT_MOONSHOT_V1_128K_VISION_PREVIEW = "moonshot/moonshot-v1-128k-vision-preview"
    MOONSHOT_MOONSHOT_V1_32K = "moonshot/moonshot-v1-32k"
    MOONSHOT_MOONSHOT_V1_32K_0430 = "moonshot/moonshot-v1-32k-0430"
    MOONSHOT_MOONSHOT_V1_32K_VISION_PREVIEW = "moonshot/moonshot-v1-32k-vision-preview"
    MOONSHOT_MOONSHOT_V1_8K = "moonshot/moonshot-v1-8k"
    MOONSHOT_MOONSHOT_V1_8K_0430 = "moonshot/moonshot-v1-8k-0430"
    MOONSHOT_MOONSHOT_V1_8K_VISION_PREVIEW = "moonshot/moonshot-v1-8k-vision-preview"
    MOONSHOT_MOONSHOT_V1_AUTO = "moonshot/moonshot-v1-auto"

    # MORPH models
    MORPH_MORPH_V3_FAST = "morph/morph-v3-fast"
    MORPH_MORPH_V3_LARGE = "morph/morph-v3-large"

    # NOUSRESEARCH models
    NOUSRESEARCH_NOUS_HERMES_LLAMA2_13B = "NousResearch/Nous-Hermes-Llama2-13b"

    # NSCALE models
    NSCALE_QWEN_QWQ_32B = "nscale/Qwen/QwQ-32B"
    NSCALE_QWEN_QWEN2_5_CODER_32B_INSTRUCT = "nscale/Qwen/Qwen2.5-Coder-32B-Instruct"
    NSCALE_QWEN_QWEN2_5_CODER_3B_INSTRUCT = "nscale/Qwen/Qwen2.5-Coder-3B-Instruct"
    NSCALE_QWEN_QWEN2_5_CODER_7B_INSTRUCT = "nscale/Qwen/Qwen2.5-Coder-7B-Instruct"
    NSCALE_BLACK_FOREST_LABS_FLUX_1_SCHNELL = "nscale/black-forest-labs/FLUX.1-schnell"
    NSCALE_DEEPSEEK_AI_DEEPSEEK_R1_DISTILL_LLAMA_70B = "nscale/deepseek-ai/DeepSeek-R1-Distill-Llama-70B"
    NSCALE_DEEPSEEK_AI_DEEPSEEK_R1_DISTILL_LLAMA_8B = "nscale/deepseek-ai/DeepSeek-R1-Distill-Llama-8B"
    NSCALE_DEEPSEEK_AI_DEEPSEEK_R1_DISTILL_QWEN_1_5B = "nscale/deepseek-ai/DeepSeek-R1-Distill-Qwen-1.5B"
    NSCALE_DEEPSEEK_AI_DEEPSEEK_R1_DISTILL_QWEN_14B = "nscale/deepseek-ai/DeepSeek-R1-Distill-Qwen-14B"
    NSCALE_DEEPSEEK_AI_DEEPSEEK_R1_DISTILL_QWEN_32B = "nscale/deepseek-ai/DeepSeek-R1-Distill-Qwen-32B"
    NSCALE_DEEPSEEK_AI_DEEPSEEK_R1_DISTILL_QWEN_7B = "nscale/deepseek-ai/DeepSeek-R1-Distill-Qwen-7B"
    NSCALE_META_LLAMA_LLAMA_3_1_8B_INSTRUCT = "nscale/meta-llama/Llama-3.1-8B-Instruct"
    NSCALE_META_LLAMA_LLAMA_3_3_70B_INSTRUCT = "nscale/meta-llama/Llama-3.3-70B-Instruct"
    NSCALE_META_LLAMA_LLAMA_4_SCOUT_17B_16E_INSTRUCT = "nscale/meta-llama/Llama-4-Scout-17B-16E-Instruct"
    NSCALE_MISTRALAI_MIXTRAL_8X22B_INSTRUCT_V0_1 = "nscale/mistralai/mixtral-8x22b-instruct-v0.1"
    NSCALE_STABILITYAI_STABLE_DIFFUSION_XL_BASE_1_0 = "nscale/stabilityai/stable-diffusion-xl-base-1.0"

    # NUMBERSSTATION models
    NUMBERSSTATION_NSQL_LLAMA_2_7B = "NumbersStation/nsql-llama-2-7B"

    # OPENAI models
    BABBAGE_002 = "babbage-002"
    DAVINCI_002 = "davinci-002"
    GPT_3_5_TURBO = "gpt-3.5-turbo"
    GPT_3_5_TURBO_0125 = "gpt-3.5-turbo-0125"
    GPT_3_5_TURBO_0301 = "gpt-3.5-turbo-0301"
    GPT_3_5_TURBO_0613 = "gpt-3.5-turbo-0613"
    GPT_3_5_TURBO_1106 = "gpt-3.5-turbo-1106"
    GPT_3_5_TURBO_16K = "gpt-3.5-turbo-16k"
    GPT_3_5_TURBO_16K_0613 = "gpt-3.5-turbo-16k-0613"
    GPT_3_5_TURBO_INSTRUCT = "gpt-3.5-turbo-instruct"
    GPT_3_5_TURBO_INSTRUCT_0914 = "gpt-3.5-turbo-instruct-0914"
    GPT_4 = "gpt-4"
    GPT_4_0125_PREVIEW = "gpt-4-0125-preview"
    GPT_4_0314 = "gpt-4-0314"
    GPT_4_0613 = "gpt-4-0613"
    GPT_4_1106_PREVIEW = "gpt-4-1106-preview"
    GPT_4_1106_VISION_PREVIEW = "gpt-4-1106-vision-preview"
    GPT_4_32K = "gpt-4-32k"
    GPT_4_32K_0314 = "gpt-4-32k-0314"
    GPT_4_32K_0613 = "gpt-4-32k-0613"
    GPT_4_TURBO = "gpt-4-turbo"
    GPT_4_TURBO_2024_04_09 = "gpt-4-turbo-2024-04-09"
    GPT_4_TURBO_PREVIEW = "gpt-4-turbo-preview"
    GPT_4_VISION_PREVIEW = "gpt-4-vision-preview"
    GPT_4_1 = "gpt-4.1"
    GPT_4_1_2025_04_14 = "gpt-4.1-2025-04-14"
    GPT_4_1_MINI = "gpt-4.1-mini"
    GPT_4_1_MINI_2025_04_14 = "gpt-4.1-mini-2025-04-14"
    GPT_4_1_NANO = "gpt-4.1-nano"
    GPT_4_1_NANO_2025_04_14 = "gpt-4.1-nano-2025-04-14"
    GPT_4_5_PREVIEW = "gpt-4.5-preview"
    GPT_4_5_PREVIEW_2025_02_27 = "gpt-4.5-preview-2025-02-27"
    GPT_4O = "gpt-4o"
    GPT_4O_2024_05_13 = "gpt-4o-2024-05-13"
    GPT_4O_2024_08_06 = "gpt-4o-2024-08-06"
    GPT_4O_2024_11_20 = "gpt-4o-2024-11-20"
    GPT_4O_AUDIO_PREVIEW = "gpt-4o-audio-preview"
    GPT_4O_AUDIO_PREVIEW_2024_10_01 = "gpt-4o-audio-preview-2024-10-01"
    GPT_4O_AUDIO_PREVIEW_2024_12_17 = "gpt-4o-audio-preview-2024-12-17"
    GPT_4O_AUDIO_PREVIEW_2025_06_03 = "gpt-4o-audio-preview-2025-06-03"
    GPT_4O_MINI = "gpt-4o-mini"
    GPT_4O_MINI_2024_07_18 = "gpt-4o-mini-2024-07-18"
    GPT_4O_MINI_AUDIO_PREVIEW = "gpt-4o-mini-audio-preview"
    GPT_4O_MINI_AUDIO_PREVIEW_2024_12_17 = "gpt-4o-mini-audio-preview-2024-12-17"
    GPT_4O_MINI_REALTIME_PREVIEW = "gpt-4o-mini-realtime-preview"
    GPT_4O_MINI_REALTIME_PREVIEW_2024_12_17 = "gpt-4o-mini-realtime-preview-2024-12-17"
    GPT_4O_MINI_SEARCH_PREVIEW = "gpt-4o-mini-search-preview"
    GPT_4O_MINI_SEARCH_PREVIEW_2025_03_11 = "gpt-4o-mini-search-preview-2025-03-11"
    GPT_4O_MINI_TRANSCRIBE = "gpt-4o-mini-transcribe"
    GPT_4O_MINI_TTS = "gpt-4o-mini-tts"
    GPT_4O_REALTIME_PREVIEW = "gpt-4o-realtime-preview"
    GPT_4O_REALTIME_PREVIEW_2024_10_01 = "gpt-4o-realtime-preview-2024-10-01"
    GPT_4O_REALTIME_PREVIEW_2024_12_17 = "gpt-4o-realtime-preview-2024-12-17"
    GPT_4O_REALTIME_PREVIEW_2025_06_03 = "gpt-4o-realtime-preview-2025-06-03"
    GPT_4O_SEARCH_PREVIEW = "gpt-4o-search-preview"
    GPT_4O_SEARCH_PREVIEW_2025_03_11 = "gpt-4o-search-preview-2025-03-11"
    GPT_4O_TRANSCRIBE = "gpt-4o-transcribe"
    GPT_5 = "gpt-5"
    GPT_5_2025_08_07 = "gpt-5-2025-08-07"
    GPT_5_CHAT = "gpt-5-chat"
    GPT_5_CHAT_LATEST = "gpt-5-chat-latest"
    GPT_5_CODEX = "gpt-5-codex"
    GPT_5_MINI = "gpt-5-mini"
    GPT_5_MINI_2025_08_07 = "gpt-5-mini-2025-08-07"
    GPT_5_NANO = "gpt-5-nano"
    GPT_5_NANO_2025_08_07 = "gpt-5-nano-2025-08-07"
    GPT_5_PRO = "gpt-5-pro"
    GPT_5_PRO_2025_10_06 = "gpt-5-pro-2025-10-06"
    GPT_IMAGE_1 = "gpt-image-1"
    GPT_IMAGE_1_MINI = "gpt-image-1-mini"
    GPT_REALTIME = "gpt-realtime"
    GPT_REALTIME_2025_08_28 = "gpt-realtime-2025-08-28"
    GPT_REALTIME_MINI = "gpt-realtime-mini"
    O1 = "o1"
    O1_2024_12_17 = "o1-2024-12-17"
    O1_MINI = "o1-mini"
    O1_MINI_2024_09_12 = "o1-mini-2024-09-12"
    O1_PREVIEW = "o1-preview"
    O1_PREVIEW_2024_09_12 = "o1-preview-2024-09-12"
    O1_PRO = "o1-pro"
    O1_PRO_2025_03_19 = "o1-pro-2025-03-19"
    O3 = "o3"
    O3_2025_04_16 = "o3-2025-04-16"
    O3_DEEP_RESEARCH = "o3-deep-research"
    O3_DEEP_RESEARCH_2025_06_26 = "o3-deep-research-2025-06-26"
    O3_MINI = "o3-mini"
    O3_MINI_2025_01_31 = "o3-mini-2025-01-31"
    O3_PRO = "o3-pro"
    O3_PRO_2025_06_10 = "o3-pro-2025-06-10"
    O4_MINI_1 = "o4-mini"
    O4_MINI_2025_04_16 = "o4-mini-2025-04-16"
    O4_MINI_DEEP_RESEARCH = "o4-mini-deep-research"
    O4_MINI_DEEP_RESEARCH_2025_06_26 = "o4-mini-deep-research-2025-06-26"
    OPENAI_CONTAINER = "openai/container"
    OPENAI_SORA_2 = "openai/sora-2"
    OPENAI_SORA_2_PRO = "openai/sora-2-pro"
    TEXT_BISON = "text-bison"
    TEXT_BISON32K = "text-bison32k"
    TEXT_BISON32K_002 = "text-bison32k@002"
    TEXT_BISON_001 = "text-bison@001"
    TEXT_BISON_002 = "text-bison@002"
    TEXT_EMBEDDING_3_LARGE = "text-embedding-3-large"
    TEXT_EMBEDDING_3_SMALL = "text-embedding-3-small"
    TEXT_EMBEDDING_ADA_002 = "text-embedding-ada-002"
    TEXT_EMBEDDING_ADA_002_V2 = "text-embedding-ada-002-v2"
    TEXT_MODERATION_007 = "text-moderation-007"
    TEXT_MODERATION_LATEST = "text-moderation-latest"
    TEXT_MODERATION_STABLE = "text-moderation-stable"
    TEXT_UNICORN = "text-unicorn"
    TEXT_UNICORN_001 = "text-unicorn@001"
    TTS_1 = "tts-1"
    TTS_1_HD = "tts-1-hd"
    WHISPER_1 = "whisper-1"

    # OPENROUTER models
    OPENROUTER_ANTHROPIC_CLAUDE_2 = "openrouter/anthropic/claude-2"
    OPENROUTER_ANTHROPIC_CLAUDE_3_5_HAIKU = "openrouter/anthropic/claude-3-5-haiku"
    OPENROUTER_ANTHROPIC_CLAUDE_3_5_HAIKU_20241022 = "openrouter/anthropic/claude-3-5-haiku-20241022"
    OPENROUTER_ANTHROPIC_CLAUDE_3_HAIKU = "openrouter/anthropic/claude-3-haiku"
    OPENROUTER_ANTHROPIC_CLAUDE_3_HAIKU_20240307 = "openrouter/anthropic/claude-3-haiku-20240307"
    OPENROUTER_ANTHROPIC_CLAUDE_3_OPUS = "openrouter/anthropic/claude-3-opus"
    OPENROUTER_ANTHROPIC_CLAUDE_3_SONNET = "openrouter/anthropic/claude-3-sonnet"
    OPENROUTER_ANTHROPIC_CLAUDE_3_5_SONNET = "openrouter/anthropic/claude-3.5-sonnet"
    OPENROUTER_ANTHROPIC_CLAUDE_3_5_SONNET_BETA = "openrouter/anthropic/claude-3.5-sonnet:beta"
    OPENROUTER_ANTHROPIC_CLAUDE_3_7_SONNET = "openrouter/anthropic/claude-3.7-sonnet"
    OPENROUTER_ANTHROPIC_CLAUDE_3_7_SONNET_BETA = "openrouter/anthropic/claude-3.7-sonnet:beta"
    OPENROUTER_ANTHROPIC_CLAUDE_HAIKU_4_5 = "openrouter/anthropic/claude-haiku-4.5"
    OPENROUTER_ANTHROPIC_CLAUDE_INSTANT_V1 = "openrouter/anthropic/claude-instant-v1"
    OPENROUTER_ANTHROPIC_CLAUDE_OPUS_4 = "openrouter/anthropic/claude-opus-4"
    OPENROUTER_ANTHROPIC_CLAUDE_OPUS_4_1 = "openrouter/anthropic/claude-opus-4.1"
    OPENROUTER_ANTHROPIC_CLAUDE_SONNET_4 = "openrouter/anthropic/claude-sonnet-4"
    OPENROUTER_ANTHROPIC_CLAUDE_SONNET_4_5 = "openrouter/anthropic/claude-sonnet-4.5"
    OPENROUTER_BYTEDANCE_UI_TARS_1_5_7B = "openrouter/bytedance/ui-tars-1.5-7b"
    OPENROUTER_COGNITIVECOMPUTATIONS_DOLPHIN_MIXTRAL_8X7B = "openrouter/cognitivecomputations/dolphin-mixtral-8x7b"
    OPENROUTER_COHERE_COMMAND_R_PLUS = "openrouter/cohere/command-r-plus"
    OPENROUTER_DATABRICKS_DBRX_INSTRUCT = "openrouter/databricks/dbrx-instruct"
    OPENROUTER_DEEPSEEK_DEEPSEEK_CHAT = "openrouter/deepseek/deepseek-chat"
    OPENROUTER_DEEPSEEK_DEEPSEEK_CHAT_V3_0324 = "openrouter/deepseek/deepseek-chat-v3-0324"
    OPENROUTER_DEEPSEEK_DEEPSEEK_CHAT_V3_1 = "openrouter/deepseek/deepseek-chat-v3.1"
    OPENROUTER_DEEPSEEK_DEEPSEEK_CODER = "openrouter/deepseek/deepseek-coder"
    OPENROUTER_DEEPSEEK_DEEPSEEK_R1 = "openrouter/deepseek/deepseek-r1"
    OPENROUTER_DEEPSEEK_DEEPSEEK_R1_0528 = "openrouter/deepseek/deepseek-r1-0528"
    OPENROUTER_FIREWORKS_FIRELLAVA_13B = "openrouter/fireworks/firellava-13b"
    OPENROUTER_GOOGLE_GEMINI_2_0_FLASH_001 = "openrouter/google/gemini-2.0-flash-001"
    OPENROUTER_GOOGLE_GEMINI_2_5_FLASH = "openrouter/google/gemini-2.5-flash"
    OPENROUTER_GOOGLE_GEMINI_2_5_PRO = "openrouter/google/gemini-2.5-pro"
    OPENROUTER_GOOGLE_GEMINI_PRO_1_5 = "openrouter/google/gemini-pro-1.5"
    OPENROUTER_GOOGLE_GEMINI_PRO_VISION = "openrouter/google/gemini-pro-vision"
    OPENROUTER_GOOGLE_PALM_2_CHAT_BISON = "openrouter/google/palm-2-chat-bison"
    OPENROUTER_GOOGLE_PALM_2_CODECHAT_BISON = "openrouter/google/palm-2-codechat-bison"
    OPENROUTER_GRYPHE_MYTHOMAX_L2_13B = "openrouter/gryphe/mythomax-l2-13b"
    OPENROUTER_JONDURBIN_AIROBOROS_L2_70B_2_1 = "openrouter/jondurbin/airoboros-l2-70b-2.1"
    OPENROUTER_MANCER_WEAVER = "openrouter/mancer/weaver"
    OPENROUTER_META_LLAMA_CODELLAMA_34B_INSTRUCT = "openrouter/meta-llama/codellama-34b-instruct"
    OPENROUTER_META_LLAMA_LLAMA_2_13B_CHAT = "openrouter/meta-llama/llama-2-13b-chat"
    OPENROUTER_META_LLAMA_LLAMA_2_70B_CHAT = "openrouter/meta-llama/llama-2-70b-chat"
    OPENROUTER_META_LLAMA_LLAMA_3_70B_INSTRUCT = "openrouter/meta-llama/llama-3-70b-instruct"
    OPENROUTER_META_LLAMA_LLAMA_3_70B_INSTRUCT_NITRO = "openrouter/meta-llama/llama-3-70b-instruct:nitro"
    OPENROUTER_META_LLAMA_LLAMA_3_8B_INSTRUCT_EXTENDED = "openrouter/meta-llama/llama-3-8b-instruct:extended"
    OPENROUTER_META_LLAMA_LLAMA_3_8B_INSTRUCT_FREE = "openrouter/meta-llama/llama-3-8b-instruct:free"
    OPENROUTER_MICROSOFT_WIZARDLM_2_8X22B_NITRO = "openrouter/microsoft/wizardlm-2-8x22b:nitro"
    OPENROUTER_MISTRALAI_MISTRAL_7B_INSTRUCT = "openrouter/mistralai/mistral-7b-instruct"
    OPENROUTER_MISTRALAI_MISTRAL_7B_INSTRUCT_FREE = "openrouter/mistralai/mistral-7b-instruct:free"
    OPENROUTER_MISTRALAI_MISTRAL_LARGE = "openrouter/mistralai/mistral-large"
    OPENROUTER_MISTRALAI_MISTRAL_SMALL_3_1_24B_INSTRUCT = "openrouter/mistralai/mistral-small-3.1-24b-instruct"
    OPENROUTER_MISTRALAI_MISTRAL_SMALL_3_2_24B_INSTRUCT = "openrouter/mistralai/mistral-small-3.2-24b-instruct"
    OPENROUTER_MISTRALAI_MIXTRAL_8X22B_INSTRUCT = "openrouter/mistralai/mixtral-8x22b-instruct"
    OPENROUTER_NOUSRESEARCH_NOUS_HERMES_LLAMA2_13B = "openrouter/nousresearch/nous-hermes-llama2-13b"
    OPENROUTER_OPENAI_GPT_3_5_TURBO = "openrouter/openai/gpt-3.5-turbo"
    OPENROUTER_OPENAI_GPT_3_5_TURBO_16K = "openrouter/openai/gpt-3.5-turbo-16k"
    OPENROUTER_OPENAI_GPT_4 = "openrouter/openai/gpt-4"
    OPENROUTER_OPENAI_GPT_4_VISION_PREVIEW = "openrouter/openai/gpt-4-vision-preview"
    OPENROUTER_OPENAI_GPT_4_1 = "openrouter/openai/gpt-4.1"
    OPENROUTER_OPENAI_GPT_4_1_2025_04_14 = "openrouter/openai/gpt-4.1-2025-04-14"
    OPENROUTER_OPENAI_GPT_4_1_MINI = "openrouter/openai/gpt-4.1-mini"
    OPENROUTER_OPENAI_GPT_4_1_MINI_2025_04_14 = "openrouter/openai/gpt-4.1-mini-2025-04-14"
    OPENROUTER_OPENAI_GPT_4_1_NANO = "openrouter/openai/gpt-4.1-nano"
    OPENROUTER_OPENAI_GPT_4_1_NANO_2025_04_14 = "openrouter/openai/gpt-4.1-nano-2025-04-14"
    OPENROUTER_OPENAI_GPT_4O = "openrouter/openai/gpt-4o"
    OPENROUTER_OPENAI_GPT_4O_2024_05_13 = "openrouter/openai/gpt-4o-2024-05-13"
    OPENROUTER_OPENAI_GPT_5 = "openrouter/openai/gpt-5"
    OPENROUTER_OPENAI_GPT_5_CHAT = "openrouter/openai/gpt-5-chat"
    OPENROUTER_OPENAI_GPT_5_CODEX = "openrouter/openai/gpt-5-codex"
    OPENROUTER_OPENAI_GPT_5_MINI = "openrouter/openai/gpt-5-mini"
    OPENROUTER_OPENAI_GPT_5_NANO = "openrouter/openai/gpt-5-nano"
    OPENROUTER_OPENAI_GPT_OSS_120B = "openrouter/openai/gpt-oss-120b"
    OPENROUTER_OPENAI_GPT_OSS_20B = "openrouter/openai/gpt-oss-20b"
    OPENROUTER_OPENAI_O1 = "openrouter/openai/o1"
    OPENROUTER_OPENAI_O1_MINI = "openrouter/openai/o1-mini"
    OPENROUTER_OPENAI_O1_MINI_2024_09_12 = "openrouter/openai/o1-mini-2024-09-12"
    OPENROUTER_OPENAI_O1_PREVIEW = "openrouter/openai/o1-preview"
    OPENROUTER_OPENAI_O1_PREVIEW_2024_09_12 = "openrouter/openai/o1-preview-2024-09-12"
    OPENROUTER_OPENAI_O3_MINI = "openrouter/openai/o3-mini"
    OPENROUTER_OPENAI_O3_MINI_HIGH = "openrouter/openai/o3-mini-high"
    OPENROUTER_PYGMALIONAI_MYTHALION_13B = "openrouter/pygmalionai/mythalion-13b"
    OPENROUTER_QWEN_QWEN_2_5_CODER_32B_INSTRUCT = "openrouter/qwen/qwen-2.5-coder-32b-instruct"
    OPENROUTER_QWEN_QWEN_VL_PLUS = "openrouter/qwen/qwen-vl-plus"
    OPENROUTER_QWEN_QWEN3_CODER = "openrouter/qwen/qwen3-coder"
    OPENROUTER_SWITCHPOINT_ROUTER = "openrouter/switchpoint/router"
    OPENROUTER_UNDI95_REMM_SLERP_L2_13B = "openrouter/undi95/remm-slerp-l2-13b"
    OPENROUTER_X_AI_GROK_4 = "openrouter/x-ai/grok-4"
    OPENROUTER_X_AI_GROK_4_FAST_FREE = "openrouter/x-ai/grok-4-fast:free"

    # OTHER models
    MODEL_31DXRJ3 = "31dxrj3"
    AI21_J2_MID_V1 = "ai21.j2-mid-v1"
    AI21_J2_ULTRA_V1 = "ai21.j2-ultra-v1"
    AI21_JAMBA_1_5_LARGE_V1_0 = "ai21.jamba-1-5-large-v1:0"
    AI21_JAMBA_1_5_MINI_V1_0 = "ai21.jamba-1-5-mini-v1:0"
    AI21_JAMBA_INSTRUCT_V1_0 = "ai21.jamba-instruct-v1:0"
    AMAZON_RERANK_V1_0 = "amazon.rerank-v1:0"
    AMAZON_TITAN_EMBED_IMAGE_V1 = "amazon.titan-embed-image-v1"
    AMAZON_TITAN_EMBED_TEXT_V1 = "amazon.titan-embed-text-v1"
    AMAZON_TITAN_EMBED_TEXT_V2_0 = "amazon.titan-embed-text-v2:0"
    AMAZON_TITAN_IMAGE_GENERATOR_V1 = "amazon.titan-image-generator-v1"
    AMAZON_TITAN_IMAGE_GENERATOR_V2 = "amazon.titan-image-generator-v2"
    AMAZON_TITAN_TEXT_EXPRESS_V1 = "amazon.titan-text-express-v1"
    AMAZON_TITAN_TEXT_LITE_V1 = "amazon.titan-text-lite-v1"
    AMAZON_TITAN_TEXT_PREMIER_V1_0 = "amazon.titan-text-premier-v1:0"
    CHATDOLPHIN = "chatdolphin"
    CHATGPT_4O_LATEST = "chatgpt-4o-latest"
    CODEX_MINI_LATEST = "codex-mini-latest"
    COMPUTER_USE_PREVIEW = "computer-use-preview"
    DOLPHIN = "dolphin"
    EMBED_ENGLISH_LIGHT_V2_0 = "embed-english-light-v2.0"
    EMBED_ENGLISH_LIGHT_V3_0 = "embed-english-light-v3.0"
    EMBED_ENGLISH_V2_0 = "embed-english-v2.0"
    EMBED_ENGLISH_V3_0 = "embed-english-v3.0"
    EMBED_MULTILINGUAL_V2_0 = "embed-multilingual-v2.0"
    EMBED_MULTILINGUAL_V3_0 = "embed-multilingual-v3.0"
    EU_TWELVELABS_MARENGO_EMBED_2_7_V1_0 = "eu.twelvelabs.marengo-embed-2-7-v1:0"
    EU_TWELVELABS_PEGASUS_1_2_V1_0 = "eu.twelvelabs.pegasus-1-2-v1:0"
    FT_BABBAGE_002 = "ft:babbage-002"
    FT_DAVINCI_002 = "ft:davinci-002"
    J2_LIGHT = "j2-light"
    J2_MID = "j2-mid"
    J2_ULTRA = "j2-ultra"
    JAMBA_1_5 = "jamba-1.5"
    JAMBA_1_5_LARGE = "jamba-1.5-large"
    JAMBA_1_5_LARGE_001 = "jamba-1.5-large@001"
    JAMBA_1_5_MINI = "jamba-1.5-mini"
    JAMBA_1_5_MINI_001 = "jamba-1.5-mini@001"
    JAMBA_LARGE_1_6 = "jamba-large-1.6"
    JAMBA_LARGE_1_7 = "jamba-large-1.7"
    JAMBA_MINI_1_6 = "jamba-mini-1.6"
    JAMBA_MINI_1_7 = "jamba-mini-1.7"
    LUMINOUS_BASE = "luminous-base"
    LUMINOUS_BASE_CONTROL = "luminous-base-control"
    LUMINOUS_EXTENDED = "luminous-extended"
    LUMINOUS_EXTENDED_CONTROL = "luminous-extended-control"
    LUMINOUS_SUPREME = "luminous-supreme"
    LUMINOUS_SUPREME_CONTROL = "luminous-supreme-control"
    MEDLM_LARGE = "medlm-large"
    MEDLM_MEDIUM = "medlm-medium"
    OMNI_MODERATION_2024_09_26 = "omni-moderation-2024-09-26"
    OMNI_MODERATION_LATEST = "omni-moderation-latest"
    OMNI_MODERATION_LATEST_INTENTS = "omni-moderation-latest-intents"
    Q841O8W = "q841o8w"
    QVV0XEQ = "qvv0xeq"
    RERANK_ENGLISH_V2_0 = "rerank-english-v2.0"
    RERANK_ENGLISH_V3_0 = "rerank-english-v3.0"
    RERANK_MULTILINGUAL_V2_0 = "rerank-multilingual-v2.0"
    RERANK_MULTILINGUAL_V3_0 = "rerank-multilingual-v3.0"
    RERANK_V3_5 = "rerank-v3.5"
    TWELVELABS_MARENGO_EMBED_2_7_V1_0 = "twelvelabs.marengo-embed-2-7-v1:0"
    TWELVELABS_PEGASUS_1_2_V1_0 = "twelvelabs.pegasus-1-2-v1:0"
    US_TWELVELABS_MARENGO_EMBED_2_7_V1_0 = "us.twelvelabs.marengo-embed-2-7-v1:0"
    US_TWELVELABS_PEGASUS_1_2_V1_0 = "us.twelvelabs.pegasus-1-2-v1:0"

    # PALM models
    PALM_CHAT_BISON = "palm/chat-bison"
    PALM_CHAT_BISON_001 = "palm/chat-bison-001"
    PALM_TEXT_BISON = "palm/text-bison"
    PALM_TEXT_BISON_001 = "palm/text-bison-001"
    PALM_TEXT_BISON_SAFETY_OFF = "palm/text-bison-safety-off"
    PALM_TEXT_BISON_SAFETY_RECITATION_OFF = "palm/text-bison-safety-recitation-off"

    # PERPLEXITY models
    PERPLEXITY_CODELLAMA_34B_INSTRUCT = "perplexity/codellama-34b-instruct"
    PERPLEXITY_CODELLAMA_70B_INSTRUCT = "perplexity/codellama-70b-instruct"
    PERPLEXITY_LLAMA_2_70B_CHAT = "perplexity/llama-2-70b-chat"
    PERPLEXITY_LLAMA_3_1_70B_INSTRUCT = "perplexity/llama-3.1-70b-instruct"
    PERPLEXITY_LLAMA_3_1_8B_INSTRUCT = "perplexity/llama-3.1-8b-instruct"
    PERPLEXITY_LLAMA_3_1_SONAR_HUGE_128K_ONLINE = "perplexity/llama-3.1-sonar-huge-128k-online"
    PERPLEXITY_LLAMA_3_1_SONAR_LARGE_128K_CHAT = "perplexity/llama-3.1-sonar-large-128k-chat"
    PERPLEXITY_LLAMA_3_1_SONAR_LARGE_128K_ONLINE = "perplexity/llama-3.1-sonar-large-128k-online"
    PERPLEXITY_LLAMA_3_1_SONAR_SMALL_128K_CHAT = "perplexity/llama-3.1-sonar-small-128k-chat"
    PERPLEXITY_LLAMA_3_1_SONAR_SMALL_128K_ONLINE = "perplexity/llama-3.1-sonar-small-128k-online"
    PERPLEXITY_MISTRAL_7B_INSTRUCT = "perplexity/mistral-7b-instruct"
    PERPLEXITY_MIXTRAL_8X7B_INSTRUCT = "perplexity/mixtral-8x7b-instruct"
    PERPLEXITY_PPLX_70B_CHAT = "perplexity/pplx-70b-chat"
    PERPLEXITY_PPLX_70B_ONLINE = "perplexity/pplx-70b-online"
    PERPLEXITY_PPLX_7B_CHAT = "perplexity/pplx-7b-chat"
    PERPLEXITY_PPLX_7B_ONLINE = "perplexity/pplx-7b-online"
    PERPLEXITY_SEARCH = "perplexity/search"
    PERPLEXITY_SONAR = "perplexity/sonar"
    PERPLEXITY_SONAR_DEEP_RESEARCH = "perplexity/sonar-deep-research"
    PERPLEXITY_SONAR_MEDIUM_CHAT = "perplexity/sonar-medium-chat"
    PERPLEXITY_SONAR_MEDIUM_ONLINE = "perplexity/sonar-medium-online"
    PERPLEXITY_SONAR_PRO = "perplexity/sonar-pro"
    PERPLEXITY_SONAR_REASONING = "perplexity/sonar-reasoning"
    PERPLEXITY_SONAR_REASONING_PRO = "perplexity/sonar-reasoning-pro"
    PERPLEXITY_SONAR_SMALL_CHAT = "perplexity/sonar-small-chat"
    PERPLEXITY_SONAR_SMALL_ONLINE = "perplexity/sonar-small-online"

    # RECRAFT models
    RECRAFT_RECRAFTV2 = "recraft/recraftv2"
    RECRAFT_RECRAFTV3 = "recraft/recraftv3"

    # REPLICATE models
    REPLICATE_DOLLY_V2_12B_EF0E1AEFC61F8E096EBE4DB6B2BACC297DAF2EF6899F0F7E001EC445893500E5 = "replicate/dolly-v2-12b:ef0e1aefc61f8e096ebe4db6b2bacc297daf2ef6899f0f7e001ec445893500e5"
    REPLICATE_LLAMA_2_70B_CHAT_2796EE9483C3FD7AA2E171D38F4CA12251A30609463DCFD4CD76703F22E96CDF = "replicate/llama-2-70b-chat:2796ee9483c3fd7aa2e171d38f4ca12251a30609463dcfd4cd76703f22e96cdf"
    REPLICATE_VICUNA_13B_6282ABE6A492DE4145D7BB601023762212F9DDBBE78278BD6771C8B3B2F2A13B = "replicate/vicuna-13b:6282abe6a492de4145d7bb601023762212f9ddbbe78278bd6771c8b3b2f2a13b"

    # REPLIT models
    REPLIT_REPLIT_CODE_V1_3B_B84F4C074B807211CD75E3E8B1589B6399052125B4C27106E43D47189E8415AD = "replit/replit-code-v1-3b:b84f4c074b807211cd75e3e8b1589b6399052125b4c27106e43d47189e8415ad"

    # SAMBANOVA models
    SAMBANOVA_DEEPSEEK_R1 = "sambanova/DeepSeek-R1"
    SAMBANOVA_DEEPSEEK_R1_DISTILL_LLAMA_70B = "sambanova/DeepSeek-R1-Distill-Llama-70B"
    SAMBANOVA_DEEPSEEK_V3_0324 = "sambanova/DeepSeek-V3-0324"
    SAMBANOVA_DEEPSEEK_V3_1 = "sambanova/DeepSeek-V3.1"
    SAMBANOVA_LLAMA_4_MAVERICK_17B_128E_INSTRUCT = "sambanova/Llama-4-Maverick-17B-128E-Instruct"
    SAMBANOVA_LLAMA_4_SCOUT_17B_16E_INSTRUCT = "sambanova/Llama-4-Scout-17B-16E-Instruct"
    SAMBANOVA_META_LLAMA_3_1_405B_INSTRUCT = "sambanova/Meta-Llama-3.1-405B-Instruct"
    SAMBANOVA_META_LLAMA_3_1_8B_INSTRUCT = "sambanova/Meta-Llama-3.1-8B-Instruct"
    SAMBANOVA_META_LLAMA_3_2_1B_INSTRUCT = "sambanova/Meta-Llama-3.2-1B-Instruct"
    SAMBANOVA_META_LLAMA_3_2_3B_INSTRUCT = "sambanova/Meta-Llama-3.2-3B-Instruct"
    SAMBANOVA_META_LLAMA_3_3_70B_INSTRUCT = "sambanova/Meta-Llama-3.3-70B-Instruct"
    SAMBANOVA_META_LLAMA_GUARD_3_8B = "sambanova/Meta-Llama-Guard-3-8B"
    SAMBANOVA_QWQ_32B = "sambanova/QwQ-32B"
    SAMBANOVA_QWEN2_AUDIO_7B_INSTRUCT = "sambanova/Qwen2-Audio-7B-Instruct"
    SAMBANOVA_QWEN3_32B = "sambanova/Qwen3-32B"
    SAMBANOVA_GPT_OSS_120B = "sambanova/gpt-oss-120b"

    # SNOWFLAKE models
    SNOWFLAKE_CLAUDE_3_5_SONNET = "snowflake/claude-3-5-sonnet"
    SNOWFLAKE_DEEPSEEK_R1 = "snowflake/deepseek-r1"
    SNOWFLAKE_GEMMA_7B = "snowflake/gemma-7b"
    SNOWFLAKE_JAMBA_1_5_LARGE = "snowflake/jamba-1.5-large"
    SNOWFLAKE_JAMBA_1_5_MINI = "snowflake/jamba-1.5-mini"
    SNOWFLAKE_JAMBA_INSTRUCT = "snowflake/jamba-instruct"
    SNOWFLAKE_LLAMA2_70B_CHAT = "snowflake/llama2-70b-chat"
    SNOWFLAKE_LLAMA3_70B = "snowflake/llama3-70b"
    SNOWFLAKE_LLAMA3_8B = "snowflake/llama3-8b"
    SNOWFLAKE_LLAMA3_1_405B = "snowflake/llama3.1-405b"
    SNOWFLAKE_LLAMA3_1_70B = "snowflake/llama3.1-70b"
    SNOWFLAKE_LLAMA3_1_8B = "snowflake/llama3.1-8b"
    SNOWFLAKE_LLAMA3_2_1B = "snowflake/llama3.2-1b"
    SNOWFLAKE_LLAMA3_2_3B = "snowflake/llama3.2-3b"
    SNOWFLAKE_LLAMA3_3_70B = "snowflake/llama3.3-70b"
    SNOWFLAKE_MISTRAL_7B = "snowflake/mistral-7b"
    SNOWFLAKE_MISTRAL_LARGE = "snowflake/mistral-large"
    SNOWFLAKE_MISTRAL_LARGE2 = "snowflake/mistral-large2"
    SNOWFLAKE_MIXTRAL_8X7B = "snowflake/mixtral-8x7b"
    SNOWFLAKE_REKA_CORE = "snowflake/reka-core"
    SNOWFLAKE_REKA_FLASH = "snowflake/reka-flash"
    SNOWFLAKE_SNOWFLAKE_ARCTIC = "snowflake/snowflake-arctic"
    SNOWFLAKE_SNOWFLAKE_LLAMA_3_1_405B = "snowflake/snowflake-llama-3.1-405b"
    SNOWFLAKE_SNOWFLAKE_LLAMA_3_3_70B = "snowflake/snowflake-llama-3.3-70b"

    # STABILITY models
    STABILITY_SD3_5_LARGE_V1_0 = "stability.sd3-5-large-v1:0"
    STABILITY_SD3_LARGE_V1_0 = "stability.sd3-large-v1:0"
    STABILITY_STABLE_IMAGE_CORE_V1_0 = "stability.stable-image-core-v1:0"
    STABILITY_STABLE_IMAGE_CORE_V1_1 = "stability.stable-image-core-v1:1"
    STABILITY_STABLE_IMAGE_ULTRA_V1_0 = "stability.stable-image-ultra-v1:0"
    STABILITY_STABLE_IMAGE_ULTRA_V1_1 = "stability.stable-image-ultra-v1:1"

    # STANDARD models
    STANDARD_1024_X_1024_DALL_E_3 = "standard/1024-x-1024/dall-e-3"
    STANDARD_1024_X_1792_DALL_E_3 = "standard/1024-x-1792/dall-e-3"
    STANDARD_1792_X_1024_DALL_E_3 = "standard/1792-x-1024/dall-e-3"

    # TEXT-COMPLETION-CODESTRAL models
    TEXT_COMPLETION_CODESTRAL_CODESTRAL_2405 = "text-completion-codestral/codestral-2405"
    TEXT_COMPLETION_CODESTRAL_CODESTRAL_LATEST = "text-completion-codestral/codestral-latest"

    # TOGETHERCOMPUTER models
    TOGETHERCOMPUTER_CODELLAMA_34B = "togethercomputer/CodeLlama-34b"
    TOGETHERCOMPUTER_CODELLAMA_34B_INSTRUCT = "togethercomputer/CodeLlama-34b-Instruct"
    TOGETHERCOMPUTER_CODELLAMA_34B_PYTHON = "togethercomputer/CodeLlama-34b-Python"
    TOGETHERCOMPUTER_LLAMA_2_7B_32K = "togethercomputer/LLaMA-2-7B-32K"
    TOGETHERCOMPUTER_LLAMA_2_7B_32K_INSTRUCT = "togethercomputer/Llama-2-7B-32K-Instruct"
    TOGETHERCOMPUTER_ALPACA_7B = "togethercomputer/alpaca-7b"
    TOGETHERCOMPUTER_FALCON_40B_INSTRUCT = "togethercomputer/falcon-40b-instruct"
    TOGETHERCOMPUTER_FALCON_7B_INSTRUCT = "togethercomputer/falcon-7b-instruct"
    TOGETHERCOMPUTER_LLAMA_2_70B = "togethercomputer/llama-2-70b"
    TOGETHERCOMPUTER_LLAMA_2_70B_CHAT = "togethercomputer/llama-2-70b-chat"
    TOGETHERCOMPUTER_LLAMA_2_7B = "togethercomputer/llama-2-7b"

    # UPSTAGE models
    UPSTAGE_SOLAR_0_70B_16BIT = "upstage/SOLAR-0-70b-16bit"

    # V0 models
    V0_V0_1_0_MD = "v0/v0-1.0-md"
    V0_V0_1_5_LG = "v0/v0-1.5-lg"
    V0_V0_1_5_MD = "v0/v0-1.5-md"

    # VOYAGE models
    VOYAGE_RERANK_2 = "voyage/rerank-2"
    VOYAGE_RERANK_2_LITE = "voyage/rerank-2-lite"
    VOYAGE_VOYAGE_2 = "voyage/voyage-2"
    VOYAGE_VOYAGE_3 = "voyage/voyage-3"
    VOYAGE_VOYAGE_3_LARGE = "voyage/voyage-3-large"
    VOYAGE_VOYAGE_3_LITE = "voyage/voyage-3-lite"
    VOYAGE_VOYAGE_CODE_2 = "voyage/voyage-code-2"
    VOYAGE_VOYAGE_CODE_3 = "voyage/voyage-code-3"
    VOYAGE_VOYAGE_CONTEXT_3 = "voyage/voyage-context-3"
    VOYAGE_VOYAGE_FINANCE_2 = "voyage/voyage-finance-2"
    VOYAGE_VOYAGE_LARGE_2 = "voyage/voyage-large-2"
    VOYAGE_VOYAGE_LAW_2 = "voyage/voyage-law-2"
    VOYAGE_VOYAGE_LITE_01 = "voyage/voyage-lite-01"
    VOYAGE_VOYAGE_LITE_02_INSTRUCT = "voyage/voyage-lite-02-instruct"
    VOYAGE_VOYAGE_MULTIMODAL_3 = "voyage/voyage-multimodal-3"

    # WATSONX models
    WATSONX_BIGSCIENCE_MT0_XXL_13B = "watsonx/bigscience/mt0-xxl-13b"
    WATSONX_CORE42_JAIS_13B_CHAT = "watsonx/core42/jais-13b-chat"
    WATSONX_GOOGLE_FLAN_T5_XL_3B = "watsonx/google/flan-t5-xl-3b"
    WATSONX_IBM_GRANITE_13B_CHAT_V2 = "watsonx/ibm/granite-13b-chat-v2"
    WATSONX_IBM_GRANITE_13B_INSTRUCT_V2 = "watsonx/ibm/granite-13b-instruct-v2"
    WATSONX_IBM_GRANITE_3_3_8B_INSTRUCT = "watsonx/ibm/granite-3-3-8b-instruct"
    WATSONX_IBM_GRANITE_3_8B_INSTRUCT = "watsonx/ibm/granite-3-8b-instruct"
    WATSONX_IBM_GRANITE_4_H_SMALL = "watsonx/ibm/granite-4-h-small"
    WATSONX_IBM_GRANITE_GUARDIAN_3_2_2B = "watsonx/ibm/granite-guardian-3-2-2b"
    WATSONX_IBM_GRANITE_GUARDIAN_3_3_8B = "watsonx/ibm/granite-guardian-3-3-8b"
    WATSONX_IBM_GRANITE_TTM_1024_96_R2 = "watsonx/ibm/granite-ttm-1024-96-r2"
    WATSONX_IBM_GRANITE_TTM_1536_96_R2 = "watsonx/ibm/granite-ttm-1536-96-r2"
    WATSONX_IBM_GRANITE_TTM_512_96_R2 = "watsonx/ibm/granite-ttm-512-96-r2"
    WATSONX_IBM_GRANITE_VISION_3_2_2B = "watsonx/ibm/granite-vision-3-2-2b"
    WATSONX_META_LLAMA_LLAMA_3_2_11B_VISION_INSTRUCT = "watsonx/meta-llama/llama-3-2-11b-vision-instruct"
    WATSONX_META_LLAMA_LLAMA_3_2_1B_INSTRUCT = "watsonx/meta-llama/llama-3-2-1b-instruct"
    WATSONX_META_LLAMA_LLAMA_3_2_3B_INSTRUCT = "watsonx/meta-llama/llama-3-2-3b-instruct"
    WATSONX_META_LLAMA_LLAMA_3_2_90B_VISION_INSTRUCT = "watsonx/meta-llama/llama-3-2-90b-vision-instruct"
    WATSONX_META_LLAMA_LLAMA_3_3_70B_INSTRUCT = "watsonx/meta-llama/llama-3-3-70b-instruct"
    WATSONX_META_LLAMA_LLAMA_4_MAVERICK_17B = "watsonx/meta-llama/llama-4-maverick-17b"
    WATSONX_META_LLAMA_LLAMA_GUARD_3_11B_VISION = "watsonx/meta-llama/llama-guard-3-11b-vision"
    WATSONX_MISTRALAI_MISTRAL_LARGE = "watsonx/mistralai/mistral-large"
    WATSONX_MISTRALAI_MISTRAL_MEDIUM_2505 = "watsonx/mistralai/mistral-medium-2505"
    WATSONX_MISTRALAI_MISTRAL_SMALL_2503 = "watsonx/mistralai/mistral-small-2503"
    WATSONX_MISTRALAI_MISTRAL_SMALL_3_1_24B_INSTRUCT_2503 = "watsonx/mistralai/mistral-small-3-1-24b-instruct-2503"
    WATSONX_MISTRALAI_PIXTRAL_12B_2409 = "watsonx/mistralai/pixtral-12b-2409"
    WATSONX_OPENAI_GPT_OSS_120B = "watsonx/openai/gpt-oss-120b"
    WATSONX_SDAIA_ALLAM_1_13B_INSTRUCT = "watsonx/sdaia/allam-1-13b-instruct"

    # WIZARDLM models
    WIZARDLM_WIZARDCODER_15B_V1_0 = "WizardLM/WizardCoder-15B-V1.0"
    WIZARDLM_WIZARDCODER_PYTHON_34B_V1_0 = "WizardLM/WizardCoder-Python-34B-V1.0"
    WIZARDLM_WIZARDLM_70B_V1_0 = "WizardLM/WizardLM-70B-V1.0"

    # XAI models
    XAI_GROK_2 = "xai/grok-2"
    XAI_GROK_2_1212 = "xai/grok-2-1212"
    XAI_GROK_2_LATEST = "xai/grok-2-latest"
    XAI_GROK_2_VISION = "xai/grok-2-vision"
    XAI_GROK_2_VISION_1212 = "xai/grok-2-vision-1212"
    XAI_GROK_2_VISION_LATEST = "xai/grok-2-vision-latest"
    XAI_GROK_3 = "xai/grok-3"
    XAI_GROK_3_BETA = "xai/grok-3-beta"
    XAI_GROK_3_FAST_BETA = "xai/grok-3-fast-beta"
    XAI_GROK_3_FAST_LATEST = "xai/grok-3-fast-latest"
    XAI_GROK_3_LATEST = "xai/grok-3-latest"
    XAI_GROK_3_MINI = "xai/grok-3-mini"
    XAI_GROK_3_MINI_BETA = "xai/grok-3-mini-beta"
    XAI_GROK_3_MINI_FAST = "xai/grok-3-mini-fast"
    XAI_GROK_3_MINI_FAST_BETA = "xai/grok-3-mini-fast-beta"
    XAI_GROK_3_MINI_FAST_LATEST = "xai/grok-3-mini-fast-latest"
    XAI_GROK_3_MINI_LATEST = "xai/grok-3-mini-latest"
    XAI_GROK_4 = "xai/grok-4"
    XAI_GROK_4_0709 = "xai/grok-4-0709"
    XAI_GROK_4_FAST_NON_REASONING = "xai/grok-4-fast-non-reasoning"
    XAI_GROK_4_FAST_REASONING = "xai/grok-4-fast-reasoning"
    XAI_GROK_4_LATEST = "xai/grok-4-latest"
    XAI_GROK_BETA = "xai/grok-beta"
    XAI_GROK_CODE_FAST = "xai/grok-code-fast"
    XAI_GROK_CODE_FAST_1 = "xai/grok-code-fast-1"
    XAI_GROK_CODE_FAST_1_0825 = "xai/grok-code-fast-1-0825"
    XAI_GROK_VISION_BETA = "xai/grok-vision-beta"


# Backwards-compatible alias
GPT4o = Models.GPT4O

# Convenience aliases for Computer Use
OPENAI_COMPUTER_USE_MINI = Models.O4_MINI
OPENAI_COMPUTER_USE = Models.O4