import numpy as np
import pytest

from bec_server.scan_server.path_optimization import PathOptimizerMixin
from bec_server.scan_server.scans import get_fermat_spiral_pos, get_round_roi_scan_positions


def test_shell_optimization():
    optim = PathOptimizerMixin()
    positions_orig = get_fermat_spiral_pos(-5, 5, -5, 5, 0.5)
    step = 2
    drs = np.linspace(step - step / 2, step + step / 2, 100)
    min_length = len(positions_orig)
    for dr in drs:
        optim_positions = optim.optimize_shell(positions_orig, 1, dr)
        assert optim.get_path_length(optim_positions) < optim.get_path_length(positions_orig)
        assert len(positions_orig) == len(optim_positions)
        if min_length > len(optim_positions):
            min_length = len(optim_positions)


@pytest.mark.parametrize(
    "positions_orig",
    [
        (get_fermat_spiral_pos(-5, 5, -5, 5, 0.5)),
        (get_round_roi_scan_positions(10, 10, 1, 3)),
        (get_fermat_spiral_pos(5, 15, -5, 5, 1.5)),
        np.asarray(
            [
                [-0.38502947, -0.42030026],
                [0.8030152, 0.07047403],
                [-0.78349739, 0.6006928],
                [0.19856742, -1.12257337],
                [0.68409143, 1.07541569],
                [-1.34834023, -0.36246191],
                [1.33834167, -0.69508386],
                [-0.55304807, 1.51437705],
                [-0.65246146, -1.5806309],
                [1.63258446, 0.76398167],
                [-1.80382449, 0.565789],
                [0.99004828, -1.70839234],
                [0.44128369, 2.00722911],
                [-1.74471832, -1.22660425],
                [2.18929536, -0.28370026],
                [-1.46933912, 1.74339971],
                [-0.09710392, -2.34816329],
                [1.70582397, 1.71416585],
                [-2.48192038, -0.11476599],
                [1.95717083, -1.63324289],
                [-0.34831142, 2.58874084],
                [-1.52694212, -2.19459512],
                [2.66696917, 0.59997953],
                [-2.42283314, 1.38833698],
                [0.86621469, -2.71517442],
                [1.21902689, 2.63844148],
                [-2.73218753, -1.14343837],
                [2.83815367, -1.02082504],
                [-1.42804626, 2.71712787],
                [-0.79577191, -3.01889832],
                [2.66942208, 1.71641654],
                [-3.177819, 0.5461377],
                [2.00492546, -2.58881708],
                [0.27441684, 3.31229458],
                [-2.47538823, -2.28996793],
                [3.41995931, 0.01668371],
                [-2.56798139, 2.32954321],
                [0.32426076, -3.4987219],
                [2.15202209, 2.83547191],
                [-3.54678327, -0.6452352],
                [3.0890415, -1.94389368],
                [-0.97637529, 3.56265228],
                [-1.70654861, -3.32541604],
                [3.54515953, 1.3143226],
                [-3.54147324, 1.44168904],
                [1.65561995, -3.49346856],
                [1.15131537, 3.73427006],
                [-3.40708457, -1.99674103],
                [3.90106934, -0.83770999],
                [-2.33412125, 3.28586031],
                [-0.50341829, -4.03936505],
                [3.12999911, 2.66418947],
                [-4.14690612, 0.15122708],
                [2.98340025, -2.94005492],
                [-0.21585929, 4.22171822],
                [-2.71692933, -3.28826626],
                [4.26212356, 0.59464507],
                [-3.57539059, 2.46186558],
                [0.98177109, -4.2667582],
                [2.17643951, 3.84149855],
                [-4.2345869, -1.37374444],
                [4.0834687, -1.86254756],
                [-1.7669696, 4.16491518],
                [-1.5223919, -4.29836282],
                [4.0573985, 2.15778066],
                [-4.48345448, 1.15846274],
                [2.54247449, -3.9120485],
                [0.77351793, 4.63625603],
                [-3.72923608, -2.91734439],
                [4.75454363, -0.3705602],
                [-3.2787141, 3.50969142],
                [0.04718809, -4.83638018],
                [3.25450081, 3.62297177],
                [-4.88013593, -0.47631219],
                [3.9466036, -2.96510034],
                [-0.91323418, 4.88450646],
                [-2.6432664, -4.24622688],
                [4.84852799, 1.35424381],
                [-4.51862215, 2.2911032],
                [1.79553073, -4.77158982],
                [1.91102723, 4.76076411],
                [-4.65344365, -2.2332179],
                [-2.66339579, 4.49420993],
                [-1.07825144, -5.14333295],
                [4.29438088, 3.08215718],
                [3.48563219, -4.05482037],
                [0.16975146, 5.37468924],
                [-3.77676048, -3.8700233],
                [-4.23163998, 3.46179478],
                [0.78610866, -5.44037987],
                [3.11186849, 4.56693272],
                [-1.75834396, 5.33137192],
                [-2.31659152, -5.1452506],
                [5.21012991, 2.24045673],
                [2.71438699, -5.04450228],
                [1.41295113, 5.58062444],
                [-4.83505221, -3.17599908],
                [-3.62119968, 4.58272985],
                [-0.42745065, -5.85292115],
                [4.28886872, 4.04597394],
                [4.44642124, -3.95517865],
                [-0.60974575, 5.94703372],
                [-3.5837355, -4.81879029],
                [1.66577589, -5.85352806],
                [2.73763886, 5.46523865],
                [-2.70615335, 5.56910532],
                [-1.77391247, -5.95955825],
                [5.35601502, 3.20953316],
                [3.69591707, -5.09686149],
                [0.72056133, 6.28040535],
                [-4.79303825, -4.16105568],
                [0.39058698, -6.41168791],
                [4.05891248, 5.01123034],
                [-1.52487763, 6.34323642],
                [-3.17243698, -5.7290875],
                [2.64593869, -6.07128557],
                [2.15801013, 6.28724044],
                [-3.71687041, 5.59874757],
                [-1.04493367, -6.66289079],
                [5.29001191, 4.2221646],
                [-0.13346172, 6.83869783],
                [-4.53673446, -5.15058641],
                [1.34084289, -6.80348002],
                [3.61915416, 5.94237521],
                [-2.53902039, 6.55272275],
                [-2.56249389, -6.56835787],
                [3.68917794, -6.08887232],
                [1.39714577, 7.0040905],
                [-0.1577744, -7.23076809],
                [5.01615753, 5.24137039],
                [-1.11773817, 7.23596306],
                [-4.07593292, -6.10902372],
                [2.38947313, -7.01416553],
                [2.98486959, 6.80614087],
                [-3.61674404, 6.56710458],
                [-1.77424666, -7.30647992],
                [0.47989035, 7.5895919],
                [-5.49592513, -5.28695631],
                [0.85890639, -7.64160191],
                [4.54092097, 6.23191277],
                [-2.20059676, 7.45578795],
                [-3.42282504, -7.00289717],
                [3.50268751, -7.03293539],
                [2.17360437, 7.57174643],
                [-0.83007366, -7.91621612],
                [-0.56720675, 8.02080897],
                [-5.01228442, -6.31324044],
                [1.97517293, -7.87739753],
                [3.87418174, 7.16032931],
                [-3.34960106, 7.48561773],
                [-2.59281078, -7.80105969],
                [1.2058016, 8.21126315],
                [0.24516539, -8.3736667],
                [5.48821351, 6.35474724],
                [-1.71562742, 8.27856404],
                [-4.33686242, -7.27973381],
                [3.15971673, -7.92425329],
                [3.02961939, 7.99524273],
                [-1.60475719, -8.47507253],
                [0.10492593, 8.70003969],
                [4.80886862, 7.36213166],
                [-3.48190329, -8.15488501],
                [2.02477238, 8.70779518],
                [-0.48094298, -8.99964965],
                [-5.28826502, -7.40835698],
                [3.94762603, 8.2804196],
                [-2.46378877, -8.90945817],
                [0.8808825, 9.27212737],
                [5.77316847, 7.41911894],
                [-4.42482118, -8.37217758],
                [2.91982982, 9.08001067],
                [4.91157785, 8.43042721],
                [-3.39098098, -9.21935724],
                [-5.40603031, -8.45540279],
                [3.87537543, 9.32738255],
                [5.90635073, 8.44732626],
                [-4.37118372, -9.40397006],
                [4.87660641, 9.44901635],
                [-5.38986893, -9.46244223],
                [5.90921802, 9.44420152],
                [-4.19567308, -10.38305],
                [4.72175873, 10.42303672],
                [-5.25585093, -10.43235501],
                [5.79626528, 10.4107929],
                [-6.34132488, -10.35819476],
            ]
        ),
    ],
)
def test_corridor_optimization(positions_orig):
    optim = PathOptimizerMixin()
    optim_positions = optim.optimize_corridor(positions_orig, num_iterations=20)
    assert optim.get_path_length(optim_positions) < optim.get_path_length(positions_orig)
    assert len(positions_orig) == len(optim_positions)

    # from matplotlib import pyplot as plt

    # plt.figure()
    # plt.plot(positions_orig[:, 0], positions_orig[:, 1], "r-x")
    # plt.plot(optim_positions[:, 0], optim_positions[:, 1], "g-x")
    # plt.show()
