import os
import subprocess
import tempfile
from datetime import date
from pathlib import Path

from wpiformat.config import Config
from wpiformat.licenseupdate import LicenseUpdate

from .test_tasktest import *


class OpenTemporaryDirectory:
    def __init__(self):
        self.prev_dir = os.getcwd()

    def __enter__(self):
        self.temp_dir = tempfile.TemporaryDirectory()
        os.chdir(self.temp_dir.name)
        return self.temp_dir

    def __exit__(self, type, value, traceback):
        os.chdir(self.prev_dir)


def test_licenseupdate():
    excluded_h = Path("./Excluded.h").resolve()
    test_h = Path("./Test.h").resolve()

    year = str(date.today().year)

    file_appendix = """#pragma once

#include <iostream>

int main() {
  std::cout << "Hello World!";
}"""

    # pragma once at top of file
    run_and_check_file(
        LicenseUpdate(),
        test_h,
        file_appendix,
        f"""/*                                Company Name                                */
/* Copyright (c) {year} Company Name. All Rights Reserved.                      */

"""
        + file_appendix,
        True,
    )

    # pragma once at top of file preceded by newline
    run_and_check_file(
        LicenseUpdate(),
        test_h,
        "\n" + file_appendix,
        f"""/*                                Company Name                                */
/* Copyright (c) {year} Company Name. All Rights Reserved.                      */

"""
        + file_appendix,
        True,
    )

    # File containing up-to-date license preceded by newline
    run_and_check_file(
        LicenseUpdate(),
        test_h,
        f"""

/*                                Company Name                                */
/* Copyright (c) {year} Company Name. All Rights Reserved.                      */

"""
        + file_appendix,
        f"""/*                                Company Name                                */
/* Copyright (c) {year} Company Name. All Rights Reserved.                      */

"""
        + file_appendix,
        True,
    )

    # File containing up-to-date range license
    run_and_check_file(
        LicenseUpdate(),
        test_h,
        f"""/*                                Company Name                                */
// Copyright (c) 2011-{year} Company Name. All Rights Reserved.

"""
        + file_appendix,
        f"""/*                                Company Name                                */
/* Copyright (c) 2011-{year} Company Name. All Rights Reserved.                 */

"""
        + file_appendix,
        True,
    )

    # File containing up-to-date license with one year
    contents = (
        f"""/*                                Company Name                                */
/* Copyright (c) {year} Company Name. All Rights Reserved.                      */

"""
        + file_appendix
    )
    run_and_check_file(LicenseUpdate(), test_h, contents, contents, True)

    # File with three newlines between license and include guard
    run_and_check_file(
        LicenseUpdate(),
        test_h,
        f"""/*                                Company Name                                */
/* Copyright (c) {year} Company Name. All Rights Reserved.                      */


"""
        + file_appendix,
        f"""/*                                Company Name                                */
/* Copyright (c) {year} Company Name. All Rights Reserved.                      */

"""
        + file_appendix,
        True,
    )

    # File with only one newline between license and include guard
    run_and_check_file(
        LicenseUpdate(),
        test_h,
        f"""/*                                Company Name                                */
/* Copyright (c) {year} Company Name. All Rights Reserved.                      */
"""
        + file_appendix,
        f"""/*                                Company Name                                */
/* Copyright (c) {year} Company Name. All Rights Reserved.                      */

"""
        + file_appendix,
        True,
    )

    # File with multiline comment spanning multiple lines of license header
    run_and_check_file(
        LicenseUpdate(),
        test_h,
        """/*
 * Autogenerated file! Do not manually edit this file. This version is regenerated
 * any time the publish task is run, or when this file is deleted.
 */

const char* WPILibVersion = "";""",
        f"""/*                                Company Name                                */
/* Copyright (c) {year} Company Name. All Rights Reserved.                      */

/*
 * Autogenerated file! Do not manually edit this file. This version is regenerated
 * any time the publish task is run, or when this file is deleted.
 */

const char* WPILibVersion = "";""",
        True,
    )

    # File containing license year range in different postion than template
    # (If the year isn't extracted, the range will be replaced with one year and
    # the test will fail.)
    run_and_check_file(
        LicenseUpdate(),
        test_h,
        f"""/*                                Company Name                                */
/* Copyright (c) Company Name 2011-{year}. All Rights Reserved.                 */

"""
        + file_appendix,
        f"""/*                                Company Name                                */
/* Copyright (c) 2011-{year} Company Name. All Rights Reserved.                 */

"""
        + file_appendix,
        True,
    )

    # Ensure "/*" after "*/" on same line is detected
    run_and_check_file(
        LicenseUpdate(),
        test_h,
        """/*----------------------------------------------------------------------------*/
/* Copyright (c) 2011 FIRST. All Rights Reserved.                             */
/* Open Source Software - may be modified and shared by FRC teams. The code   */
/* must be accompanied by the FIRST BSD license file in the root directory of */
/* the project.                                                               */
/*----------------------------------------------------------------------------*//*

blah

*/
""",
        f"""/*                                Company Name                                */
/* Copyright (c) 2011-{year} Company Name. All Rights Reserved.                 */

""",
        True,
    )

    # File excluded from license update isn't modified
    contents = f"""/* Copyright (c) Company Name 2011-{year}. */

"""
    run_and_check_file(LicenseUpdate(), excluded_h, contents, contents, True)

    # Ensure license regex matches
    run_and_check_file(
        LicenseUpdate(),
        test_h,
        """/* Company Name */
/* Copyright (c) 1992-2015 Company Name. All Rights Reserved. */

"""
        + file_appendix,
        f"""/*                                Company Name                                */
/* Copyright (c) 1992-{year} Company Name. All Rights Reserved.                 */

"""
        + file_appendix,
        True,
    )

    # Ensure excluded files won't be processed
    config_file = Config(Path.cwd(), Path(".wpiformat"))
    assert not LicenseUpdate().should_process_file(config_file, excluded_h)

    # Create git repo to test license years for commits
    with OpenTemporaryDirectory():
        last_year_cpp = Path("last-year.cpp").resolve()
        next_year_cpp = Path("next-year.cpp").resolve()
        no_year_cpp = Path("no-year.cpp").resolve()
        this_year_cpp = Path("this-year.cpp").resolve()

        subprocess.run(["git", "init", "-q"])

        # Add base files
        Path(".wpiformat-license").write_text("// Copyright (c) {year}")
        Path(".wpiformat").write_text("cppSrcFileInclude {\n" + r"\.cpp$")
        subprocess.run(["git", "add", ".wpiformat-license"])
        subprocess.run(["git", "add", ".wpiformat"])
        subprocess.run(["git", "commit", "-q", "-m", '"Initial commit"'])

        # Add file with commit date of last year and range through this year
        Path("last-year.cpp").write_text(f"// Copyright (c) 2017-{year}")
        subprocess.run(["git", "add", "last-year.cpp"])
        subprocess.run(["git", "commit", "-q", "-m", '"Last year"'])
        last_iso_year = f"{int(year) - 1}-01-01T00:00:00"
        subprocess.Popen(
            ["git", "commit", "-q", "--amend", "--no-edit", f"--date={last_iso_year}"],
            env={**os.environ, "GIT_COMMITTER_DATE": last_iso_year},
        ).wait()

        # Add file with commit date of this year and range through this year
        Path("this-year.cpp").write_text(f"// Copyright (c) 2017-{year}")
        subprocess.run(["git", "add", "this-year.cpp"])
        subprocess.run(["git", "commit", "-q", "-m", '"This year"'])

        # Add file with commit date of next year and range through this year
        Path("next-year.cpp").write_text(f"// Copyright (c) 2017-{year}")
        subprocess.run(["git", "add", "next-year.cpp"])
        subprocess.run(["git", "commit", "-q", "-m", '"Next year"'])
        next_iso_year = f"{int(year) + 1}-01-01T00:00:00"
        subprocess.Popen(
            ["git", "commit", "-q", "--amend", "--no-edit", f"--date={next_iso_year}"],
            env={**os.environ, "GIT_COMMITTER_DATE": next_iso_year},
        ).wait()

        # Create uncommitted file with no year
        Path("no-year.cpp").touch()

        # Run wpiformat on last-year.cpp
        lines = Path("last-year.cpp").read_text()
        output, success = LicenseUpdate().run_pipeline(
            config_file, last_year_cpp, lines
        )
        assert output == f"// Copyright (c) 2017-{int(year) - 1}\n\n"

        # Run wpiformat on last-year.cpp with uncommitted changes. It should
        # update to next year instead of keeping previous year
        with open("last-year.cpp", "a") as f:
            f.write("change\n")
        output, success = LicenseUpdate().run_pipeline(
            config_file, last_year_cpp, lines + "change\n"
        )
        assert (
            output
            == f"""// Copyright (c) 2017-{year}

change
"""
        )

        # Erase changes made to last-year.cpp in previous test
        Path("last-year.cpp").write_text(lines)

        # Run wpiformat on this-year.cpp
        lines = Path("last-year.cpp").read_text()
        output, success = LicenseUpdate().run_pipeline(
            config_file, this_year_cpp, lines
        )
        assert (
            output
            == f"""// Copyright (c) 2017-{year}

"""
        )

        # Run wpiformat on next-year.cpp
        lines = Path("next-year.cpp").read_text()
        output, success = LicenseUpdate().run_pipeline(
            config_file, next_year_cpp, lines
        )
        assert (
            output
            == f"""// Copyright (c) 2017-{int(year) + 1}

"""
        )

        # Run wpiformat on no-year.cpp
        # Should have current calendar year
        lines = Path("no-year.cpp").read_text()
        output, success = LicenseUpdate().run_pipeline(config_file, no_year_cpp, lines)
        assert (
            output
            == f"""// Copyright (c) {year}

"""
        )

    # Create git repo to test filename expansion
    with OpenTemporaryDirectory():
        subprocess.run(["git", "init", "-q"])

        # Add base files
        Path(".wpiformat-license").write_text(
            """// Copyright (c) {year}
// https://github.com/wpilibsuite/styleguide/blob/main/{filename}
"""
        )
        Path(".wpiformat").write_text("cppSrcFileInclude {\n" + r"\.cpp$")
        subprocess.run(["git", "add", ".wpiformat-license"])
        subprocess.run(["git", "add", ".wpiformat"])
        subprocess.run(["git", "commit", "-q", "-m", '"Initial commit"'])

        # Create uncommitted empty file in subdirectory
        file = (Path("dir") / "empty.cpp").resolve()
        file.parent.mkdir(parents=True, exist_ok=True)
        file.touch()

        # Empty file
        output, success = LicenseUpdate().run_pipeline(config_file, file, "")
        assert (
            output
            == f"""// Copyright (c) {year}
// https://github.com/wpilibsuite/styleguide/blob/main/dir/empty.cpp

"""
        )
