"""Meltano run command and supporting functions."""

from __future__ import annotations

import time
import typing as t
import uuid

import click
import structlog

from meltano.cli.params import (
    InstallPlugins,
    UUIDParamType,
    get_install_options,
    pass_project,
)
from meltano.cli.utils import CliEnvironmentBehavior, CliError, PartialInstrumentedCmd
from meltano.core._state import StateStrategy
from meltano.core.block.block_parser import BlockParser, validate_block_sets
from meltano.core.block.blockset import BlockSet
from meltano.core.block.plugin_command import PluginCommandBlock
from meltano.core.logging.utils import change_console_log_level
from meltano.core.plugin_install_service import PluginInstallReason
from meltano.core.project_settings_service import ProjectSettingsService
from meltano.core.runner import RunnerError
from meltano.core.tracking import BlockEvents, Tracker
from meltano.core.tracking.contexts import CliEvent
from meltano.core.tracking.contexts.plugins import PluginsTrackingContext
from meltano.core.utils import run_async

if t.TYPE_CHECKING:
    from meltano.core.project import Project

logger = structlog.getLogger(__name__)

install, no_install, only_install = get_install_options(include_only_install=True)


@click.command(
    cls=PartialInstrumentedCmd,
    short_help="Run a set of plugins in series.",
    environment_behavior=CliEnvironmentBehavior.environment_required,
)
@click.option(
    "--dry-run",
    help=(
        "Do not run, just parse the invocation, validate it, and explain what "
        "would be executed."
    ),
    is_flag=True,
)
@click.option(
    "--full-refresh",
    help=(
        "Perform a full refresh (ignore state left behind by any previous "
        "runs). Applies to all pipelines."
    ),
    envvar="MELTANO_RUN_FULL_REFRESH",
    is_flag=True,
)
@click.option(
    "--refresh-catalog",
    help="Invalidates catalog cache and forces running discovery before this run.",
    is_flag=True,
)
@click.option(
    "--no-state-update",
    help="Run without state saving. Applies to all pipelines.",
    is_flag=True,
)
@click.option(
    "--force",
    "-f",
    help=(
        "Force a new run even if a pipeline with the same State ID is already "
        "present. Applies to all pipelines."
    ),
    is_flag=True,
)
@click.option(
    "--state-id-suffix",
    help="Define a custom suffix to autogenerate state IDs with.",
)
@click.option(
    "--merge-state",
    is_flag=True,
    help="Merges state with that of previous runs.",
    hidden=True,
)
@click.option(
    "--state-strategy",
    # TODO: use click.Choice(StateStrategy) once we drop support for Python 3.9 and use
    # click 8.2+ exclusively
    type=click.Choice([strategy.value for strategy in StateStrategy]),
    default=StateStrategy.AUTO.value,
    help="Strategy to use for state updates.",
)
@click.option(
    "--run-id",
    type=UUIDParamType(),
    help="Use a custom run ID.",
)
@click.argument(
    "blocks",
    nargs=-1,
)
@install
@no_install
@only_install
@pass_project(migrate=True)
@click.pass_context
@run_async
async def run(
    ctx: click.Context,
    project: Project,
    *,
    dry_run: bool,
    full_refresh: bool,
    refresh_catalog: bool,
    no_state_update: bool,
    force: bool,
    state_id_suffix: str,
    merge_state: bool,
    state_strategy: str,
    run_id: uuid.UUID | None,
    blocks: list[str],
    install_plugins: InstallPlugins,
) -> None:
    """Run a set of command blocks in series.

    Blocks are specified as either:\n
      - a list of plugin names\n
      - a job name\n
    An example of a list of plugin names is: `meltano run some_extractor some_loader some_plugin:some_optional_command`. These would be run in the order they are specified from left to right. A failure in any block will cause the entire run to abort.

    Multiple command blocks can be chained together or repeated, and tap/target pairs will automatically be linked:

        `meltano run tap-gitlab target-postgres dbt:test dbt:run`\n
        `meltano run tap-gitlab target-postgres tap-salesforce target-mysql ...`\n
        `meltano run tap-gitlab target-postgres dbt:run tap-postgres target-bigquery ...`\n

    When running within an active environment, meltano run activates incremental job support. State ID's are autogenerated
    using the format `{environment.name}:{extractor_name}-to-{loader_name}(:{state-id-suffix})` for each extract/load pair found:

        `meltano --environment=prod run tap-gitlab target-postgres tap-salesforce target-mysql`\n

    The above command will create two jobs with state IDs `prod:tap-gitlab-to-target-postgres` and `prod:tap-salesforce-to-target-mysql`.

    \b
    Read more at https://docs.meltano.com/reference/command-line-interface#run
    """  # noqa: D301, E501
    if dry_run and not ProjectSettingsService.config_override.get("cli.log_level"):
        logger.info("Setting 'console' handler log level to 'debug' for dry run")
        change_console_log_level()

    # Bind run_id at the start of the CLI entrypoint
    run_id = run_id or uuid.uuid4()
    structlog.contextvars.bind_contextvars(run_id=str(run_id))

    tracker: Tracker = ctx.obj["tracker"]

    _state_strategy = StateStrategy.from_cli_args(
        merge_state=merge_state,
        state_strategy=state_strategy,
    )

    try:
        parser = BlockParser(
            logger,
            project,
            blocks,
            full_refresh=full_refresh,
            refresh_catalog=refresh_catalog,
            no_state_update=no_state_update,
            force=force,
            state_id_suffix=state_id_suffix,
            state_strategy=_state_strategy,
            run_id=run_id,
        )
        parsed_blocks = list(parser.find_blocks(0))
        if not parsed_blocks:
            tracker.track_command_event(CliEvent.aborted)
            logger.info("No valid blocks found.")
            return
    except Exception as parser_err:
        tracker.track_command_event(CliEvent.aborted)
        raise parser_err

    if validate_block_sets(logger, parsed_blocks):
        logger.debug("All ExtractLoadBlocks validated, starting execution.")
    else:
        tracker.track_command_event(CliEvent.aborted)
        raise CliError("Some ExtractLoadBlocks set failed validation.")  # noqa: EM101

    await install_plugins(
        project,
        parser.plugins,
        reason=PluginInstallReason.AUTO,
    )

    run_start_time = time.perf_counter()
    success = False
    try:
        await _run_blocks(tracker, parsed_blocks, dry_run=dry_run)
        success = True
    except Exception as err:
        tracker.track_command_event(CliEvent.failed)
        raise err
    finally:
        run_end_time = time.perf_counter()
        total_duration = run_end_time - run_start_time
        logger.info(
            "Run completed",
            duration_seconds=round(total_duration, 3),
            status="success" if success else "failure",
        )
    tracker.track_command_event(CliEvent.completed)


async def _run_blocks(
    tracker: Tracker,
    parsed_blocks: list[BlockSet | PluginCommandBlock],
    *,
    dry_run: bool,
) -> None:
    for idx, blk in enumerate(parsed_blocks):
        blk_name = blk.__class__.__name__
        tracking_ctx = PluginsTrackingContext.from_block(blk)
        with tracker.with_contexts(tracking_ctx):
            tracker.track_block_event(blk_name, BlockEvents.initialized)
        if dry_run:
            msg = f"Dry run, but would have run block {idx + 1}/{len(parsed_blocks)}."
            if isinstance(blk, BlockSet):
                logger.info(
                    msg,
                    block_type=blk_name,
                    comprised_of=[plugin.string_id for plugin in blk.blocks],
                )
            elif isinstance(blk, PluginCommandBlock):
                logger.info(
                    msg,
                    block_type=blk_name,
                    comprised_of=f"{blk.string_id}:{blk.command}",
                )
            continue

        block_start_time = time.perf_counter()
        try:
            await blk.run()
        except RunnerError as err:
            block_end_time = time.perf_counter()
            block_duration = block_end_time - block_start_time
            logger.error(
                "Block run completed",
                set_number=idx,
                block_type=blk_name,
                success=False,
                err=err,
                exit_codes=err.exitcodes,
                duration_seconds=round(block_duration, 3),
            )
            with tracker.with_contexts(tracking_ctx):
                tracker.track_block_event(blk_name, BlockEvents.failed)
            raise CliError(
                f"Run invocation could not be completed as block failed: {err}",  # noqa: EM102
            ) from err
        except Exception as bare_err:
            # make sure we also fire block failed events for all other exceptions
            with tracker.with_contexts(tracking_ctx):
                tracker.track_block_event(blk_name, BlockEvents.failed)
            raise bare_err

        block_end_time = time.perf_counter()
        block_duration = block_end_time - block_start_time

        logger.info(
            "Block run completed",
            set_number=idx,
            block_type=blk_name,
            success=True,
            err=None,
            duration_seconds=round(block_duration, 3),
        )
        with tracker.with_contexts(tracking_ctx):
            tracker.track_block_event(blk_name, BlockEvents.completed)
