"""
Vault configuration management.
Stores vault addresses so users don't have to type them repeatedly.

SIMPLIFIED: Now only stores employee + admin wallets!
Everything else (vault PDA, token accounts) is derived automatically.
"""

import json
import os
from pathlib import Path
from .blockchain import derive_all_vault_addresses


class VaultConfig:
    """Manages vault configuration."""

    def __init__(self, config_path: str = None):
        if config_path is None:
            config_path = os.path.expanduser("~/.loggerheads_vault.json")
        self.config_path = config_path
        self.config = self.load()

    def load(self) -> dict:
        """Load configuration from file."""
        if os.path.exists(self.config_path):
            with open(self.config_path, 'r') as f:
                return json.load(f)
        return {}

    def save(self):
        """Save configuration to file."""
        with open(self.config_path, 'w') as f:
            json.dump(self.config, f, indent=2)

    def set_vault(self, employee_pubkey: str, admin_pubkey: str):
        """
        Store vault configuration - SIMPLIFIED!

        Now only requires 2 addresses. Everything else is derived automatically.

        Args:
            employee_pubkey: Employee wallet address
            admin_pubkey: Admin wallet address
        """
        self.config['vault'] = {
            'employee_pubkey': employee_pubkey,
            'admin_pubkey': admin_pubkey
        }
        self.save()

    def get_vault(self) -> dict:
        """
        Get vault configuration with ALL addresses derived automatically.

        Returns:
            Dict with employee_pubkey, admin_pubkey, vault_pda,
            vault_token_account, employee_token_account (all derived!)
        """
        stored = self.config.get('vault', {})

        if not stored:
            return {}

        # Derive all addresses from just the 2 stored addresses
        return derive_all_vault_addresses(
            stored['employee_pubkey'],
            stored['admin_pubkey']
        )

    def has_vault(self) -> bool:
        """Check if vault is configured."""
        return 'vault' in self.config and bool(self.config['vault'])

    def enable_auto_submit(self, enabled: bool = True, time: str = "18:00"):
        """Enable/disable automatic submission."""
        self.config['auto_submit'] = {
            'enabled': enabled,
            'time': time
        }
        self.save()

    def is_auto_submit_enabled(self) -> bool:
        """Check if auto-submit is enabled."""
        return self.config.get('auto_submit', {}).get('enabled', False)

    def get_auto_submit_time(self) -> str:
        """Get auto-submit time."""
        return self.config.get('auto_submit', {}).get('time', '18:00')

    def set_keypair_path(self, keypair_path: str):
        """
        Store default keypair path.

        Args:
            keypair_path: Path to user's Solana keypair file
        """
        self.config['keypair_path'] = os.path.expanduser(keypair_path)
        self.save()

    def get_keypair_path(self) -> str:
        """
        Get stored keypair path.

        Returns:
            Path to keypair file, or None if not set
        """
        return self.config.get('keypair_path')

    def has_keypair_path(self) -> bool:
        """Check if keypair path is configured."""
        return 'keypair_path' in self.config and bool(self.config['keypair_path'])

    def clear(self):
        """Clear all configuration."""
        self.config = {}
        self.save()

    def print_config(self, show_technical=False):
        """
        Print current configuration.

        Args:
            show_technical: If True, show technical blockchain details (PDAs, token accounts)
                          Default False - users don't need to see these!
        """
        if not self.has_vault():
            print("\n❌ No work account set up yet")
            print("\n💡 Get started: loggerheads setup-vault")
            return

        vault = self.get_vault()

        # Detect user's actual role
        try:
            from .blockchain import load_keypair
            my_wallet = str(load_keypair().pubkey())
            is_employee = (my_wallet == vault['employee_pubkey'])
            is_employer = (my_wallet == vault['admin_pubkey'])
        except:
            is_employee = False
            is_employer = False

        print("\n" + "="*60)
        print("📋 Vault Configuration")
        print("="*60)

        # Show your role prominently
        if is_employee and is_employer:
            print(f"\n🔑 Your Role: Employee & Employer (same wallet)")
        elif is_employee:
            print(f"\n🔑 Your Role: EMPLOYEE")
        elif is_employer:
            print(f"\n🔑 Your Role: EMPLOYER")
        else:
            print(f"\n🔑 Your Role: Viewer (not in this vault)")

        # Show vault participants
        print(f"\n👤 Employee Wallet:")
        print(f"   {vault['employee_pubkey'][:16]}...{vault['employee_pubkey'][-8:]}")
        if is_employee:
            print(f"   ✓ This is you")

        print(f"\n👔 Employer Wallet:")
        print(f"   {vault['admin_pubkey'][:16]}...{vault['admin_pubkey'][-8:]}")
        if is_employer and not is_employee:
            print(f"   ✓ This is you")

        # Check both config and actual cron job status
        from .cron_manager import check_auto_submit_status
        config_enabled = self.is_auto_submit_enabled()
        config_time = self.get_auto_submit_time()
        cron_status = check_auto_submit_status()
        cron_installed = cron_status['installed']

        print(f"\n⏰ Auto-Submit:")
        if config_enabled and cron_installed:
            print(f"   Status: ✅ Enabled and installed")
            print(f"   Time: {config_time}")
            print(f"   Your hours are automatically submitted at {config_time}")
        elif config_enabled and not cron_installed:
            print(f"   Status: ⚠️  Enabled but cron not installed")
            print(f"   Run: loggerheads enable-autosubmit")
        elif not config_enabled and cron_installed:
            print(f"   Status: ⚠️  Disabled but cron still installed")
            print(f"   Run: loggerheads disable-autosubmit")
        else:
            print(f"   Status: ❌ Disabled")
            print(f"   Enable with: loggerheads enable-autosubmit")
            print(f"   Or manually run: loggerheads submit")

        # Only show technical details if explicitly requested (for debugging)
        if show_technical:
            print(f"\n" + "-"*60)
            print(f"🔧 Technical Details (for debugging):")
            print(f"-"*60)
            print(f"   Vault PDA:      {vault['vault_pda'][:20]}...")
            print(f"   Vault Token:    {vault['vault_token_account'][:20]}...")
            print(f"   Employee Token: {vault['employee_token_account'][:20]}...")
            print(f"   Config file:    {self.config_path}")

        print("\n" + "="*60 + "\n")
