"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppMeshExtension = exports.Protocol = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const appmesh = require("aws-cdk-lib/aws-appmesh");
const ec2 = require("aws-cdk-lib/aws-ec2");
const ecr = require("aws-cdk-lib/aws-ecr");
const ecs = require("aws-cdk-lib/aws-ecs");
const iam = require("aws-cdk-lib/aws-iam");
const regionInfo = require("aws-cdk-lib/region-info");
const extension_interfaces_1 = require("./extension-interfaces");
// The version of the App Mesh envoy sidecar to add to the task.
const APP_MESH_ENVOY_SIDECAR_VERSION = 'v1.15.1.0-prod';
/**
 * Enum of supported AppMesh protocols
 */
var Protocol;
(function (Protocol) {
    Protocol["HTTP"] = "http";
    Protocol["TCP"] = "tcp";
    Protocol["HTTP2"] = "http2";
    Protocol["GRPC"] = "grpc";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * This extension adds an Envoy sidecar to the task definition and
 * creates the App Mesh resources required to route network traffic
 * to the container in a service mesh.
 *
 * The service will then be available to other App Mesh services at the
 * address `<service name>.<environment name>`. For example, a service called
 * `orders` deploying in an environment called `production` would be accessible
 * to other App Mesh enabled services at the address `http://orders.production`.
 */
class AppMeshExtension extends extension_interfaces_1.ServiceExtension {
    constructor(props) {
        super('appmesh');
        this.mesh = props.mesh;
        if (props.protocol) {
            this.protocol = props.protocol;
        }
        else {
            this.protocol = Protocol.HTTP;
        }
    }
    prehook(service, scope) {
        this.parentService = service;
        this.scope = scope;
        // Make sure that the parent cluster for this service has
        // a namespace attached.
        if (!this.parentService.cluster.defaultCloudMapNamespace) {
            this.parentService.environment.addDefaultCloudMapNamespace({
                // Name the namespace after the environment name.
                // Service DNS will be like <service id>.<environment id>
                name: this.parentService.environment.id,
            });
        }
    }
    modifyTaskDefinitionProps(props) {
        // Find the app extension, to get its port
        const containerextension = this.parentService.serviceDescription.get('service-container');
        if (!containerextension) {
            throw new Error('Appmesh extension requires an application extension');
        }
        return {
            ...props,
            // App Mesh requires AWS VPC networking mode so that each
            // task can have its own IP address
            networkMode: ecs.NetworkMode.AWS_VPC,
            // This configures the envoy container as a proxy for all
            // traffic going into and out of the task, with a few exceptions
            // for metadata endpoints or other ports that need direct
            // communication
            proxyConfiguration: new ecs.AppMeshProxyConfiguration({
                containerName: 'envoy',
                properties: {
                    appPorts: [containerextension.trafficPort],
                    proxyEgressPort: 15001,
                    proxyIngressPort: 15000,
                    // The App Mesh proxy runs with this user ID, and this keeps its
                    // own outbound connections from recursively attempting to infinitely proxy.
                    ignoredUID: 1337,
                    // This GID is ignored and any outbound traffic originating from containers that
                    // use this group ID will be ignored by the proxy. This is primarily utilized by
                    // the FireLens extension, so that outbound application logs don't have to go through Envoy
                    // and therefore add extra burden to the proxy sidecar. Instead the logs can go directly
                    // to CloudWatch
                    ignoredGID: 1338,
                    egressIgnoredIPs: [
                        '169.254.170.2',
                        '169.254.169.254',
                    ],
                    // If there is outbound traffic to specific ports that you want to
                    // ignore the proxy those ports can be added here.
                    egressIgnoredPorts: [],
                },
            }),
        };
    }
    accountIdForRegion(region) {
        return { ecrRepo: regionInfo.RegionInfo.get(region).appMeshRepositoryAccount };
    }
    useTaskDefinition(taskDefinition) {
        var region = aws_cdk_lib_1.Stack.of(this.scope).region;
        var partition = aws_cdk_lib_1.Stack.of(this.scope).partition;
        var appMeshRepo;
        // This is currently necessary because App Mesh has different images in each region,
        // and some regions have their images in a different account. See:
        // https://docs.aws.amazon.com/app-mesh/latest/userguide/envoy.html
        const mapping = new aws_cdk_lib_1.CfnMapping(this.scope, `${this.parentService.id}-envoy-image-account-mapping`, {
            mapping: {
                'ap-northeast-1': this.accountIdForRegion('ap-northeast-1'),
                'ap-northeast-2': this.accountIdForRegion('ap-northeast-2'),
                'ap-south-1': this.accountIdForRegion('ap-south-1'),
                'ap-southeast-1': this.accountIdForRegion('ap-southeast-1'),
                'ap-southeast-2': this.accountIdForRegion('ap-southeast-1'),
                'ca-central-1': this.accountIdForRegion('ca-central-1'),
                'cn-north-1': this.accountIdForRegion('cn-north-1'),
                'cn-northwest-1': this.accountIdForRegion('cn-northwest-1'),
                'eu-central-1': this.accountIdForRegion('eu-central-1'),
                'eu-north-1': this.accountIdForRegion('eu-north-1'),
                'eu-south-1': this.accountIdForRegion('eu-south-1'),
                'eu-west-1': this.accountIdForRegion('eu-west-1'),
                'eu-west-2': this.accountIdForRegion('eu-west-2'),
                'eu-west-3': this.accountIdForRegion('eu-west-3'),
                'sa-east-1': this.accountIdForRegion('sa-east-1'),
                'us-east-1': this.accountIdForRegion('us-east-1'),
                'us-east-2': this.accountIdForRegion('us-east-2'),
                'us-west-1': this.accountIdForRegion('us-west-1'),
                'us-west-2': this.accountIdForRegion('us-west-2'),
                'me-south-1': this.accountIdForRegion('me-south-1'),
                'ap-east-1': this.accountIdForRegion('ap-east-1'),
                'af-south-1': this.accountIdForRegion('af-south-1'),
            },
        });
        // WHEN
        const ownerAccount = mapping.findInMap(region, 'ecrRepo');
        appMeshRepo = ecr.Repository.fromRepositoryAttributes(this.scope, `${this.parentService.id}-envoy-repo`, {
            repositoryName: 'aws-appmesh-envoy',
            repositoryArn: `arn:${partition}:ecr:${region}:${ownerAccount}:repository/aws-appmesh-envoy`,
        });
        this.container = taskDefinition.addContainer('envoy', {
            image: ecs.ContainerImage.fromEcrRepository(appMeshRepo, APP_MESH_ENVOY_SIDECAR_VERSION),
            essential: true,
            environment: {
                APPMESH_VIRTUAL_NODE_NAME: `mesh/${this.mesh.meshName}/virtualNode/${this.parentService.id}`,
                AWS_REGION: aws_cdk_lib_1.Stack.of(this.parentService).region,
                ENABLE_ENVOY_STATS_TAGS: '1',
                ENABLE_ENVOY_DOG_STATSD: '1',
            },
            healthCheck: {
                command: [
                    'CMD-SHELL',
                    'curl -s http://localhost:9901/server_info | grep state | grep -q LIVE',
                ],
                startPeriod: aws_cdk_lib_1.Duration.seconds(10),
                interval: aws_cdk_lib_1.Duration.seconds(5),
                timeout: aws_cdk_lib_1.Duration.seconds(2),
            },
            memoryReservationMiB: 128,
            user: '1337',
            logging: new ecs.AwsLogDriver({ streamPrefix: 'envoy' }),
        });
        // Modify the task definition role to allow the Envoy sidecar to get
        // configuration from the Envoy control plane, for this particular
        // mesh only.
        new iam.Policy(this.scope, `${this.parentService.id}-envoy-to-appmesh`, {
            roles: [taskDefinition.taskRole],
            statements: [
                new iam.PolicyStatement({
                    resources: [this.mesh.meshArn],
                    actions: ['appmesh:StreamAggregatedResources'],
                }),
            ],
        });
        // Raise the number of open file descriptors allowed. This is
        // necessary when the Envoy proxy is handling large amounts of
        // traffic.
        this.container.addUlimits({
            softLimit: 1024000,
            hardLimit: 1024000,
            name: ecs.UlimitName.NOFILE,
        });
    }
    // Enable CloudMap for the service.
    modifyServiceProps(props) {
        return {
            ...props,
            // Ensure that service tasks are registered into
            // CloudMap so that the App Mesh proxy can find them.
            cloudMapOptions: {
                dnsRecordType: 'A',
                dnsTtl: aws_cdk_lib_1.Duration.seconds(10),
                failureThreshold: 2,
                name: this.parentService.id,
            },
            // These specific deployment settings are currently required in order to
            // maintain availability during a rolling deploy of the service with App Mesh
            // https://docs.aws.amazon.com/app-mesh/latest/userguide/best-practices.html#reduce-deployment-velocity
            minHealthyPercent: 100,
            maxHealthyPercent: 125,
        };
    }
    // Now that the service is defined, we can create the AppMesh virtual service
    // and virtual node for the real service
    useService(service) {
        const containerextension = this.parentService.serviceDescription.get('service-container');
        if (!containerextension) {
            throw new Error('Firelens extension requires an application extension');
        }
        const cloudmapNamespace = this.parentService.cluster.defaultCloudMapNamespace;
        if (!cloudmapNamespace) {
            throw new Error('You must add a CloudMap namespace to the ECS cluster in order to use the AppMesh extension');
        }
        function addListener(protocol, port) {
            switch (protocol) {
                case Protocol.HTTP:
                    return appmesh.VirtualNodeListener.http({ port });
                case Protocol.HTTP2:
                    return appmesh.VirtualNodeListener.http2({ port });
                case Protocol.GRPC:
                    return appmesh.VirtualNodeListener.grpc({ port });
                case Protocol.TCP:
                    return appmesh.VirtualNodeListener.tcp({ port });
            }
        }
        // Create a virtual node for the name service
        this.virtualNode = new appmesh.VirtualNode(this.scope, `${this.parentService.id}-virtual-node`, {
            mesh: this.mesh,
            virtualNodeName: this.parentService.id,
            serviceDiscovery: service.cloudMapService
                ? appmesh.ServiceDiscovery.cloudMap(service.cloudMapService)
                : undefined,
            listeners: [addListener(this.protocol, containerextension.trafficPort)],
        });
        // Create a virtual router for this service. This allows for retries
        // and other similar behaviors.
        this.virtualRouter = new appmesh.VirtualRouter(this.scope, `${this.parentService.id}-virtual-router`, {
            mesh: this.mesh,
            listeners: [
                this.virtualRouterListener(containerextension.trafficPort),
            ],
            virtualRouterName: `${this.parentService.id}`,
        });
        // Form the service name that requests will be made to
        const serviceName = `${this.parentService.id}.${cloudmapNamespace.namespaceName}`;
        const weightedTargets = [{
                virtualNode: this.virtualNode,
                weight: 1,
            }];
        // Now add the virtual node as a route in the virtual router
        // Ensure that the route type matches the protocol type.
        this.route = this.virtualRouter.addRoute(`${this.parentService.id}-route`, {
            routeSpec: this.routeSpec(weightedTargets, serviceName),
        });
        // Now create a virtual service. Relationship goes like this:
        // virtual service -> virtual router -> virtual node
        this.virtualService = new appmesh.VirtualService(this.scope, `${this.parentService.id}-virtual-service`, {
            virtualServiceProvider: appmesh.VirtualServiceProvider.virtualRouter(this.virtualRouter),
            virtualServiceName: serviceName,
        });
    }
    // Connect the app mesh extension for this service to an app mesh
    // extension on another service.
    connectToService(otherService, _connectToProps) {
        const otherAppMesh = otherService.serviceDescription.get('appmesh');
        const otherContainer = otherService.serviceDescription.get('service-container');
        // Do a check to ensure that these services are in the same environment.
        // Currently this extension only supports connecting services within
        // the same VPC, same App Mesh service mesh, and same Cloud Map namespace
        if (otherAppMesh.parentService.environment.id !== this.parentService.environment.id) {
            throw new Error(`Unable to connect service '${this.parentService.id}' in environment '${this.parentService.environment.id}' to service '${otherService.id}' in environment '${otherAppMesh.parentService.environment.id}' because services can not be connected across environment boundaries`);
        }
        // First allow this service to talk to the other service
        // at a network level. This opens the security groups so that
        // the security groups of these two services to each other
        this.parentService.ecsService.connections.allowTo(otherService.ecsService, ec2.Port.tcp(otherContainer.trafficPort), `Accept inbound traffic from ${this.parentService.id}`);
        // Next update the app mesh config so that the local Envoy
        // proxy on this service knows how to route traffic to
        // nodes from the other service.
        this.virtualNode.addBackend(appmesh.Backend.virtualService(otherAppMesh.virtualService));
    }
    routeSpec(weightedTargets, serviceName) {
        switch (this.protocol) {
            case Protocol.HTTP: return appmesh.RouteSpec.http({
                weightedTargets: weightedTargets,
            });
            case Protocol.HTTP2: return appmesh.RouteSpec.http2({
                weightedTargets: weightedTargets,
            });
            case Protocol.GRPC: return appmesh.RouteSpec.grpc({
                weightedTargets: weightedTargets,
                match: {
                    serviceName: serviceName,
                },
            });
            case Protocol.TCP: return appmesh.RouteSpec.tcp({
                weightedTargets: weightedTargets,
            });
        }
    }
    virtualRouterListener(port) {
        switch (this.protocol) {
            case Protocol.HTTP: return appmesh.VirtualRouterListener.http(port);
            case Protocol.HTTP2: return appmesh.VirtualRouterListener.http2(port);
            case Protocol.GRPC: return appmesh.VirtualRouterListener.grpc(port);
            case Protocol.TCP: return appmesh.VirtualRouterListener.tcp(port);
        }
    }
}
exports.AppMeshExtension = AppMeshExtension;
_a = JSII_RTTI_SYMBOL_1;
AppMeshExtension[_a] = { fqn: "@aws-cdk-containers/ecs-service-extensions.AppMeshExtension", version: "2.0.1-alpha.275" };
//# sourceMappingURL=data:application/json;base64,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