"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Container = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ecs = require("aws-cdk-lib/aws-ecs");
const awslogs = require("aws-cdk-lib/aws-logs");
const cxapi = require("aws-cdk-lib/cx-api");
const extension_interfaces_1 = require("./extension-interfaces");
/**
 * The main container of a service. This is generally the container
 * which runs your application business logic. Other extensions will attach
 * sidecars alongside this main container.
 */
class Container extends extension_interfaces_1.ServiceExtension {
    constructor(props) {
        super('service-container');
        this.props = props;
        this.trafficPort = props.trafficPort;
        this.logGroup = props.logGroup;
    }
    prehook(service, scope) {
        this.parentService = service;
        this.scope = scope;
    }
    // This hook sets the overall task resource requirements to the
    // resource requirements of the application itself.
    modifyTaskDefinitionProps(props) {
        return {
            ...props,
            cpu: this.props.cpu.toString(),
            memoryMiB: this.props.memoryMiB.toString(),
        };
    }
    // This hook adds the application container to the task definition.
    useTaskDefinition(taskDefinition) {
        let containerProps = {
            image: this.props.image,
            cpu: Number(this.props.cpu),
            memoryLimitMiB: Number(this.props.memoryMiB),
            environment: this.props.environment,
            environmentFiles: this.props.environmentFiles,
            secrets: this.props.secrets,
        };
        // Let other extensions mutate the container definition. This is
        // used by extensions which want to add environment variables, modify
        // logging parameters, etc.
        this.containerMutatingHooks.forEach((hookProvider) => {
            containerProps = hookProvider.mutateContainerDefinition(containerProps);
        });
        // If no observability extensions have been added to the service description then we can configure the `awslogs` log driver
        if (!containerProps.logging) {
            // Create a log group for the service if one is not provided by the user (only if feature flag is set)
            if (!this.logGroup && this.parentService.node.tryGetContext(cxapi.ECS_SERVICE_EXTENSIONS_ENABLE_DEFAULT_LOG_DRIVER)) {
                this.logGroup = new awslogs.LogGroup(this.scope, `${this.parentService.id}-logs`, {
                    logGroupName: `${this.parentService.id}-logs`,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                    retention: awslogs.RetentionDays.ONE_MONTH,
                });
            }
            if (this.logGroup) {
                containerProps = {
                    ...containerProps,
                    logging: new ecs.AwsLogDriver({
                        streamPrefix: this.parentService.id,
                        logGroup: this.logGroup,
                    }),
                };
            }
        }
        else {
            if (this.logGroup) {
                throw Error(`Log configuration already specified. You cannot provide a log group for the application container of service '${this.parentService.id}' while also adding log configuration separately using service extensions.`);
            }
        }
        this.container = taskDefinition.addContainer('app', containerProps);
        // Create a port mapping for the container if not already created by another extension.
        if (!this.container.findPortMapping(this.trafficPort, ecs.Protocol.TCP)) {
            this.container.addPortMappings({
                containerPort: this.trafficPort,
            });
        }
        // Raise the ulimits for this main application container
        // so that it can handle more concurrent requests
        this.container.addUlimits({
            softLimit: 1024000,
            hardLimit: 1024000,
            name: ecs.UlimitName.NOFILE,
        });
    }
    resolveContainerDependencies() {
        if (!this.container) {
            throw new Error('The container dependency hook was called before the container was created');
        }
        const firelens = this.parentService.serviceDescription.get('firelens');
        if (firelens && firelens.container) {
            this.container.addContainerDependencies({
                container: firelens.container,
                condition: ecs.ContainerDependencyCondition.START,
            });
        }
        const appmeshextension = this.parentService.serviceDescription.get('appmesh');
        if (appmeshextension && appmeshextension.container) {
            this.container.addContainerDependencies({
                container: appmeshextension.container,
                condition: ecs.ContainerDependencyCondition.HEALTHY,
            });
        }
        const cloudwatchextension = this.parentService.serviceDescription.get('cloudwatchAgent');
        if (cloudwatchextension && cloudwatchextension.container) {
            this.container.addContainerDependencies({
                container: cloudwatchextension.container,
                condition: ecs.ContainerDependencyCondition.START,
            });
        }
        const xrayextension = this.parentService.serviceDescription.get('xray');
        if (xrayextension && xrayextension.container) {
            this.container.addContainerDependencies({
                container: xrayextension.container,
                condition: ecs.ContainerDependencyCondition.HEALTHY,
            });
        }
    }
}
exports.Container = Container;
_a = JSII_RTTI_SYMBOL_1;
Container[_a] = { fqn: "@aws-cdk-containers/ecs-service-extensions.Container", version: "2.0.1-alpha.275" };
//# sourceMappingURL=data:application/json;base64,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