"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpLoadBalancerExtension = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const alb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const extension_interfaces_1 = require("./extension-interfaces");
/**
 * This extension add a public facing load balancer for sending traffic
 * to one or more replicas of the application container.
 */
class HttpLoadBalancerExtension extends extension_interfaces_1.ServiceExtension {
    constructor(props = {}) {
        super('load-balancer');
        this.requestsPerTarget = props.requestsPerTarget;
    }
    // Before the service is created, go ahead and create the load balancer itself.
    prehook(service, scope) {
        this.parentService = service;
        this.loadBalancer = new alb.ApplicationLoadBalancer(scope, `${this.parentService.id}-load-balancer`, {
            vpc: this.parentService.vpc,
            internetFacing: true,
        });
        this.listener = this.loadBalancer.addListener(`${this.parentService.id}-listener`, {
            port: 80,
            open: true,
        });
        // Automatically create an output
        new aws_cdk_lib_1.CfnOutput(scope, `${this.parentService.id}-load-balancer-dns-output`, {
            value: this.loadBalancer.loadBalancerDnsName,
        });
    }
    // Minor service configuration tweaks to work better with a load balancer
    modifyServiceProps(props) {
        return {
            ...props,
            // Give the task a little bit of grace time to start passing
            // healthchecks. Without this it is possible for a slow starting task
            // to cause the ALB to consider the task unhealthy, causing ECS to stop
            // the task before it actually has a chance to finish starting up
            healthCheckGracePeriod: aws_cdk_lib_1.Duration.minutes(1),
        };
    }
    // After the service is created add the service to the load balancer's listener
    useService(service) {
        const targetGroup = this.listener.addTargets(this.parentService.id, {
            deregistrationDelay: aws_cdk_lib_1.Duration.seconds(10),
            port: 80,
            targets: [service],
        });
        this.parentService.targetGroup = targetGroup;
        if (this.requestsPerTarget) {
            if (!this.parentService.scalableTaskCount) {
                throw Error(`Auto scaling target for the service '${this.parentService.id}' hasn't been configured. Please use Service construct to configure 'minTaskCount' and 'maxTaskCount'.`);
            }
            this.parentService.scalableTaskCount.scaleOnRequestCount(`${this.parentService.id}-target-request-count-${this.requestsPerTarget}`, {
                requestsPerTarget: this.requestsPerTarget,
                targetGroup: this.parentService.targetGroup,
            });
            this.parentService.enableAutoScalingPolicy();
        }
    }
}
exports.HttpLoadBalancerExtension = HttpLoadBalancerExtension;
_a = JSII_RTTI_SYMBOL_1;
HttpLoadBalancerExtension[_a] = { fqn: "@aws-cdk-containers/ecs-service-extensions.HttpLoadBalancerExtension", version: "2.0.1-alpha.275" };
//# sourceMappingURL=data:application/json;base64,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