"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Service = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ecs = require("aws-cdk-lib/aws-ecs");
const constructs_1 = require("constructs");
const extension_interfaces_1 = require("./extensions/extension-interfaces");
/**
 * This Service construct serves as a Builder class for an ECS service. It
 * supports various extensions and keeps track of any mutating state, allowing
 * it to build up an ECS service progressively.
 */
class Service extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The flag to track if auto scaling policies have been configured
         * for the service.
         */
        this.autoScalingPoliciesEnabled = false;
        /**
         * The list of URLs associated with this service.
         */
        this.urls = {};
        this.scope = scope;
        this.id = id;
        this.environment = props.environment;
        this.vpc = props.environment.vpc;
        this.cluster = props.environment.cluster;
        this.capacityType = props.environment.capacityType;
        this.serviceDescription = props.serviceDescription;
        // Check to make sure that the user has actually added a container
        const containerextension = this.serviceDescription.get('service-container');
        if (!containerextension) {
            throw new Error(`Service '${this.id}' must have a Container extension`);
        }
        // First set the scope for all the extensions
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                this.serviceDescription.extensions[extensions].prehook(this, this.scope);
            }
        }
        // At the point of preparation all extensions have been defined on the service
        // so give each extension a chance to now add hooks to other extensions if
        // needed
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                this.serviceDescription.extensions[extensions].addHooks();
            }
        }
        // Give each extension a chance to mutate the task def creation properties
        let taskDefProps = {
            // Default CPU and memory
            cpu: '256',
            memory: '512',
            // Allow user to pre-define the taskRole so that it can be used in resource policies that may
            // be defined before the ECS service exists in a CDK application
            taskRole: props.taskRole,
            // Ensure that the task definition supports both EC2 and Fargate
            compatibility: ecs.Compatibility.EC2_AND_FARGATE,
        };
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                taskDefProps = this.serviceDescription.extensions[extensions].modifyTaskDefinitionProps(taskDefProps);
            }
        }
        // Now that the task definition properties are assembled, create it
        this.taskDefinition = new ecs.TaskDefinition(this.scope, `${this.id}-task-definition`, taskDefProps);
        // Now give each extension a chance to use the task definition
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                this.serviceDescription.extensions[extensions].useTaskDefinition(this.taskDefinition);
            }
        }
        // Now that all containers are created, give each extension a chance
        // to bake its dependency graph
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                this.serviceDescription.extensions[extensions].resolveContainerDependencies();
            }
        }
        // Give each extension a chance to mutate the service props before
        // service creation
        let serviceProps = {
            cluster: this.cluster,
            taskDefinition: this.taskDefinition,
            minHealthyPercent: 100,
            maxHealthyPercent: 200,
            desiredCount: props.desiredCount ?? 1,
        };
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                serviceProps = this.serviceDescription.extensions[extensions].modifyServiceProps(serviceProps);
            }
        }
        // If a maxHealthyPercent and desired count has been set while minHealthyPercent == 100% then we
        // need to do some failsafe checking to ensure that the maxHealthyPercent
        // actually allows a rolling deploy. Otherwise it is possible to end up with
        // blocked deploys that can take no action because minHealtyhPercent == 100%
        // prevents running, healthy tasks from being stopped, but a low maxHealthyPercent
        // can also prevents new parallel tasks from being started.
        if (serviceProps.maxHealthyPercent && serviceProps.desiredCount && serviceProps.minHealthyPercent && serviceProps.minHealthyPercent == 100) {
            if (serviceProps.desiredCount == 1) {
                // If there is one task then we must allow max percentage to be at
                // least 200% for another replacement task to be added
                serviceProps = {
                    ...serviceProps,
                    maxHealthyPercent: Math.max(200, serviceProps.maxHealthyPercent),
                };
            }
            else if (serviceProps.desiredCount <= 3) {
                // If task count is 2 or 3 then max percent must be at least 150% to
                // allow one replacement task to be launched at a time.
                serviceProps = {
                    ...serviceProps,
                    maxHealthyPercent: Math.max(150, serviceProps.maxHealthyPercent),
                };
            }
            else {
                // For anything higher than 3 tasks set max percent to at least 125%
                // For 4 tasks this will allow exactly one extra replacement task
                // at a time, for any higher task count it will allow 25% of the tasks
                // to be replaced at a time.
                serviceProps = {
                    ...serviceProps,
                    maxHealthyPercent: Math.max(125, serviceProps.maxHealthyPercent),
                };
            }
        }
        // Set desiredCount to `undefined` if auto scaling is configured for the service
        if (props.autoScaleTaskCount || this.autoScalingPoliciesEnabled) {
            serviceProps = {
                ...serviceProps,
                desiredCount: undefined,
            };
        }
        // Now that the service props are determined we can create
        // the service
        if (this.capacityType === extension_interfaces_1.EnvironmentCapacityType.EC2) {
            this.ecsService = new ecs.Ec2Service(this.scope, `${this.id}-service`, serviceProps);
        }
        else if (this.capacityType === extension_interfaces_1.EnvironmentCapacityType.FARGATE) {
            this.ecsService = new ecs.FargateService(this.scope, `${this.id}-service`, serviceProps);
        }
        else {
            throw new Error(`Unknown capacity type for service ${this.id}`);
        }
        // Create the auto scaling target and configure target tracking policies after the service is created
        if (props.autoScaleTaskCount) {
            this.scalableTaskCount = this.ecsService.autoScaleTaskCount({
                maxCapacity: props.autoScaleTaskCount.maxTaskCount,
                minCapacity: props.autoScaleTaskCount.minTaskCount,
            });
            if (props.autoScaleTaskCount.targetCpuUtilization) {
                const targetCpuUtilizationPercent = props.autoScaleTaskCount.targetCpuUtilization;
                this.scalableTaskCount.scaleOnCpuUtilization(`${this.id}-target-cpu-utilization-${targetCpuUtilizationPercent}`, {
                    targetUtilizationPercent: targetCpuUtilizationPercent,
                });
                this.enableAutoScalingPolicy();
            }
            if (props.autoScaleTaskCount.targetMemoryUtilization) {
                const targetMemoryUtilizationPercent = props.autoScaleTaskCount.targetMemoryUtilization;
                this.scalableTaskCount.scaleOnMemoryUtilization(`${this.id}-target-memory-utilization-${targetMemoryUtilizationPercent}`, {
                    targetUtilizationPercent: targetMemoryUtilizationPercent,
                });
                this.enableAutoScalingPolicy();
            }
        }
        // Now give all extensions a chance to use the service
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                this.serviceDescription.extensions[extensions].useService(this.ecsService);
            }
        }
        // Error out if the auto scaling target is created but no scaling policies have been configured
        if (this.scalableTaskCount && !this.autoScalingPoliciesEnabled) {
            throw Error(`The auto scaling target for the service '${this.id}' has been created but no auto scaling policies have been configured.`);
        }
    }
    /**
     * Tell extensions from one service to connect to extensions from
     * another sevice if they have implemented a hook for it.
     *
     * @param service
     */
    connectTo(service, connectToProps) {
        for (const extensions in this.serviceDescription.extensions) {
            if (this.serviceDescription.extensions[extensions]) {
                this.serviceDescription.extensions[extensions].connectToService(service, connectToProps);
            }
        }
    }
    /**
     * This method adds a new URL for the service. This allows extensions to
     * submit a URL for the service. For example, a load balancer might add its
     * URL, or App Mesh can add its DNS name for the service.
     *
     * @param urlName - The identifier name for this URL
     * @param url - The URL itself.
     */
    addURL(urlName, url) {
        this.urls[urlName] = url;
    }
    /**
     * Retrieve a URL for the service. The URL must have previously been
     * stored by one of the URL providing extensions.
     *
     * @param urlName - The URL to look up.
     */
    getURL(urlName) {
        if (!this.urls[urlName]) {
            throw new Error(`Unable to find a URL with name '${urlName}'`);
        }
        return this.urls[urlName];
    }
    /**
     * This helper method is used to set the `autoScalingPoliciesEnabled` attribute
     * whenever an auto scaling policy is configured for the service.
     */
    enableAutoScalingPolicy() {
        this.autoScalingPoliciesEnabled = true;
    }
}
exports.Service = Service;
_a = JSII_RTTI_SYMBOL_1;
Service[_a] = { fqn: "@aws-cdk-containers/ecs-service-extensions.Service", version: "2.0.1-alpha.275" };
//# sourceMappingURL=data:application/json;base64,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