# PXMeter - Structural Quality Assessment for Biomolecular Structure Prediction Models

[![License](https://img.shields.io/badge/License-Apache%202.0-blue.svg)](https://opensource.org/licenses/Apache-2.0)
[![Python 3.11+](https://img.shields.io/badge/python-3.11%2B-blue.svg)](https://www.python.org/downloads/)

<div align="left" style="margin: 20px 0;">
<span style="margin: 0 10px;">📄 <a href="https://www.biorxiv.org/content/10.1101/2025.07.17.664878v1">From Dataset Curation to Unified Evaluation: Revisiting
 Structure Prediction Benchmarks with PXMeter</a></span>
</div>

PXMeter is a comprehensive toolkit for evaluating the quality of structures generated by biomolecular structure prediction models, with support for proteins, nucleic acids, and small molecules.

## 🌟 Features
- **Full-Atom Matching Between Reference and Model Structures**: Automatically matches entities, aligns sequences, and permutes chains and atoms to establish one-to-one correspondence.
- **Multi-metric Evaluation**:
  - Local Distance Difference Test (LDDT) - A local superposition-free score for comparing reference and model structures.
  - DockQ - Interface interaction quality.
  - Pocket-aligned Root Mean Squared Deviation (RMSD) - Ligand pose quality.
  - PoseBusters Validity Checks - Plausibility checks for generated molecule poses.
- **Dual Interfaces**: CLI & Python API

## 🛠️ Installation

```bash
# Install from PyPI
pip install pxmeter

# Install from source
git clone https://github.com/bytedance/PXMeter.git
cd PXMeter
pip install -r requirements.txt
pip install -e .
```

PXMeter will automatically download the Chemical Component Dictionary (CCD) upon its first run. To update the CCD files:

```bash
pxm ccd update
```

## 🚀 Quick Start

### Command Line Interface
```bash
pxm -r examples/7rss.cif -m examples/7rss_protenix_pred.cif -o pxm_output.json
```

**Key Parameters**:
- `-r` or `--ref_cif`: Path to reference CIF file
- `-m` or `--model_cif`: Path to model CIF file
- `-o` or `--output`: Path to save evaluation results (default: "pxm_output.json")
- `--ref_model`: Specify model number of reference CIF (default: 1)
- `--ref_assembly_id`: Specify the assembly ID for the reference CIF (default: None; uses the Asymmetric Unit for evaluation)
- `ref_altloc`: Specify the alternative location identifier for the reference CIF (default: "first", uses the first alternative location code for each residue).
- `--chain_id_to_mol_json`: JSON file defining custom ligands, where keys are chain IDs (label_asym_id) and values are the corresponding ligand SMILES strings.
- `-l` or `--interested_lig_label_asym_id`: Indicate the `label_asym_id` of ligands for metrics like pocket-aligned RMSD. Multiple ligands should be comma-separated.

To access the full list of parameters, use the `--help` option.

### Python API
**Note**: For batch evaluation of multiple structures, the Python API is highly recommended. This approach is more efficient than repeated command line calls because it caches CCD CIF files in memory and thus avoids redundant disk I/O.


```python
from pxmeter.eval import evaluate

ref_cif = "examples/7rss.cif"
model_cif = "examples/7rss_protenix_pred.cif"
metric_result = evaluate(
    ref_cif=ref_cif,
    model_cif=model_cif,
)

json_dict = metric_result.to_json_dict()
print(json_dict)
```

For detailed descriptions of additional parameters, use the `help()` function:
```python
help(evaluate)
```

## 📊 Benchmarking
Refer to [benchmark/README.md](./benchmark/README.md) for evaluation protocols on:
- RecentPDB dataset
- PoseBusters V2

The benchmark data is released under the CC0 license.
We include code in the `benchmark` directory that evaluates various models using PXMeter and aggregates their metrics.
This serves as an example of best practices for using the tool. For more details, please refer to our paper:

📄 <a href="https://www.biorxiv.org/content/10.1101/2025.07.17.664878v1">From Dataset Curation to Unified Evaluation: Revisiting
 Structure Prediction Benchmarks with PXMeter</a>



## 💪 Contributing to PXMeter
We welcome contributions from the community to help improve PXMeter!

Check out the [Contributing Guide](CONTRIBUTING.md) to get started.

Code Quality: We use `pre-commit` hooks to maintain consistent programming style and code quality. Please install them before committing.

```bash
pip install pre-commit
pre-commit install
```


## ✍️ Citing PXMeter
If you use PXMeter in your research, please cite the following:
```bibtex
@article {Ma2025.07.17.664878,
	author = {Ma, Wenzhi and Liu, Zhenyu and Yang, Jincai and Lu, Chan and Zhang, Hanyu and Xiao, Wenzhi},
	title = {From Dataset Curation to Unified Evaluation: Revisiting Structure Prediction Benchmarks with PXMeter},
	year = {2025},
	doi = {10.1101/2025.07.17.664878},
	publisher = {Cold Spring Harbor Laboratory},
	URL = {https://www.biorxiv.org/content/early/2025/07/22/2025.07.17.664878},
	eprint = {https://www.biorxiv.org/content/early/2025/07/22/2025.07.17.664878.full.pdf},
	journal = {bioRxiv}
}
```


## 🚧 Limitations
- PXMeter supports chain/atom permutations but not residue-level permutations. As a result, the accuracy of evaluation for branched chains, such as glycans, cannot be fully guaranteed.
- It is recommended to use CIF files from the RCSB PDB as references, as they ensure content accuracy. All development and testing were conducted exclusively on CIF files from this source.


## 🛡️ Security
If you discover a potential security issue in this project, or think you may
have discovered a security issue, we ask that you notify Bytedance Security via our [security center](https://security.bytedance.com/src) or [vulnerability reporting email](sec@bytedance.com).

Please do **not** create a public GitHub issue.

## ⚖️ License
The PXMeter project is made available under the [Apache 2.0 License](./LICENSE), it is free for both academic research and commercial use.
