"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FireLensExtension = exports.FirelensMutatingHook = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ecs = require("aws-cdk-lib/aws-ecs");
const awslogs = require("aws-cdk-lib/aws-logs");
const extension_interfaces_1 = require("./extension-interfaces");
/**
 * This hook modifies the application container's settings so that
 * it routes logs using FireLens.
 */
class FirelensMutatingHook extends extension_interfaces_1.ContainerMutatingHook {
    constructor(props) {
        super();
        this.parentService = props.parentService;
        this.logGroup = props.logGroup;
    }
    mutateContainerDefinition(props) {
        return {
            ...props,
            logging: ecs.LogDrivers.firelens({
                options: {
                    Name: 'cloudwatch',
                    region: aws_cdk_lib_1.Stack.of(this.parentService).region,
                    log_group_name: this.logGroup.logGroupName,
                    log_stream_prefix: `${this.parentService.id}/`,
                },
            }),
        };
    }
}
exports.FirelensMutatingHook = FirelensMutatingHook;
_a = JSII_RTTI_SYMBOL_1;
FirelensMutatingHook[_a] = { fqn: "cdk-ecs-service-extensions.FirelensMutatingHook", version: "0.0.2" };
/**
 * This extension adds a FluentBit log router to the task definition
 * and does all the configuration necessarily to enable log routing
 * for the task using FireLens.
 */
class FireLensExtension extends extension_interfaces_1.ServiceExtension {
    constructor() {
        super('firelens');
    }
    prehook(service, scope) {
        this.parentService = service;
        // Create a log group for the service, into which FireLens
        // will route the service's logs
        this.logGroup = new awslogs.LogGroup(scope, `${service.id}-logs`, {
            logGroupName: `${service.id}-logs`,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            retention: awslogs.RetentionDays.ONE_WEEK,
        });
    }
    // Add hooks to the main application extension so that it is modified to
    // have logging properties that enable sending logs via the
    // Firelens log router container
    addHooks() {
        const container = this.parentService.serviceDescription.get('service-container');
        if (!container) {
            throw new Error('Firelens extension requires an application extension');
        }
        container.addContainerMutatingHook(new FirelensMutatingHook({
            parentService: this.parentService,
            logGroup: this.logGroup,
        }));
    }
    useTaskDefinition(taskDefinition) {
        // Manually add a firelens log router, so that we can manually manage the dependencies
        // to ensure that the Firelens log router depends on the Envoy proxy
        this.container = taskDefinition.addFirelensLogRouter('firelens', {
            image: ecs.obtainDefaultFluentBitECRImage(taskDefinition, {
                logDriver: 'awsfirelens',
                options: {
                    Name: 'cloudwatch',
                },
            }),
            firelensConfig: {
                type: ecs.FirelensLogRouterType.FLUENTBIT,
            },
            logging: new ecs.AwsLogDriver({ streamPrefix: 'firelens' }),
            memoryReservationMiB: 50,
            user: '0:1338',
        });
    }
    resolveContainerDependencies() {
        if (!this.container) {
            throw new Error('The container dependency hook was called before the container was created');
        }
        const appmeshextension = this.parentService.serviceDescription.get('appmesh');
        if (appmeshextension && appmeshextension.container) {
            this.container.addContainerDependencies({
                container: appmeshextension.container,
                condition: ecs.ContainerDependencyCondition.HEALTHY,
            });
        }
    }
}
exports.FireLensExtension = FireLensExtension;
_b = JSII_RTTI_SYMBOL_1;
FireLensExtension[_b] = { fqn: "cdk-ecs-service-extensions.FireLensExtension", version: "0.0.2" };
//# sourceMappingURL=data:application/json;base64,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