"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpLoadBalancerExtension = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const alb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const extension_interfaces_1 = require("./extension-interfaces");
/**
 * This extension add a public facing load balancer for sending traffic
 * to one or more replicas of the application container.
 */
class HttpLoadBalancerExtension extends extension_interfaces_1.ServiceExtension {
    constructor(props = {}) {
        super('load-balancer');
        this.requestsPerTarget = props.requestsPerTarget;
    }
    // Before the service is created, go ahead and create the load balancer itself.
    prehook(service, scope) {
        this.parentService = service;
        this.loadBalancer = new alb.ApplicationLoadBalancer(scope, `${this.parentService.id}-load-balancer`, {
            vpc: this.parentService.vpc,
            internetFacing: true,
        });
        this.listener = this.loadBalancer.addListener(`${this.parentService.id}-listener`, {
            port: 80,
            open: true,
        });
        // Automatically create an output
        new aws_cdk_lib_1.CfnOutput(scope, `${this.parentService.id}-load-balancer-dns-output`, {
            value: this.loadBalancer.loadBalancerDnsName,
        });
    }
    // Minor service configuration tweaks to work better with a load balancer
    modifyServiceProps(props) {
        return {
            ...props,
            // Give the task a little bit of grace time to start passing
            // healthchecks. Without this it is possible for a slow starting task
            // to cause the ALB to consider the task unhealthy, causing ECS to stop
            // the task before it actually has a chance to finish starting up
            healthCheckGracePeriod: aws_cdk_lib_1.Duration.minutes(1),
        };
    }
    // After the service is created add the service to the load balancer's listener
    useService(service) {
        const targetGroup = this.listener.addTargets(this.parentService.id, {
            deregistrationDelay: aws_cdk_lib_1.Duration.seconds(10),
            port: 80,
            targets: [service],
        });
        if (this.requestsPerTarget) {
            if (!this.parentService.scalableTaskCount) {
                throw Error(`Auto scaling target for the service '${this.parentService.id}' hasn't been configured. Please use Service construct to configure 'minTaskCount' and 'maxTaskCount'.`);
            }
            this.parentService.scalableTaskCount.scaleOnRequestCount(`${this.parentService.id}-target-request-count-${this.requestsPerTarget}`, {
                requestsPerTarget: this.requestsPerTarget,
                targetGroup,
            });
            this.parentService.enableAutoScalingPolicy();
        }
    }
}
exports.HttpLoadBalancerExtension = HttpLoadBalancerExtension;
_a = JSII_RTTI_SYMBOL_1;
HttpLoadBalancerExtension[_a] = { fqn: "cdk-ecs-service-extensions.HttpLoadBalancerExtension", version: "0.0.2" };
//# sourceMappingURL=data:application/json;base64,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